#include <stdio.h>
#include <stdlib.h>

/*---------------------------------------------------------------------*
 * this program generates the binaries for the 2 ALU-EPROMS.           *
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
 * (c) Dieter Mueller 2003                                             *
 *---------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*/

#define ALU_PASS_ID     0x00
#define ALU_PASS_IX     0x01

#define ALU_ADC         0x02
#define ALU_SBC         0x03
#define ALU_INC         0x04
#define ALU_DEC         0x05

#define ALU_ADDAC       0x06
#define ALU_ADD0C       0x07

#define ALU_ASL         0x08
#define ALU_LSR         0x09
#define ALU_ROL         0x0A
#define ALU_ROR         0x0B

#define ALU_AND         0x0C
#define ALU_OR          0x0D
#define ALU_XOR         0x0E
#define ALU_BIT         0x0F

#define ALU_C_00        0x10
#define ALU_C_01        0x11
#define ALU_C_FA        0x12
#define ALU_C_FC        0x13
#define ALU_C_FE        0x14
#define ALU_C_FF        0x15

#define ALU_CLC         0x16
#define ALU_SEC         0x17
#define ALU_CLICLB      0x18
#define ALU_SEICLB      0x19
#define ALU_CLD         0x1A
#define ALU_SED         0x1B
#define ALU_SEB         0x1C
#define ALU_CLV         0x1D

#define ALU_MNZC        0x1E
#define ALU_MNZVC       0x1F

/*-------------------------------------------------------------------------*/

#define Q_Z             0x10
#define Q_V             0x40
#define Q_C             0x80
#define Q_TC            0x20

/*-------------------------------------------------------------------------*/

int main(void);
int rom_L(unsigned long i);
int rom_H(unsigned long i);

/*-------------------------------------------------------------------------*/
//Bit 0..3

int rom_L(unsigned long i)
{
  int tmp;

  int cmd;     //command
  int id;      //input from internal Data Bus
  int ix;      //input from external Data Bus
  int t_c;     //temp carry from prev. cycle
  int c_f;     //carry flag

  int result = 0; //result
  int q_z    = 0; //Z flag output
  int q_v    = 0; //V flag output
  int q_c    = 0; //C flag output
  int q_tc   = 0; //C flag output (temp. C-Flag  for INC,DEC,addr.calculation)

  id  =  i       & 0x001f;
  ix  = (i >> 5) & 0x000f;
  cmd = (i >> 9) & 0x001f;

  t_c = 0; if(((unsigned long)(i & 0x4000)) != 0) t_c = -1;
  c_f = 0; if(((unsigned long)(i & 0x8000)) != 0) c_f = -1;

  switch(cmd)
  {
    /*----------------------------------------------------*/
    //pass

    case ALU_PASS_ID:   result = id & 0x0f;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    case ALU_PASS_IX:   result = ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    /*----------------------------------------------------*/

    case ALU_INC:       result = (id & 0x0f) +1;

                        q_c  = c_f; q_v  = 0;
                        q_tc = 0;   if(result > 0x0f) q_tc = -1;

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_DEC:       result = (id & 0x0f) -1;

                        q_c  = c_f; q_v = 0;
                        q_tc = -1;  if(result < 0) q_tc = 0;

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_ADDAC:     result = (id & 0x0f) + ix;
                        if(t_c) result++;

                        q_v  = 0;
                        q_tc = 0;   if(result > 0x0f) q_tc = -1;

                        q_c = q_tc;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_ADD0C:     result = id & 0x0f;
                        if(t_c) result++;

                        q_v  = 0;
                        q_tc = 0; if(result > 0x0f) q_tc = -1;

                        q_c = q_tc;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_ADC:       id &= 0x0f;
                        if(c_f) ix++;

                        result = id + ix;

                        q_v = 0;
                        if(!((id ^ ix) & 0x08) && ((id ^ result) & 0x08))
                        {
                          q_v = -1;
                        }

                        q_c = 0; if(result > 0x0f) q_c = -1;

                        q_tc = q_c;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;
    case ALU_SBC:       id &= 0x0f;
                        tmp = ix;
                        if(c_f == 0) ix++;

                        result =  id - ix;

                        q_v = 0;
                        if(((id ^ result) & 0x08) && ((id ^ tmp) & 0x08))
                        {
                         q_v = -1;
                        }

                        q_c = -1; if(result < 0) q_c = 0;

                        q_tc = q_c;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    /*----------------------------------------------------*/
    //logic

    case ALU_AND:       result = (id & 0x0f) & ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    case ALU_OR:        result = (id & 0x0f) | ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    case ALU_XOR:       result = (id & 0x0f) ^ ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    case ALU_BIT:       result = (id & 0x0f) & ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    /*----------------------------------------------------*/
    //shift

    case ALU_LSR:       result = (id >> 1) & 0x0f;

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, set C
                        q_v = 0; q_c = 0; q_tc = 0;

                        if(((int)(id & 0x01))!=0) {q_c = -1; q_tc = -1;}

                        break;

    case ALU_ASL:       result = (id << 1) & 0x0f;

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, set C
                        q_v = 0; q_c = 0; q_tc = 0;

                        if(((int)(id & 0x08))!=0) {q_c = -1; q_tc = -1;}

                        break;

    case ALU_ROR:       result = (id >> 1) & 0x0f;

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, set C
                        q_v = 0; q_c = 0; q_tc = 0;

                        if(((int)(id & 0x01))!=0) {q_c = -1; q_tc = -1;}

                        break;

    case ALU_ROL:       result = (id << 1) & 0x0f;
                        if(c_f) result |= 0x1;

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, set C
                        q_v = 0; q_c = 0; q_tc = 0;

                        if(((int)(id & 0x08))!=0) {q_c = -1; q_tc = -1;}

                        break;

    /*----------------------------------------------------*/
    //generate constant

    case ALU_C_00:      result = 0x00;
                        q_z = 1; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_01:      result = 0x01;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_FA:      result = 0x0a;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_FC:      result = 0x0c;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_FE:      result = 0x0e;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_FF:      result = 0x0f;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    /*----------------------------------------------------*/
    //Flag register manipulation.

    case ALU_CLC:       result = id & 0x0e;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_SEC:       result = id | 0x01;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_CLICLB:    result = id & 0x0b;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_SEICLB:    result = id | 0x04;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_CLD:       result = id & 0x07;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_SED:       result = id | 0x08;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_SEB:       result = id | 0x00;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_CLV:       result = id & 0x0f;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    /*----------------------------------------------------*/
    //modify Flags

    case ALU_MNZC:      result = id & 0x0c; //pass V,B,D,I trough

                        result |= (ix & 0x01); //C

                        //Z-Flag
                        if(((unsigned int)(ix & 0x06))==6) result |= 0x02;

                        break;

    case ALU_MNZVC:     result = id & 0x0c; //pass B,D,I trough

                        result |= (ix & 0x01); //C

                        //Z-Flag
                        if(((unsigned int)(ix & 0x06))==6) result |= 0x02;

                        break;

  }

  result &= 0x0f; 

  if(q_z ) result |= Q_Z;
  if(q_v ) result |= Q_V;
  if(q_c ) result |= Q_C;
  if(q_tc) result |= Q_TC;

  return(result);
}

/*-------------------------------------------------------------------------*/
//Bit 4..7

int rom_H(unsigned long i)
{
  int tmp;

  int cmd;     //command
  int id;      //input from internal Data Bus
  int ix;      //input from external Data Bus
  int t_c;     //temp carry from prev. cycle
  int c_f;     //carry flag

  int result = 0; //result
  int q_z    = 0; //Z flag output
  int q_v    = 0; //V flag output
  int q_c    = 0; //C flag output
  int q_tc   = 0; //C flag output (temp. C-Flag  for INC,DEC,addr.calculation)

  id  =  i       & 0x001f;
  ix  = (i >> 5) & 0x000f;
  cmd = (i >> 9) & 0x001f;

  t_c = 0; if(((unsigned long)(i & 0x4000)) != 0) t_c = -1;
  c_f = 0; if(((unsigned long)(i & 0x8000)) != 0) c_f = -1;

  switch(cmd)
  {
    /*----------------------------------------------------*/
    //pass

    case ALU_PASS_ID:   result = id & 0x0f;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    case ALU_PASS_IX:   result = ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    /*----------------------------------------------------*/
    //logic

    case ALU_AND:       result = (id & 0x0f) & ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    case ALU_OR:        result = (id & 0x0f) | ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    case ALU_XOR:       result = (id & 0x0f) ^ ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, and pass C trough.
                        q_v = 0; q_c = c_f; q_tc = t_c;

                        break;

    case ALU_BIT:       result = (id & 0x0f) & ix;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //pass C trough.
                        q_c = c_f; q_tc = t_c;

                        q_v = 0; if(((unsigned int)(ix & 0x04))!=0) q_v = -1;
                        result = ix & 0x0f; //set N-Flag

                        break;

    /*----------------------------------------------------*/

    case ALU_INC:       result = id & 0x0f;

                        if(t_c) result += 1;  //carry from Bit 0..3

                        q_v  = 0;
                        q_tc = 0;   if(result > 0x0f) q_tc = -1;

                        q_c  = c_f;

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_DEC:       result = id & 0x0f;

                        if(!t_c) result -= 1; //borrow from Bit 0..3

                        q_v  = 0;
                        q_tc = -1;  if(result < 0) q_tc = 0;

                        q_c  = c_f;

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_ADDAC:     result = (id & 0x0f) + ix;
                        if(t_c) result++;
                        q_v  = 0;
                        q_tc = 0;   if(result > 0x0f) q_tc = -1;

                        q_c = q_tc;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_ADD0C:     result = id & 0x0f;
                        if(t_c) result++;

                        q_v  = 0;
                        q_tc = 0;   if(result > 0x0f) q_tc = -1;

                        q_c = q_tc;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_ADC:       id &= 0x0f;
                        if(c_f) ix++;

                        result = id + ix;

                        q_v = 0;
                        if(!((id ^ ix) & 0x08) && ((id ^ result) & 0x08))
                        {
                         q_v = -1;
                        }

                        q_c = 0; if(result > 0x0f) q_c = -1;

                        q_tc = q_c;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    case ALU_SBC:       id &= 0x0f;
                        tmp = ix;
                        if(c_f == 0) ix++;

                        result =  id - ix;

                        q_v = 0;
                        if(((id ^ result) & 0x08) && ((id ^ tmp) & 0x08))
                        {
                          q_v = -1;                     
                        }

                        q_c = -1; if(result < 0) q_c = 0;

                        q_tc = q_c;

                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;

                        break;

    /*----------------------------------------------------*/
    //shift

    case ALU_LSR:       result = (id & 0x0f) >> 1; //ignore C-Flag

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, set C
                        q_v = 0; q_c = 0; q_tc = 0;

                        if(c_f) {q_c = -1; q_tc = -1;}//take carry from Bit 0

                        break;

    case ALU_ASL:       result = (id << 1) & 0x0f;
                        if(c_f) result |= 0x1;     //Bit 3

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, set C
                        q_v = 0; q_c = 0; q_tc = 0;

                        if(((int)(id & 0x08))!=0) {q_c = -1; q_tc = -1;}

                        break;

    case ALU_ROR:       result = (id >> 1) & 0x0f; get carry + Bit 4..7

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, set C
                        q_v = 0; q_c = 0; q_tc = 0;

                        if(c_f) {q_c = -1; q_tc = -1;}//take carry from Bit 0

                        break;

    case ALU_ROL:       result = (id << 1) & 0x0f;
                        if(c_f) result |= 0x1;     //Bit 3

                        q_z = 0;
                        //set Z-Flag, if result is zero.
                        if(((unsigned int)(result & 0x0f)) == 0) q_z = -1;
                        //clear V, set C
                        q_v = 0; q_c = 0; q_tc = 0;

                        if(((int)(id & 0x08))!=0) {q_c = -1; q_tc = -1;}

                        break;

    /*----------------------------------------------------*/
    //generate constant

    case ALU_C_00:      result = 0x00;
                        q_z = 1; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_01:      result = 0x00;
                        q_z = 1; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_FA:      result = 0x0f;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_FC:      result = 0x0f;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_FE:      result = 0x0f;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_C_FF:      result = 0x0f;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    /*----------------------------------------------------*/
    //Flag register manipulation.

    case ALU_CLC:       result = id & 0x0f;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_SEC:       result = id | 0x00;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_CLICLB:    result = id & 0x0e;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_SEICLB:    result = id & 0x0e;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_CLD:       result = id & 0x0f;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_SED:       result = id | 0x00;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_SEB:       result = id | 0x01;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    case ALU_CLV:       result = id & 0x0b;
                        q_z = 0; q_v = 0; q_c = c_f; q_tc = t_c;
                        break;

    /*----------------------------------------------------*/
    //modify Flags

    case ALU_MNZC:      result = id & 0x07; //pass V,B,D,I trough

                        //N-Flag
                        if(((unsigned int)(ix & 0x08))!=0) result |= 0x08;

                        break;

    case ALU_MNZVC:     result = id & 0x03; //pass B,D,I trough

                        //N-Flag
                        if(((unsigned int)(ix & 0x08))!=0) result |= 0x08;

                        //V-Flag
                        if(((unsigned int)(ix & 0x04))!=0) result |= 0x04;

                        break;

  }

  result &= 0x0f; 

  if(q_z ) result |= Q_Z;
  if(q_v ) result |= Q_V;
  if(q_c ) result |= Q_C;
  if(q_tc) result |= Q_TC;

  return(result);
}

/*-------------------------------------------------------------------------*/

int main(void)
{
 unsigned long i;
 char string[40];

 FILE *outstream_L;
 FILE *outstream_H;

 sprintf(string,"m1al.bin");
 printf("\nOpening Output File '%s'", string);
 outstream_L = fopen(string, "wb");

 if(outstream_L == NULL)
 {
        printf(" Error.\n");
        exit(1);
 }

 sprintf(string,"m1ah.bin");
 printf("\nOpening Output File '%s'", string);
 outstream_H = fopen(string, "wb");

 if(outstream_H == NULL)
 {
        printf(" Error.\n");
        exit(1);
 }

 for(i=0; i<0x10000; i++)
 {
        fputc(rom_L(i),outstream_L);
        fputc(rom_H(i),outstream_H);
 }

 fclose(outstream_L);
 fclose(outstream_H);

 printf("\nDone.\n");
 return(0);
}

