/*----------------------------------------------------------------* 
 * d03_1.c generates the binaries for the Address and Controll    *
 *         EPROMS, to be used in the D03 CRT/LCD controller.      *
 *                                                                *
 *(c) Dieter Mueller 2002,2003                                    *
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------* 
 * Description:
 *
 * ------------------------------------------------------------
 * Pixel:
 *
 *  SYS_CLK:     System clock, (typ. 16 MHz)
 *               one Pixel     takes  2 * SYS_CLK,
 *               one LCD clock takes  8 * SYS_CLK,
 *               one Character takes 16 * SYS_CLK.
 *
 * ------------------------------------------------------------
 * Horizontal:
 *
 *  H_TOTAL:        number of clock Pulses for one Line
 *              
 *  H_DISP_ON:      number of clock Pulses with enabled Display per Line
 *                  to enable the Shift Register and LCD Pixel Data
 *
 *  H_XSCL_ON:      number of clock Pulses with enabled LCD clock
 *
 *  H_SYNC_START:   defines the clock Cycle, that starts the
 *                  horizontal Sync Pulse
 *                  (counting starts at 0)
 *
 *  H_SYNC_WIDTH:   defines the width of the horizontal Sync Pulse
 *                  in clock Cycles
 *
 * ------------------------------------------------------------
 * Vertical:
 *
 *  V50_TOTAL:      number of Lines for a complete Frame (50 Hz)
 *  V60_TOTAL:      number of Lines for a complete Frame (60 Hz)
 *
 *  V_DISP_ON:      number of displayed Lines for a complete Frame
 *                  (8 * number of Character Rows)
 *
 *  V_XSCL_ON:      number of Lines with enabled LCD clock
 *
 *  V50_SYNC_START: defines the Line, that starts the 50 Hz VerticalSyncPulse
 *  V60_SYNC_START: defines the Line, that starts the 60 Hz VerticalSyncPulse
 *
 *  V_SYNC_WIDTH:   defines the width of the vertical Sync Pulse
 *                  in Lines
 *
 * ------------------------------------------------------------
 */



/********************************************************************/
/* Includes */

#include <stdlib.h>
#include <stdio.h>


/********************************************************************/

/* System Clock = 16 MHz */
#define SYS_CLK         16000000

/* Display Memory Start Address */
#define MEM_START       0x0400


/********************************************************************/
//Horizontal

/* values in (SYS_CLK / 16) pulses */

#define H_TOTAL           64
#define H_DISP_ON         40
#define Hlcd_XSCL_ON      40
#define H_SYNC_START      47
#define H_SYNC_WIDTH       5

/********************************************************************/
//Vertical

/* values in Lines (1 Line = 1 H_TOTAL) */
#define V50_TOTAL        312
#define V50_SYNC_START   250
#define V_DISP_ON        200
#define V_SYNC_WIDTH      10

#define V60_TOTAL        260
#define V60_SYNC_START   224

/* values in Lines (1 Line = 1 H_TOTAL) */
#define Vlcd_XSCL_ON        240

/********************************************************************/

/* Control Bits in Output Stream, from the d03 schematics. */

#define RESET_CNT       0x01
#define CRT_ON          0x02
#define HSYNC           0x04
#define VSYNC           0x08
#define XSCL_ON         0x10
#define SYNC            0x20
#define LCD_ON          0x40

/********************************************************************/

int  main        (void);
void report      (int v_total);

int  h_disp_on   (int x);
int  h_sync      (int x);
int  h_xscl_on   (int x);

int  v_disp_on   (int y);
int  v50_sync    (int y);
int  v60_sync    (int y);
int  v_xscl_on   (int y);

int  reset_cnt50 (int x, int y);
int  reset_cnt60 (int x, int y);


/********************************************************************/
/* Functions */


/* report Status */
void report(int v_total)
{
  float clock;
  float f_hor, f_vert;

  clock = SYS_CLK / 16; /* CHR clock */

  f_hor  = clock / (float)H_TOTAL;
  f_vert = f_hor / (float)v_total;

  printf("\n\n");
  printf("\n If CHR CLK = %10.4f MHz:", clock  / 1000000.0);
  printf("\n Hor.Freq   = %10.4f kHz ", f_hor  / 1000.0);
  printf("\n Vert.Freq. = %10.4f  Hz ", f_vert);


  printf("\n\n Control Lines:\n");

  printf(" RESET_CNT: 0x%02X\n", RESET_CNT);
  printf(" CRT_ON:    0x%02X\n", CRT_ON);
  printf(" HSYNC:     0x%02X\n", HSYNC);
  printf(" VSYNC:     0x%02X\n", VSYNC);
  printf(" XSCL_ON:   0x%02X\n", XSCL_ON);
  printf(" SYNC:      0x%02X\n", SYNC);
  printf(" LCD_ON:    0x%02X\n", LCD_ON);
  printf("\n");

  return;
}


/* hor. Display enabled ? */
int h_disp_on(int x)
{
  if(x < H_DISP_ON)  return(-1);
  else               return(0);

}

/* hor. Sync active ? */
int h_sync(int x)
{
  if(x <  H_SYNC_START)                 return(0);
  if(x >= H_SYNC_START  + H_SYNC_WIDTH) return(0);

  return(-1);
}

/* hor. LCD clock enabled ? */
int h_xscl_on(int x)
{
  if(x < Hlcd_XSCL_ON) return(-1);
  else                 return(0);
}

/* vert. Display enabled ? */
int v_disp_on(int y)
{
  if(y < V_DISP_ON)   return(-1);
  else                return(0);
}

/* vert. Sync active ? */
int v50_sync(int y)
{
  if(y <  V50_SYNC_START)                 return(0);
  if(y >= V50_SYNC_START  + V_SYNC_WIDTH) return(0);

  return(-1);
}
int v60_sync(int y)
{
  if(y <  V60_SYNC_START)                 return(0);
  if(y >= V60_SYNC_START  + V_SYNC_WIDTH) return(0);

  return(-1);
}

/* hor. LCD clock enabled ? */
int v_xscl_on(int y)
{
  if(y < Vlcd_XSCL_ON)   return(-1);
  else                   return(0);
}

/* reset Address Counter (last Cycle in Frame) ? */
int reset_cnt50(int x, int y)
{
  if(x < H_TOTAL   -1)  return(0);
  if(y < V50_TOTAL -1)  return(0);

  return(-1);
}
int reset_cnt60(int x, int y)
{
  if(x < H_TOTAL   -1)  return(0);
  if(y < V60_TOTAL -1)  return(0);

  return(-1);
}

/********************************************************************/
/* MAIN */

int main(void)
{
  char val;
  char str[20];

  long i,j;
  int  x,y,vsync;
  int  lcnt;


  FILE *outstream;

  /* open Output File */

  printf("\n V1.0 (c) by Dieter Mueller 2002");
  printf("\n D03_1.c: make ROMs for D03 Display");

 /*------------------------------------------------------------------------*/
  printf("\n\nControl Signals:\n");

  sprintf(str, "ctrl1.bin");
  printf("\n Writing Output File '%s' ", str);

  outstream = fopen(str ,"wb");
  if(outstream == NULL)
  {
     printf("\nFile open Error.\n");
     return(-1);
  }

  /*-----------------------------------------------------------------------*/
  printf("\n\n[60Hz]\n");
  report(V60_TOTAL);

  vsync = 0;
  for(y=0; y<V60_TOTAL; y++)
  {
     printf("\n    ");
     for(x=0; x<H_TOTAL; x++)
     {
        val = vsync;

        if(y==0)                          val |= VSYNC; //LCD vsync
        if(h_sync(x)&&v_xscl_on(y))       val |= HSYNC; //LCD hsync

        if(reset_cnt60(x,y))              val |= RESET_CNT; //reset counter
        if(h_disp_on(x) && v_disp_on(y))  val |= (CRT_ON | LCD_ON);
        if(h_xscl_on(x) && v_xscl_on(y))  val |= XSCL_ON;

        if(h_sync(x))
        {
          val |= SYNC;     //CRT hsync
          if(v60_sync(y))  vsync = SYNC; else vsync = 0; //CRT vsync                                        
        }

        fputc(val,outstream);
        printf("%c", 0x30+val);
     }
  }

  /* fill empty Bytes */
  /* default: reset Counter */
  for(i=0; i<32768 - (V60_TOTAL * H_TOTAL); i++) fputc(RESET_CNT,outstream);

  /*-----------------------------------------------------------------------*/
  printf("\n\n[50Hz]\n");
  report(V50_TOTAL);

  vsync = 0;
  for(y=0; y<V50_TOTAL; y++)
  {
     printf("\n    ");
     for(x=0; x<H_TOTAL; x++)
     {
        val = vsync;

        if(y==0)                          val |= VSYNC; //LCD vsync
        if(h_sync(x)&&v_xscl_on(y))       val |= HSYNC; //LCD hsync

        if(reset_cnt50(x,y))              val |= RESET_CNT; //reset counter
        if(h_disp_on(x) && v_disp_on(y))  val |= (CRT_ON | LCD_ON);
        if(h_xscl_on(x) && v_xscl_on(y))  val |= XSCL_ON;

        if(h_sync(x))
        {
          val |= SYNC;    //CRT hsync
          if(v50_sync(y)) vsync = SYNC; else vsync = 0; //CRT vsync
        }

        fputc(val,outstream);
        printf("%c", 0x30+val);
     }
  }

  /* fill empty Bytes */
  /* default: reset Counter */
  for(i=0; i<32768 - (V50_TOTAL * H_TOTAL); i++) fputc(RESET_CNT,outstream);


  /*-----------------------------------------------------------------------*/
  fclose(outstream);
  printf("\ndone.");


  /*-----------------------------------------------------------------------*/
  printf("\n\nAddress Table:  \n");

  sprintf(str, "addr1.bin");
  printf("\n Writing Output File '%s' ", str);

  outstream = fopen(str ,"wb");
  if(outstream == NULL)
  {
     printf("\nFile open Error.\n");
     return(-1);
  }

  /* WARNING: H_TOTAL has to be modulo 8. */
  /*-----------------------------------------------------------------------*/

  i = MEM_START;

  lcnt = 0; /* Line Counter */

  printf("\n\nMemory Display Address Counter: 60 Hz\n");
  for(y=0; y<V60_TOTAL; y++)
  {
    j = i;

    printf("\n[%d] ",lcnt);
    for(x=0; x<H_TOTAL; x++)
    {
        printf("%c",(j % 100 ) + 0x30);
        fputc(j >>3 ,outstream);

        /* increment counter, if display is on */
        if(h_disp_on(x) && v_disp_on(y)) j++;
    }

    if(lcnt++ >= 7) /* done with 8 Lines ? */
    {
      lcnt = 0;    /* reset Line Counter */
      i    = j;    /* advance Address Counter */
    }
  }

  /* fill empty Bytes */
  for(i=0; i<32768 - (V60_TOTAL * H_TOTAL); i++) fputc(0x00ff,outstream);

  /*-----------------------------------------------------------------------*/

  i = MEM_START;

  lcnt = 0; /* Line Counter */

  printf("\n\nMemory Display Address Counter: 50 Hz\n");
  for(y=0; y<V50_TOTAL; y++)
  {
    j = i;

    printf("\n[%d] ",lcnt);
    for(x=0; x<H_TOTAL; x++)
    {
        printf("%c",(j % 100 ) + 0x30);
        fputc(j >>3 ,outstream);

        /* increment counter, if display is on */
        if(h_disp_on(x) && v_disp_on(y)) j++;        
    }


    if(lcnt++ >= 7) /* done with 8 Lines ? */
    {
      lcnt = 0;    /* reset Line Counter */
      i    = j;    /* advance Address Counter */
    }
  }

  /* fill empty Bytes */
  for(i=0; i<32768 - (V50_TOTAL * H_TOTAL); i++) fputc(0x00ff,outstream);

  /*-----------------------------------------------------------------------*/
  fclose(outstream);
  printf("\ndone.");

  printf("\n");
  return(0);

}

