%{
#include "syntax.tab.h"
#include "tree.h"

extern int error_occurred;

/* 列号追踪变量 */
int yycolno = 1;

/* 每次匹配 Token 时自动更新位置信息 */
#define YY_USER_ACTION \
    yylloc.first_line = yylineno; \
    yylloc.first_column = yycolno; \
    yylloc.last_line = yylineno; \
    yylloc.last_column = yycolno + yyleng; \
    yycolno += yyleng;

void createTokenNode(char* name, NodeType type, char* text) {
    yylval.node = createNode(name, yylineno, type);
    if (strcmp(name, "ID") == 0 || strcmp(name, "TYPE") == 0) {
        yylval.node->value.str_val = strdup(text);
    } else if (strcmp(name, "INT") == 0) {
        /* 根据前缀判断进制：0x->16, 0->8, 其他->10 */
        if (strlen(text) > 1 && text[0] == '0' && (text[1] == 'x' || text[1] == 'X')) {
            yylval.node->value.int_val = (int)strtol(text, NULL, 16);
        } else if (strlen(text) > 1 && text[0] == '0') {
            yylval.node->value.int_val = (int)strtol(text, NULL, 8);
        } else {
            yylval.node->value.int_val = (int)strtol(text, NULL, 10);
        }
    } else if (strcmp(name, "FLOAT") == 0) {
        yylval.node->value.float_val = atof(text);
    } else if (strcmp(name, "CHAR") == 0) {
        yylval.node->value.str_val = strdup(text);
    }
}
%}

%option yylineno
/* 定义注释状态 */
%x COMMENT

digit   [0-9]
letter  [_a-zA-Z]
hex     [0-9a-fA-F]

%%

 /* ==========================================================
    1. 注释处理 (Comments)
    ========================================================== */
"//"[^\n]* { /* 单行注释：忽略直到行尾 */ }

"/*"        { BEGIN(COMMENT); }
<COMMENT>"*/" { BEGIN(INITIAL); }
<COMMENT>\n   { yycolno = 1; /* 重置列号 */ }
<COMMENT>.    { /* 忽略注释内容 */ }

 /* ==========================================================
    2. 关键字 (Keywords)
    ========================================================== */
"int"|"float"|"char" { createTokenNode("TYPE", TOKEN, yytext); return TYPE; }
"struct" { createTokenNode("STRUCT", TOKEN, yytext); return STRUCT; }
"return" { createTokenNode("RETURN", TOKEN, yytext); return RETURN; }
"if"     { createTokenNode("IF", TOKEN, yytext); return IF; }
"else"   { createTokenNode("ELSE", TOKEN, yytext); return ELSE; }
"while"  { createTokenNode("WHILE", TOKEN, yytext); return WHILE; }
"for"    { createTokenNode("FOR", TOKEN, yytext); return FOR; }
"break"    { createTokenNode("BREAK", TOKEN, yytext); return BREAK; }
"continue" { createTokenNode("CONTINUE", TOKEN, yytext); return CONTINUE; }

 /* ==========================================================
    3. 标识符与常量 (Identifiers & Constants)
    ========================================================== */
{letter}({letter}|{digit})* { createTokenNode("ID", TOKEN, yytext); return ID; }

 /* --- 十六进制整数 --- */
0[xX]{hex}+ { createTokenNode("INT", TOKEN, yytext); return INT; }

 /* Error Type A: 非法十六进制数 (如 0x5gg) */
0[xX][0-9a-fA-F]*[g-zG-Z]+[0-9a-fA-F]* {
    error_occurred = 1;
    printf("Error type A at Line %d: Illegal hexadecimal number \'%s\'\n", yylineno, yytext);
    createTokenNode("INT", TOKEN, "0");
    return INT;
}

 /* --- 八进制整数 (以0开头) --- */
0[0-7]+ { createTokenNode("INT", TOKEN, yytext); return INT; }

 /* Error Type A: 非法八进制 (如 018) */
0[0-9]+ {
    error_occurred = 1;
    printf("Error type A at Line %d: Illegal octal number \'%s\'\n", yylineno, yytext);
    createTokenNode("INT", TOKEN, "0");
    return INT;
}

 /* --- 普通整数 --- */
0|([1-9]{digit}*) { createTokenNode("INT", TOKEN, yytext); return INT; }

 /* --- 科学计数法浮点数 (如 1.2e-3) --- */
(({digit}+\.{digit}+)|({digit}+))[eE][+-]?{digit}+ {
    createTokenNode("FLOAT", TOKEN, yytext); 
    return FLOAT; 
}

 /* 普通浮点数 */
{digit}+\.{digit}+ { createTokenNode("FLOAT", TOKEN, yytext); return FLOAT; }

 /* Error Type A: 非法十六进制字符 */
\'\\x[0-9a-fA-F]{3,}\' {
    error_occurred = 1;
    printf("Error type A at Line %d: Illegal hexadecimal char \'%s\'\n", yylineno, yytext);
    createTokenNode("CHAR", TOKEN, yytext); return CHAR;
}
\'\\x[0-9a-fA-F]*[^0-9a-fA-F'\n]+[^'\n]*\' {
    error_occurred = 1;
    printf("Error type A at Line %d: Illegal hexadecimal char \'%s\'\n", yylineno, yytext);
    createTokenNode("CHAR", TOKEN, yytext); return CHAR;
}

'(\\.|[^'\\])*' { createTokenNode("CHAR", TOKEN, yytext); return CHAR; }

 /* Error Type A: 非法标识符 (数字开头) */
{digit}+{letter}({letter}|{digit})* {
    error_occurred = 1;
    printf("Error type A at Line %d: Invalid identifier \'%s\'\n", yylineno, yytext);
    createTokenNode("ID", TOKEN, yytext); return ID;
}

 /* ==========================================================
    4. 运算符与标点
    ========================================================== */
";" { createTokenNode("SEMI", TOKEN, yytext); return SEMI; }
"," { createTokenNode("COMMA", TOKEN, yytext); return COMMA; }
"=" { createTokenNode("ASSIGN", TOKEN, yytext); return ASSIGN; }
"<" { createTokenNode("LT", TOKEN, yytext); return LT; }
"<=" { createTokenNode("LE", TOKEN, yytext); return LE; }
">" { createTokenNode("GT", TOKEN, yytext); return GT; }
">=" { createTokenNode("GE", TOKEN, yytext); return GE; }
"!=" { createTokenNode("NE", TOKEN, yytext); return NE; }
"==" { createTokenNode("EQ", TOKEN, yytext); return EQ; }
"+" { createTokenNode("PLUS", TOKEN, yytext); return PLUS; }
"-" { createTokenNode("MINUS", TOKEN, yytext); return MINUS; }
"*" { createTokenNode("MUL", TOKEN, yytext); return MUL; }
"/" { createTokenNode("DIV", TOKEN, yytext); return DIV; }
"&&" { createTokenNode("AND", TOKEN, yytext); return AND; }
"||" { createTokenNode("OR", TOKEN, yytext); return OR; }
"." { createTokenNode("DOT", TOKEN, yytext); return DOT; }
"!" { createTokenNode("NOT", TOKEN, yytext); return NOT; }
"(" { createTokenNode("LP", TOKEN, yytext); return LP; }
")" { createTokenNode("RP", TOKEN, yytext); return RP; }
"[" { createTokenNode("LB", TOKEN, yytext); return LB; }
"]" { createTokenNode("RB", TOKEN, yytext); return RB; }
"{" { createTokenNode("LC", TOKEN, yytext); return LC; }
"}" { createTokenNode("RC", TOKEN, yytext); return RC; }

 /* 复合赋值运算符 */
"+=" { createTokenNode("ADD_ASSIGN", TOKEN, yytext); return ADD_ASSIGN; }
"-=" { createTokenNode("SUB_ASSIGN", TOKEN, yytext); return SUB_ASSIGN; }
"*=" { createTokenNode("MUL_ASSIGN", TOKEN, yytext); return MUL_ASSIGN; }
"/=" { createTokenNode("DIV_ASSIGN", TOKEN, yytext); return DIV_ASSIGN; }

 /* Error Type A: 不支持的位运算符 */
"&" { 
    error_occurred = 1; 
    printf("Error type A at Line %d: unknown lexeme %s\n", yylineno, yytext);
    createTokenNode("AND", TOKEN, yytext); 
    return AND; 
}
"|" { 
    error_occurred = 1; 
    printf("Error type A at Line %d: unknown lexeme %s\n", yylineno, yytext);
    createTokenNode("OR", TOKEN, yytext); 
    return OR; 
}

[\n]        { yycolno = 1; }
[ \t\r]     { /* 忽略空白 */ }

. { 
    error_occurred = 1; 
    printf("Error type A at Line %d: unknown lexeme %s\n", yylineno, yytext);
    createTokenNode("INT", TOKEN, "0");
    return INT; 
}

%%