%{
#include <stdio.h>
#include <string.h>
#include "tree.h"

extern int yylineno;
extern int error_occurred;

void yyerror(const char* msg);
int yylex();

struct TreeNode* root;
%}

/* 开启位置追踪 */
%locations
/* 开启详细报错 */
%define parse.error verbose

%union {
    struct TreeNode* node;
}

/* Tokens 定义 */
%token <node> INT FLOAT CHAR ID TYPE STRUCT IF ELSE WHILE RETURN FOR BREAK CONTINUE
%token <node> SEMI COMMA ASSIGN ADD_ASSIGN SUB_ASSIGN MUL_ASSIGN DIV_ASSIGN
%token <node> LT LE GT GE NE EQ PLUS MINUS MUL DIV AND OR DOT NOT LP RP LB RB LC RC

%type <node> Program ExtDefList ExtDef ExtDecList Specifier FunDec CompSt StmtList Stmt DefList Def DecList Dec VarDec VarList ParamDec Exp Args StructSpecifier

/* 优先级定义 (包含复合赋值) */
%right ASSIGN ADD_ASSIGN SUB_ASSIGN MUL_ASSIGN DIV_ASSIGN
%left OR
%left AND
%left LT LE GT GE NE EQ
%left PLUS MINUS
%left MUL DIV
%right NOT
%left LP RP LB RB DOT

%%

Program: ExtDefList {
    $$ = createNode("Program", $1 ? $1->lineno : yylineno, NOT_TOKEN);
    addChild($$, $1);
    root = $$; 
}
;

ExtDefList: ExtDef ExtDefList {
    $$ = createNode("ExtDefList", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
    addChild($$, $2);
}
| /* empty */ { $$ = NULL; }
;

ExtDef: Specifier ExtDecList SEMI {
    $$ = createNode("ExtDef", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Specifier SEMI {
    $$ = createNode("ExtDef", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
| Specifier FunDec CompSt {
    $$ = createNode("ExtDef", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| error SEMI { error_occurred = 1; }
;

ExtDecList: VarDec {
    $$ = createNode("ExtDecList", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| VarDec COMMA ExtDecList {
    $$ = createNode("ExtDecList", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
;

Specifier: TYPE {
    $$ = createNode("Specifier", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| StructSpecifier {
    $$ = createNode("Specifier", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
;

StructSpecifier: STRUCT ID LC DefList RC {
    $$ = createNode("StructSpecifier", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4); addChild($$, $5);
}
| STRUCT ID {
    $$ = createNode("StructSpecifier", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
;

VarDec: ID {
    $$ = createNode("VarDec", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| VarDec LB INT RB {
    $$ = createNode("VarDec", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4);
}
;

FunDec: ID LP VarList RP {
    $$ = createNode("FunDec", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4);
}
| ID LP RP {
    $$ = createNode("FunDec", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
;

VarList: ParamDec COMMA VarList {
    $$ = createNode("VarList", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| ParamDec {
    $$ = createNode("VarList", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
;

ParamDec: Specifier VarDec {
    $$ = createNode("ParamDec", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
;

CompSt: LC DefList StmtList RC {
    $$ = createNode("CompSt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4);
}
| LC DefList StmtList error { error_occurred = 1; }
;

StmtList: Stmt StmtList {
    $$ = createNode("StmtList", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
| /* empty */ { $$ = NULL; }
;

Stmt: Exp SEMI {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
| Exp error { error_occurred = 1; }
| CompSt {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| RETURN Exp SEMI {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| RETURN Exp error { error_occurred = 1; }
| IF LP Exp RP Stmt {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4); addChild($$, $5);
}
| IF LP Exp RP Stmt ELSE Stmt {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4); addChild($$, $5); addChild($$, $6); addChild($$, $7);
}
| WHILE LP Exp RP Stmt {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4); addChild($$, $5);
}
/* FOR 循环 */
| FOR LP Exp SEMI Exp SEMI Exp RP Stmt {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4);
    addChild($$, $5); addChild($$, $6); addChild($$, $7); addChild($$, $8); addChild($$, $9);
}
/* Break 和 Continue */
| BREAK SEMI {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
| CONTINUE SEMI {
    $$ = createNode("Stmt", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
| error SEMI { error_occurred = 1; }
;

DefList: Def DefList {
    $$ = createNode("DefList", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
| /* empty */ { $$ = NULL; }
;

Def: Specifier DecList SEMI {
    $$ = createNode("Def", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Specifier error SEMI { error_occurred = 1; }
;

DecList: Dec {
    $$ = createNode("DecList", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| Dec COMMA DecList {
    $$ = createNode("DecList", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
;

Dec: VarDec {
    $$ = createNode("Dec", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| VarDec ASSIGN Exp {
    $$ = createNode("Dec", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
;

/* 表达式（包括复合赋值） */
Exp: Exp ASSIGN Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp ADD_ASSIGN Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp SUB_ASSIGN Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp MUL_ASSIGN Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp DIV_ASSIGN Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp AND Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp OR Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp LT Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp LE Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp GT Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp GE Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp NE Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp EQ Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp PLUS Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp MINUS Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp MUL Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp DIV Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| LP Exp RP {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| MINUS Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
| NOT Exp {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2);
}
| ID LP Args RP {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4);
}
| ID LP RP {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp LB Exp RB {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3); addChild($$, $4);
}
| Exp DOT ID {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| ID {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| INT {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| FLOAT {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
| CHAR {
    $$ = createNode("Exp", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
;

Args: Exp COMMA Args {
    $$ = createNode("Args", $1->lineno, NOT_TOKEN);
    addChild($$, $1); addChild($$, $2); addChild($$, $3);
}
| Exp {
    $$ = createNode("Args", $1->lineno, NOT_TOKEN);
    addChild($$, $1);
}
;

%%

void yyerror(const char* msg) {
    error_occurred = 1;
    if (strstr(msg, "expecting SEMI") || strstr(msg, "unexpected RC")) {
        printf("Error type B at Line %d: Missing semicolon ';'\n", yylineno);
    } 
    else if (strstr(msg, "unexpected LC") && strstr(msg, "expecting TYPE or STRUCT or RP")) {
        printf("Error type B at Line %d: Missing closing parenthesis ')'\n", yylineno);
    }
    else if (strstr(msg, "unexpected TYPE")) {
        printf("Error type B at Line %d: Variable declaration is not allowed after statements\n", yylineno);
    }
    else {
        printf("Error type B at Line %d: %s\n", yylineno, msg);
    }
}