DECLARE procedures STRING;
DECLARE proceduresArray ARRAY<STRING>;
DECLARE i INT64 DEFAULT 0;

CREATE TABLE IF NOT EXISTS @@workflows_temp@@.WORKFLOWS_EXTENSIONS (
    name STRING,
    metadata STRING,
    procedures STRING
);

-- remove procedures from previous installations

SET procedures = (
    SELECT procedures
    FROM @@workflows_temp@@.WORKFLOWS_EXTENSIONS
    WHERE name = 'a5-extension'
);
IF (procedures IS NOT NULL) THEN
    SET proceduresArray = SPLIT(procedures, ',');
    LOOP
        SET i = i + 1;
        IF i > ARRAY_LENGTH(proceduresArray) THEN
            LEAVE;
        END IF;
        EXECUTE IMMEDIATE 'DROP PROCEDURE @@workflows_temp@@.' || proceduresArray[ORDINAL(i)];
    END LOOP;
END IF;

DELETE FROM @@workflows_temp@@.WORKFLOWS_EXTENSIONS
WHERE name = 'a5-extension';

-- create procedures

CREATE OR REPLACE PROCEDURE @@workflows_temp@@.`__proc_a5_celltoboundary_73610783`(
    input_table STRING,
    cell_column STRING,
    output_column_name STRING,
    output_table STRING,
    dry_run BOOLEAN,
    env_vars STRING
)
BEGIN
    DECLARE __parsed JSON default PARSE_JSON(env_vars);
    IF (dry_run) THEN
        BEGIN
        -- A5 Cell to Boundary Component - Dry Run
        -- Creates the output schema without executing the A5 function
        EXECUTE IMMEDIATE '''
        CREATE TABLE IF NOT EXISTS ''' || output_table || '''
        OPTIONS (expiration_timestamp = TIMESTAMP_ADD(CURRENT_TIMESTAMP(), INTERVAL 30 DAY))
        AS SELECT *, 
            "boundary_coordinates_json" AS ''' || output_column_name || '''
        FROM ''' || input_table || '''
        WHERE 1 = 0;
        '''; 
        END;
    ELSE
        BEGIN
        -- A5 Cell to Boundary Component
        -- Returns the vertices that define the boundary of an A5 cell using the A5 library
        -- Create the A5 cellToBoundary function
        CREATE TEMP FUNCTION cellToBoundary(cell STRING)
        RETURNS STRING
        LANGUAGE js
        OPTIONS (
            library = ["gs://carto-workflows-extension-a5/a5.umd.js"]
        )
        AS r"""
            if (cell === null) {
                return null;
            }
            const options = { closedRing: true, segments: 'auto' };
            try {
                const boundary = A5.cellToBoundary(A5.hexToBigInt(cell), options);
                // Convert to WKT format
                if (boundary && boundary.length > 0) {
                    const coords = boundary.map(coord => coord[0] + ' ' + coord[1]).join(', ');
                    return 'POLYGON((' + coords + '))';
                }
                return null;
            } catch (error) {
                return null;
            }
        """;
        -- Create the output table with A5 cell boundary coordinates column
        EXECUTE IMMEDIATE '''
        CREATE TABLE IF NOT EXISTS ''' || output_table || '''
        OPTIONS (expiration_timestamp = TIMESTAMP_ADD(CURRENT_TIMESTAMP(), INTERVAL 30 DAY))
        AS SELECT *, 
            cellToBoundary(CAST(''' || cell_column || ''' AS STRING)) AS ''' || output_column_name || '''
        FROM ''' || input_table || ''';
        '''; 
        END;
    END IF;
END;
CREATE OR REPLACE PROCEDURE @@workflows_temp@@.`__proc_a5_geotocell_6274027`(
    input_table STRING,
    geo_column STRING,
    resolution FLOAT64,
    output_column_name STRING,
    output_table STRING,
    dry_run BOOLEAN,
    env_vars STRING
)
BEGIN
    DECLARE __parsed JSON default PARSE_JSON(env_vars);
    IF (dry_run) THEN
        BEGIN
        -- A5 Geography to Cell Component - Dry Run
        -- Creates the output schema without executing the A5 function
        EXECUTE IMMEDIATE '''
        CREATE TABLE IF NOT EXISTS ''' || output_table || '''
        OPTIONS (expiration_timestamp = TIMESTAMP_ADD(CURRENT_TIMESTAMP(), INTERVAL 30 DAY))
        AS SELECT *, 
            "a5_cell_string" AS ''' || output_column_name || '''
        FROM ''' || input_table || '''
        WHERE 1 = 0;
        '''; 
        END;
    ELSE
        BEGIN
        -- A5 Geography to Cell Component
        -- Converts a GEOGRAPHY column to an A5 cell index using the A5 library
        -- Create the A5 lonLatToCell function
        CREATE TEMP FUNCTION lonLatToCell(lng FLOAT64, lat FLOAT64, res FLOAT64)
        RETURNS STRING
        LANGUAGE js
        OPTIONS (
            library = ["gs://carto-workflows-extension-a5/a5.umd.js"]
        )
        AS r"""
            if (lng === null || lat === null || res === null) {
                return null;
            }
            return A5.bigIntToHex(A5.lonLatToCell([lng, lat], res));
        """;
        -- Create the output table with A5 cell index column
        EXECUTE IMMEDIATE '''
        CREATE TABLE IF NOT EXISTS ''' || output_table || '''
        OPTIONS (expiration_timestamp = TIMESTAMP_ADD(CURRENT_TIMESTAMP(), INTERVAL 30 DAY))
        AS SELECT *, 
            lonLatToCell(ST_X(''' || geo_column || '''), ST_Y(''' || geo_column || '''), ''' || resolution || ''') AS ''' || output_column_name || '''
        FROM ''' || input_table || ''';
        '''; 
        END;
    END IF;
END;
CREATE OR REPLACE PROCEDURE @@workflows_temp@@.`__proc_a5_celltocenter_87363652`(
    input_table STRING,
    cell_column STRING,
    output_column_name STRING,
    output_table STRING,
    dry_run BOOLEAN,
    env_vars STRING
)
BEGIN
    DECLARE __parsed JSON default PARSE_JSON(env_vars);
    IF (dry_run) THEN
        BEGIN
        -- A5 Cell to Center Component - Dry Run
        -- Creates the output schema without executing the A5 function
        EXECUTE IMMEDIATE '''
        CREATE TABLE IF NOT EXISTS ''' || output_table || '''
        OPTIONS (expiration_timestamp = TIMESTAMP_ADD(CURRENT_TIMESTAMP(), INTERVAL 30 DAY))
        AS SELECT *, 
            "center_coordinate" AS ''' || output_column_name || '''
        FROM ''' || input_table || '''
        WHERE 1 = 0;
        '''; 
        END;
    ELSE
        BEGIN
        -- A5 Cell to Center Component
        -- Returns the geospatial coordinate at the center of an A5 cell using the A5 library
        -- Create the A5 cellToLonLat function
        CREATE TEMP FUNCTION cellToLonLat(cell STRING)
        RETURNS STRING
        LANGUAGE js
        OPTIONS (
            library = ["gs://carto-workflows-extension-a5/a5.umd.js"]
        )
        AS r"""
            if (cell === null) {
                return null;
            }
            try {
                const center = A5.cellToLonLat(A5.hexToBigInt(cell));
                return 'POINT(' + center[0] + ' ' + center[1] + ')';
            } catch (error) {
                return null;
            }
        """;
        -- Create the output table with A5 cell center coordinate column
        EXECUTE IMMEDIATE '''
        CREATE TABLE IF NOT EXISTS ''' || output_table || '''
        OPTIONS (expiration_timestamp = TIMESTAMP_ADD(CURRENT_TIMESTAMP(), INTERVAL 30 DAY))
        AS SELECT *, 
            cellToLonLat(CAST(''' || cell_column || ''' AS STRING)) AS ''' || output_column_name || '''
        FROM ''' || input_table || ''';
        '''; 
        END;
    END IF;
END;

-- add to extensions table

INSERT INTO @@workflows_temp@@.WORKFLOWS_EXTENSIONS (name, metadata, procedures)
VALUES ('a5-extension', '''{"name": "a5-extension", "title": "A5 Geospatial Extension", "industry": "Geospatial", "description": "Extension providing A5 geospatial indexing functions for BigQuery", "icon": "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iODAiIGhlaWdodD0iODAiIHZpZXdCb3g9IjAgMCA4MCA4MCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggb3BhY2l0eT0iMC44IiBmaWxsLXJ1bGU9ImV2ZW5vZGQiIGNsaXAtcnVsZT0iZXZlbm9kZCIgZD0iTTMzLjMzMzMgMzBDMzEuNDkyNCAzMCAzMCAzMS40OTI0IDMwIDMzLjMzMzNWNjYuNjY2N0MzMCA2OC41MDc2IDMxLjQ5MjQgNzAgMzMuMzMzMyA3MEg2Ni42NjY3QzY4LjUwNzYgNzAgNzAgNjguNTA3NiA3MCA2Ni42NjY3VjMzLjMzMzNDNzAgMzEuNDkyNCA2OC41MDc2IDMwIDY2LjY2NjcgMzBIMzMuMzMzM1pNNTAgNjVDNTguMjg0MyA2NSA2NSA1OC4yODQzIDY1IDUwQzY1IDQxLjcxNTcgNTguMjg0MyAzNSA1MCAzNUM0MS43MTU3IDM1IDM1IDQxLjcxNTcgMzUgNTBDMzUgNTguMjg0MyA0MS43MTU3IDY1IDUwIDY1WiIgZmlsbD0iIzAzNkZFMiIvPgo8cGF0aCBkPSJNNDguOTgzNiA2MS42NjY3VjUyLjVMNTEuMjUwMiA0Ny41SDU2LjI1MDJMNDguOTgzNiA2MS42NjY3WiIgZmlsbD0iIzAyNEQ5RSIvPgo8cGF0aCBkPSJNNDMuNzUgNTIuNTAwN0w1MS4yNSAzOC4zMzRWNDcuNTAwN0w0OC45ODMzIDUyLjUwMDdINDMuNzVaIiBmaWxsPSIjMDM2RkUyIi8+CjxwYXRoIG9wYWNpdHk9IjAuMyIgZD0iTTEzLjMzMzMgMTBDMTEuNDkyNCAxMCAxMCAxMS40OTI0IDEwIDEzLjMzMzNWNDYuNjY2N0MxMCA0OC41MDc2IDExLjQ5MjQgNTAgMTMuMzMzMyA1MEgyMFYyMy4zMzMzQzIwIDIxLjQ5MjQgMjEuNDkyNCAyMCAyMy4zMzMzIDIwSDUwVjEzLjMzMzNDNTAgMTEuNDkyNCA0OC41MDc2IDEwIDQ2LjY2NjcgMTBIMTMuMzMzM1oiIGZpbGw9IiMwMzZGRTIiLz4KPHBhdGggb3BhY2l0eT0iMC41IiBkPSJNMjMuMzMzMyAyMEMyMS40OTI0IDIwIDIwIDIxLjQ5MjQgMjAgMjMuMzMzM1Y1Ni42NjY3QzIwIDU4LjUwNzYgMjEuNDkyNCA2MCAyMy4zMzMzIDYwSDMwVjMzLjMzMzNDMzAgMzEuNDkyNCAzMS40OTI0IDMwIDMzLjMzMzMgMzBINjBWMjMuMzMzM0M2MCAyMS40OTI0IDU4LjUwNzYgMjAgNTYuNjY2NyAyMEgyMy4zMzMzWiIgZmlsbD0iIzAzNkZFMiIvPgo8L3N2Zz4K", "version": "1.0.0", "lastUpdate": "2024-01-31", "provider": "bigquery", "author": {"value": "CARTO", "link": {"label": "CARTO", "href": "https://carto.com"}}, "license": {"value": "CARTO Proprietary", "link": {"label": "CARTO Proprietary", "href": "https://carto.com/legal"}}, "details": [{"name": "Optional detail", "link": {"label": "Whatever", "href": "https://carto.com/"}}], "components": [{"name": "a5_celltoboundary", "title": "A5 Cell to Boundary", "description": "Returns the vertices that define the boundary of an A5 cell", "version": "1.0.0", "icon": "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjQiIGhlaWdodD0iNjQiIHZpZXdCb3g9IjAgMCA2NCA2NCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KICA8ZyB0cmFuc2Zvcm09InRyYW5zbGF0ZSgtNywgLTcpIHNjYWxlKDAuNSkgdHJhbnNsYXRlKDczLCA3MSkiPgogICAgPCEtLSBDZW50ZXIgcGVudGFnb24gLS0+CiAgICA8cG9seWdvbiBwb2ludHM9IjcsNyA3LDM5IDMyLjIzNTA5MzM4MTcxODQxNCw1OC42NzcxNDU2Nzc1NjA4NDQgNTguNzQ3MjQzNzEwMDQxNDI0LDQwLjc1NzcxODA5MjcxODY5IDM3LjQzMzgwODUyMTQ0NDkyLDE2Ljg4ODU0MzgxOTk5ODMyIiBmaWxsPSIjNjJhMGVhIiBmaWxsLW9wYWNpdHk9IjAuNSIgc3Ryb2tlPSIjMWE1ZmI0IiBzdHJva2Utd2lkdGg9IjQiLz4KICAgIDwhLS0gNSByb3RhdGVkIHBlbnRhZ29ucyAtLT4KICAgIDxwb2x5Z29uIHBvaW50cz0iNyw3IDcsMzkgMzIuMjM1MDkzMzgxNzE4NDE0LDU4LjY3NzE0NTY3NzU2MDg0NCA1OC43NDcyNDM3MTAwNDE0MjQsNDAuNzU3NzE4MDkyNzE4NjkgMzcuNDMzODA4NTIxNDQ0OTIsMTYuODg4NTQzODE5OTk4MzIiIGZpbGw9IiMwMjRkOWUiIGZpbGwtb3BhY2l0eT0iMC41IiBzdHJva2U9IiMxYTVmYjQiIHN0cm9rZS13aWR0aD0iNCIgdHJhbnNmb3JtPSJyb3RhdGUoNzIgNyA3KSIvPgogICAgPHBvbHlnb24gcG9pbnRzPSI3LDcgNywzOSAzMi4yMzUwOTMzODE3MTg0MTQsNTguNjc3MTQ1Njc3NTYwODQ0IDU4Ljc0NzI0MzcxMDA0MTQyNCw0MC43NTc3MTgwOTI3MTg2OSAzNy40MzM4MDg1MjE0NDQ5MiwxNi44ODg1NDM4MTk5OTgzMiIgZmlsbD0iIzQyOTllMCIgZmlsbC1vcGFjaXR5PSIwLjUiIHN0cm9rZT0iIzFhNWZiNCIgc3Ryb2tlLXdpZHRoPSI0IiB0cmFuc2Zvcm09InJvdGF0ZSgxNDQgNyA3KSIvPgogICAgPHBvbHlnb24gcG9pbnRzPSI3LDcgNywzOSAzMi4yMzUwOTMzODE3MTg0MTQsNTguNjc3MTQ1Njc3NTYwODQ0IDU4Ljc0NzI0MzcxMDA0MTQyNCw0MC43NTc3MTgwOTI3MTg2OSAzNy40MzM4MDg1MjE0NDQ5MiwxNi44ODg1NDM4MTk5OTgzMiIgZmlsbD0iIzk5YzFmMSIgZmlsbC1vcGFjaXR5PSIwLjUiIHN0cm9rZT0iIzFhNWZiNCIgc3Ryb2tlLXdpZHRoPSI0IiB0cmFuc2Zvcm09InJvdGF0ZSgyMTYgNyA3KSIvPgogICAgPHBvbHlnb24gcG9pbnRzPSI3LDcgNywzOSAzMi4yMzUwOTMzODE3MTg0MTQsNTguNjc3MTQ1Njc3NTYwODQ0IDU4Ljc0NzI0MzcxMDA0MTQyNCw0MC43NTc3MTgwOTI3MTg2OSAzNy40MzM4MDg1MjE0NDQ5MiwxNi44ODg1NDM4MTk5OTgzMiIgZmlsbD0iIzAzNmZlMiIgZmlsbC1vcGFjaXR5PSIwLjUiIHN0cm9rZT0iIzFhNWZiNCIgc3Ryb2tlLXdpZHRoPSI0IiB0cmFuc2Zvcm09InJvdGF0ZSgyODggNyA3KSIvPgogIDwvZz4KPC9zdmc+Cg==", "cartoEnvVars": [], "inputs": [{"name": "input_table", "title": "Input table", "description": "The table containing A5 cell identifiers", "type": "Table"}, {"name": "cell_column", "title": "Cell column", "description": "The column containing A5 cell identifiers", "type": "String", "placeholder": "a5_cell"}, {"name": "output_column_name", "title": "Output column name", "description": "Name for the output boundary coordinates column", "type": "String", "default": "a5_boundary", "placeholder": "a5_boundary"}], "outputs": [{"name": "output_table", "title": "Output table", "description": "The table with A5 cell boundary coordinates column added", "type": "Table"}], "group": "A5 Geospatial Extension", "procedureName": "__proc_a5_celltoboundary_73610783"}, {"name": "a5_geotocell", "title": "A5 Geography to Cell", "description": "Converts a GEOGRAPHY column to an A5 cell index at the specified resolution using the A5 library.", "version": "1.0.0", "icon": "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjQiIGhlaWdodD0iNjQiIHZpZXdCb3g9IjAgMCA2NCA2NCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KICA8ZyB0cmFuc2Zvcm09InRyYW5zbGF0ZSgtNywgLTcpIHNjYWxlKDAuNSkgdHJhbnNsYXRlKDczLCA3MSkiPgogICAgPCEtLSBDZW50ZXIgcGVudGFnb24gLS0+CiAgICA8cG9seWdvbiBwb2ludHM9IjcsNyA3LDM5IDMyLjIzNTA5MzM4MTcxODQxNCw1OC42NzcxNDU2Nzc1NjA4NDQgNTguNzQ3MjQzNzEwMDQxNDI0LDQwLjc1NzcxODA5MjcxODY5IDM3LjQzMzgwODUyMTQ0NDkyLDE2Ljg4ODU0MzgxOTk5ODMyIiBmaWxsPSIjNjJhMGVhIi8+CiAgICA8IS0tIDUgcm90YXRlZCBwZW50YWdvbnMgLS0+CiAgICA8cG9seWdvbiBwb2ludHM9IjcsNyA3LDM5IDMyLjIzNTA5MzM4MTcxODQxNCw1OC42NzcxNDU2Nzc1NjA4NDQgNTguNzQ3MjQzNzEwMDQxNDI0LDQwLjc1NzcxODA5MjcxODY5IDM3LjQzMzgwODUyMTQ0NDkyLDE2Ljg4ODU0MzgxOTk5ODMyIiBmaWxsPSIjMDI0ZDllIiB0cmFuc2Zvcm09InJvdGF0ZSg3MiA3IDcpIi8+CiAgICA8cG9seWdvbiBwb2ludHM9IjcsNyA3LDM5IDMyLjIzNTA5MzM4MTcxODQxNCw1OC42NzcxNDU2Nzc1NjA4NDQgNTguNzQ3MjQzNzEwMDQxNDI0LDQwLjc1NzcxODA5MjcxODY5IDM3LjQzMzgwODUyMTQ0NDkyLDE2Ljg4ODU0MzgxOTk5ODMyIiBmaWxsPSIjNDI5OWUwIiB0cmFuc2Zvcm09InJvdGF0ZSgxNDQgNyA3KSIvPgogICAgPHBvbHlnb24gcG9pbnRzPSI3LDcgNywzOSAzMi4yMzUwOTMzODE3MTg0MTQsNTguNjc3MTQ1Njc3NTYwODQ0IDU4Ljc0NzI0MzcxMDA0MTQyNCw0MC43NTc3MTgwOTI3MTg2OSAzNy40MzM4MDg1MjE0NDQ5MiwxNi44ODg1NDM4MTk5OTgzMiIgZmlsbD0iIzk5YzFmMSIgdHJhbnNmb3JtPSJyb3RhdGUoMjE2IDcgNykiLz4KICAgIDxwb2x5Z29uIHBvaW50cz0iNyw3IDcsMzkgMzIuMjM1MDkzMzgxNzE4NDE0LDU4LjY3NzE0NTY3NzU2MDg0NCA1OC43NDcyNDM3MTAwNDE0MjQsNDAuNzU3NzE4MDkyNzE4NjkgMzcuNDMzODA4NTIxNDQ0OTIsMTYuODg4NTQzODE5OTk4MzIiIGZpbGw9IiMwMzZmZTIiIHRyYW5zZm9ybT0icm90YXRlKDI4OCA3IDcpIi8+CiAgPC9nPgo8L3N2Zz4K", "cartoEnvVars": [], "inputs": [{"name": "input_table", "title": "Input table", "description": "The table containing the GEOGRAPHY column", "type": "Table"}, {"name": "geo_column", "title": "Geography column", "description": "The column containing GEOGRAPHY values", "type": "String", "placeholder": "geom"}, {"name": "resolution", "title": "A5 resolution", "description": "The A5 resolution level to index at", "type": "Number", "min": 0, "max": 20, "default": 9, "step": 0.1}, {"name": "output_column_name", "title": "Output column name", "description": "Name for the output A5 cell index column", "type": "String", "default": "a5_cell", "placeholder": "a5_cell"}], "outputs": [{"name": "output_table", "title": "Output table", "description": "The table with A5 cell index column added", "type": "Table"}], "group": "A5 Geospatial Extension", "procedureName": "__proc_a5_geotocell_6274027"}, {"name": "a5_celltocenter", "title": "A5 Cell to Center", "description": "Returns the geospatial coordinate at the center of an A5 cell.", "version": "1.0.0", "icon": "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjQiIGhlaWdodD0iNjQiIHZpZXdCb3g9IjAgMCA2NCA2NCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KICA8cG9seWdvbiBwb2ludHM9IjcsNyA3LDM5IDMyLjIzNTA5MzM4MTcxODQxNCw1OC42NzcxNDU2Nzc1NjA4NDQgNTguNzQ3MjQzNzEwMDQxNDI0LDQwLjc1NzcxODA5MjcxODY5IDM3LjQzMzgwODUyMTQ0NDkyLDE2Ljg4ODU0MzgxOTk5ODMyIiBmaWxsPSIjNjJhMGVhIiBzdHJva2U9IiMxYTVmYjQiIHN0cm9rZS13aWR0aD0iMiIvPgo8cGF0aCB0cmFuc2Zvcm09InRyYW5zbGF0ZSg2LCAxMCkiIGQ9Ik0yNCAyMkMyNS4xMDQ2IDIyIDI2IDIyLjg5NTQgMjYgMjRDMjYgMjUuMTA0NiAyNS4xMDQ2IDI2IDI0IDI2QzIyLjg5NTQgMjYgMjIgMjUuMTA0NiAyMiAyNEMyMiAyMi44OTU0IDIyLjg5NTQgMjIgMjQgMjJaIiBmaWxsPSIjMDI0RDlFIi8+CjxwYXRoIHRyYW5zZm9ybT0idHJhbnNsYXRlKDYsIDEwKSIgZmlsbC1ydWxlPSJldmVub2RkIiBjbGlwLXJ1bGU9ImV2ZW5vZGQiIGQ9Ik0yNS4wMDA5IDI2LjgyOUwyNSAzMEgyM0wyMy4wMDAxIDI2LjgyOTNDMjMuMzEyOCAyNi45Mzk5IDIzLjY0OTQgMjcgMjQgMjdDMjQuMzUxIDI3IDI0LjY4NzkgMjYuOTM5NyAyNS4wMDA5IDI2LjgyOVpNMzAgMjNWMjVMMjYuODI5IDI1LjAwMDlDMjYuOTM5NyAyNC42ODc5IDI3IDI0LjM1MSAyNyAyNEMyNyAyMy42NDk0IDI2LjkzOTkgMjMuMzEyOCAyNi44MjkzIDIzLjAwMDFMMzAgMjNaTTIxLjE3MDcgMjMuMDAwMUMyMS4wNjAxIDIzLjMxMjggMjEgMjMuNjQ5NCAyMSAyNEMyMSAyNC4zNTEgMjEuMDYwMyAyNC42ODc5IDIxLjE3MSAyNS4wMDA5TDE4IDI1VjIzTDIxLjE3MDcgMjMuMDAwMVpNMjUgMThMMjUuMDAwOSAyMS4xNzFDMjQuNjg3OSAyMS4wNjAzIDI0LjM1MSAyMSAyNCAyMUMyMy42NDk0IDIxIDIzLjMxMjggMjEuMDYwMSAyMy4wMDAxIDIxLjE3MDdMMjMgMThIMjVaIiBmaWxsPSIjMDM2RkUyIi8+Cjwvc3ZnPgo=", "cartoEnvVars": [], "inputs": [{"name": "input_table", "title": "Input table", "description": "The table containing A5 cell identifiers", "type": "Table"}, {"name": "cell_column", "title": "Cell column", "description": "The column containing A5 cell identifiers", "type": "String", "placeholder": "a5_cell"}, {"name": "output_column_name", "title": "Output column name", "description": "Name for the output center coordinate column", "type": "String", "default": "a5_center", "placeholder": "a5_center"}], "outputs": [{"name": "output_table", "title": "Output table", "description": "The table with A5 cell center coordinate column added", "type": "Table"}], "group": "A5 Geospatial Extension", "procedureName": "__proc_a5_celltocenter_87363652"}]}''', '__proc_a5_celltoboundary_73610783,__proc_a5_geotocell_6274027,__proc_a5_celltocenter_87363652');