% This file generates interpolated monthly GDP series constructed using the
% algorithm described in the paper 
% "Towards a Monthly Business Cycle Chronology for the Euro Area" by Emanuel Moench and Harald Uhlig,  
% Journal of Business Cycle Measurement and Analysis Vol. 2(1), May 2005. 
% We interpolate real GDP (GDPC96) using three monthly series as interpolators: IP (INDPRO), Civilian Employment (CE160V), 
% and Real Disposable Income (DSPIC96),all obtained from the St. Louis Fed website. 
% The interpolation method we employ is equivalent to the Chow-Lin approach, i.e. the model is estimated in levels, 
% allowing for serially correlated regression errors.  
% The first generated series is based on Kalman filtering, the second is derived from Kalman smoothing of the 
% estimated states, i.e. the smoothed interpolated series is based on the full sample information. 
% For further information, feel free to contact the authors at huhlig@uchicao.edu or emanuel.moench@ny.frb.org. 
% Note that these series and any comments or statements in this file represent the views of the authors only 
% and not those of the Federal Reserve Bank of New York or the Federal Reserve System. 

% Read in Data
[US_GDP96_qly, gdp_info] = xlsread('GDPC96.xls');
gdp_dateinfo = datevec(gdp_info(14:end,1));
time_GDP96 = [gdp_dateinfo(:,1)+(floor(gdp_dateinfo(:,2)/3)+mod(gdp_dateinfo(:,2),3))/10];

[US_IP_mly, ip_info] = xlsread('INDPRO.xls');
ip_dateinfo = datevec(ip_info(13:end,1));
time_US_IP = [ip_dateinfo(:,1)+ip_dateinfo(:,2)/100];

[US_RDI_mly, rdi_info] = xlsread('DSPIC96.xls');
rdi_dateinfo = datevec(rdi_info(14:end,1));
time_US_RDI = [rdi_dateinfo(:,1)+rdi_dateinfo(:,2)/100];

[US_empl_mly, emp_info] = xlsread('CivilianEmpl.xls'); % This is the series CE160V from the St. Louis Fed website
emp_dateinfo = datevec(emp_info(22:end,1)); % For some reason, Matlab was only able to read the file after renaming it
time_US_empl = [emp_dateinfo(:,1)+emp_dateinfo(:,2)/100];

min_length_ins = min([length(time_US_IP) length(time_US_RDI) length(time_US_empl)]);

% Interpolation
[GDP_ipol_filtered, GDP_ipol_smoothed] = kalman_ipol_bgw(US_GDP96_qly, [US_IP_mly(1:min_length_ins) US_RDI_mly(1:min_length_ins) US_empl_mly(1:min_length_ins)]);

format bank;
diary('US_GDP_ipol_ChowLin_6701_0712.txt');
disp('       Time    Filtered mly GDP    Smoothed mly GDP');
disp([time_US_IP(1:min_length_ins) GDP_ipol_filtered GDP_ipol_smoothed]);
diary off;
