% bb_with_ampl_crit.m determines cycles according to the Bry-Boschen procedure,
% Bry, Gerhard and Boschen, Charlotte (1971), "Cyclical Analysis of Time Series: 
%                                              Selected Procedures and Computer Programs"
%                                              NBER Technical Working Paper No. 20
% 
% Here, the minimum phase-length criterion in the original Bry-Boschan
% procedure has been replaced by a combined amplitude/phase-length criterion 
% ruling out expansions that are not longer than 21 months and during which annualized growth is lower than 1.5%  
%
%
% copyright : Emanuel Moench and Harald Uhlig
% Email: emanuel.moench@ny.frb.org and huhlig@uchicago.edu
% Commercial use is permitted, but not allowed to impinge on the free
% distribution of this program.
% No guarantee that this works. Use and modify at your own risk.
%
% ***************************************************************************
% **                                                                       **
% **  Inputs          ser : time series to be dated                        **
% **            startdate : first monthly observation, e.g. 1970           **
% **                        (only needed for plotting)                     **
% **                                                                       **
% **  Outputs  peak_bry_boschan : binary vector with same length as ser,   **
% **                              ones indicate identifed peaks            **
% **         trough_bry_boschan : binary vector, ones indicate troughs     **
% **                                                                       **
% ***************************************************************************


function [peak_bry_boschan, trough_bry_boschan] = bb_with_ampl_crit(ser, startdate);

len=length(ser);
ser_ma = 0*ser;

%%%%%%%%%%%%%%%%%% Setting program options
%%%%%%%%%%%%%%%%%% DO NOT CHANGE UNLESS YOU KNOW WHAT YOU ARE DOING

MA_WINDOW = 4;
DO_ELIMINATE_SHORT_CYCLES = 1;
MCD = 3;

%%%%%%%%%%%%%%%%%% Bry-Boschan Step I: Determination of extremes and substitution of values

% Nothing is done here.  If this is relevant, add code or clean the data by hand.

%%%%%%%%%%%%%%%%%% Bry-Boschan Step II: Determination of cycles in 12-month MA

LAG = MA_WINDOW;
LEAD = MA_WINDOW;
for j = LAG+1 : len - LEAD,
    ser_ma(j) = sum(ser(j-LAG:j+LEAD))/(LAG+LEAD+1);
end;
save 'ser_ma' ser_ma;
peak_ma = ser*0;
peak_ma_ind = [];
trough_ma = ser*0;
trough_ma_ind = [];
COMP = 5;
LOOK_FOR_PEAK = 0;
LOOK_FOR_TROUGH = 0;
LAST_PEAK = 1;
LAST_TROUGH = 1;
num_peaks = 1;
num_troughs = 1;
for j = LAG+1+COMP : len-LEAD-COMP,
    if min( ser_ma(j)*ones(2*COMP,1) > ser_ma([max(j-COMP:j-1,LAG+1),min(j+1:j+COMP,len-LEAD)]) ) == 1,
        if LOOK_FOR_TROUGH,
            if (ser_ma(j) > ser_ma(LAST_PEAK)) | (LAST_PEAK == 1),,
                peak_ma(j) = 1;
                peak_ma_ind(num_peaks-1) = j;
                peak_ma(LAST_PEAK) = 0;
                LAST_PEAK = j;
            end;
        else
            peak_ma(j) = 1;
            peak_ma_ind(num_peaks) = j;
            num_peaks = num_peaks + 1;
            LAST_PEAK = j;
            LOOK_FOR_TROUGH = 1;
            LOOK_FOR_PEAK = 0;
        end;     
    end;
    if min( ser_ma(j)*ones(2*COMP,1) < ser_ma([max(j-COMP:j-1,LAG+1),min(j+1:j+COMP,len-LEAD)]) ) == 1,
        if LOOK_FOR_PEAK,
            if (ser_ma(j) < ser_ma(LAST_TROUGH)) | (LAST_TROUGH == 1),
                trough_ma(j) = 1;
                trough_ma_ind(num_troughs-1) = j;
                trough_ma(LAST_TROUGH) = 0;
                LAST_TROUGH = j;
            end;
        else
            trough_ma(j) = 1;
            trough_ma_ind(num_troughs) = j;
            num_troughs = num_troughs + 1;
            LAST_TROUGH = j;
            LOOK_FOR_PEAK = 1;
            LOOK_FOR_TROUGH = 0;
        end;  
    end;
end;
num_troughs = num_troughs-1;
num_peaks = num_peaks - 1;


%%%%%%%%%%%%%%%%%% Bry-Boschan Step III: Determination of corresponding turns in Spencer curve

SPENCER_WEIGHTS = [ -0.0094, -0.0188, -0.0156, 0.0094, 0.0656, 0.1438, 0.2094, 0.2313, ...
        0.2094, 0.1438, 0.0656,  0.0094, -0.0156,  -0.0188,  -0.0094 ];
% source: King-Plosser, Real bus. cycles and the test of the Adelmans, Journal of Monetary Econ.,
% vol. 33 (1994), 405-438, footnote 2 on page 410
SPENCER_LAG = 7;
SPENCER_LEAD = 7;
ser_spencer = 0*ser;
for j = SPENCER_LAG+1 : len - SPENCER_LEAD,
    ser_spencer(j) = SPENCER_WEIGHTS*ser(j-SPENCER_LAG:j+SPENCER_LEAD);
end;
peak_spencer = ser*0;
peak_spencer_ind = [];
trough_spencer = ser*0;
trough_spencer_ind = [];
for j = 1 : num_peaks,
    [peak,peak_ind]=max(ser_spencer(min(max(SPENCER_LAG+1,peak_ma_ind(j)-5:peak_ma_ind(j)+5),len-SPENCER_LEAD)));
    peak_spencer_ind(j) = peak_ind + peak_ma_ind(j)-6;
    peak_spencer(peak_ind + peak_ma_ind(j)-6)=1;
end;
for j = 1 : num_troughs,
    [trough,trough_ind]=min(ser_spencer(min(max(SPENCER_LAG+1,trough_ma_ind(j)-5:trough_ma_ind(j)+5),len-SPENCER_LEAD)));
    trough_spencer_ind(j) = trough_ind + trough_ma_ind(j)-6;
    trough_spencer(trough_ind + trough_ma_ind(j)-6)=1;
end;

% check for minimal cycle length of 15 months
% if cycle is shorter than 15 months, then eliminate lower of two peaks and higher of two troughs
if DO_ELIMINATE_SHORT_CYCLES,
    ser_mod = ser_spencer;
    peak_old = peak_spencer_ind;
    trough_old = trough_spencer_ind;
    le_old = min(length(peak_old),length(trough_old));
    peak_new = [];
    trough_new = [];
    le_new = 0;
    for j = 1 : le_old,
        if j == 1,
            peak_new = peak_old(1);
            trough_new = trough_old(1);
            le_new = 1;
        else
            if (peak_old(j) - peak_new(le_new) >= 15) & (trough_old(j) - trough_new(le_new) >= 15),
                peak_new = [peak_new,peak_old(j)];
                trough_new = [trough_new,trough_old(j)];
                le_new = le_new+1; 
            else % eliminate the lower of the two peaks and the higher of the two troughs
                % but be careful not to mess up the alterations of peaks and troughs
                if ser_mod(peak_old(j)) > ser_mod(peak_new(le_new)),
                    higher_peak = peak_old(j);
                    lower_peak  = peak_new(le_new);
                else
                    higher_peak = peak_new(le_new);
                    lower_peak  = peak_old(j);
                end;             
                if ser_mod(trough_old(j)) < ser_mod(trough_new(le_new)),
                    lower_trough = trough_old(j);
                    higher_trough = trough_new(le_new);
                else
                    lower_trough = trough_new(le_new);
                    higher_trough = trough_old(j);
                end;
                % Check: is alteration of peaks and troughs still fine?
                % Then store the new result,
                % else eliminate a LOW trough or a HIGH peak
                if ((peak_new(le_new) < trough_new(le_new)) + (lower_trough < higher_trough)) == 1,
                    peak_new(le_new) = higher_peak;
                    trough_new(le_new) = lower_trough;
                else  % This case unfortunately is not clearly described in Bry-Boschan
                    % We shall keep the higher of the two peaks, if the distance between this peak and the higher trough 
                    % is larger than the distance between the lower peak and the lower trough, else keep the lower of the two troughs.
                    % It probably is better to eliminate one of the earlier or one of the latter troughs or peaks,
                    % but that necessitates even more complicated comparisons.
                    if (abs(higher_trough - higher_peak)) > (abs(lower_trough - lower_peak)),
                        peak_new(le_new) = higher_peak;
                        trough_new(le_new) = higher_trough;
                    else
                        peak_new(le_new) = lower_peak;
                        trough_new(le_new) = lower_trough;
                    end;                
                end;
            end;
        end;
    end;
    % Now, take care of the last additional peak or trough, if it is there, i.e.
    % if there are not exactly as many peaks and troughs already.
    if length(peak_old) > le_old,
        if (peak_old(le_old + 1) - peak_new(le_new) >= 15),
            peak_new = [peak_new,peak_old(le_old + 1)];
        else % eliminate the lower of the two peaks and the higher of the two troughs
            % but be careful not to mess up the alterations of peaks and troughs
            if ser_mod(peak_old(le_old + 1)) > ser_mod(peak_new(le_new)), 
                % the only interesting case!
                % Question: should one eliminate the previous trough and peak?     
                if le_new == 1,
                    peak_new(le_new) = peak_old(le_old + 1);
                else
                    if ser_mod(peak_old(le_old+1))-ser_mod(peak_new(le_new)) > ...
                            ser_mod(trough_new(le_new-1))-ser_mod(trough_new(le_new)),
                        % This case unfortunately is not clearly described in Bry-Boschan
                        % We shall keep the higher of the two peaks, if the distance between the two peaks
                        % is larger than the distance between the two troughs, else keep the lower of the two troughs.
                        % It probably is better to eliminate one of the earlier or one of the latter troughs or peaks,
                        % but that necessitates even more complicated comparisons.
                        peak_new(le_new) = peak_old(le_old+1);
                        trough_new = trough_new(1:le_new-1);
                        le_new = le_new - 1;
                    end;
                end;
            end;
        end;
    end;
    if length(trough_old) > le_old,
        if (trough_old(le_old + 1) - trough_new(le_new) >= 15),
            trough_new = [trough_new,trough_old(le_old + 1)],
        else % eliminate the lower of the two peaks and the higher of the two troughs
            % but be careful not to mess up the alterations of peaks and troughs
            if ser_mod(trough_old(le_old + 1)) < ser_mod(trough_new(le_new)), 
                % the only interesting case!
                % Question: should one eliminate the previous trough and peak?     
                if le_new == 1,
                    trough_new(le_new) = trough_old(le_old + 1);
                else
                    if ser_mod(peak_new(le_new))-ser_mod(peak_new(le_new-1)) <  ...
                            ser_mod(trough_new(le_new))-ser_mod(trough_old(le_old + 1)),
                        % This case unfortunately is not clearly described in Bry-Boschan
                        % We shall keep the higher of the two peaks, if the distance between the two peaks
                        % is larger than the distance between the two troughs, else keep the lower of the two troughs.
                        % It probably is better to eliminate one of the earlier or one of the latter troughs or peaks,
                        % but that necessitates even more complicated comparisons.
                        trough_new(le_new) = trough_old(le_old+1);
                        peak_new = peak_new(1:le_new-1);
                        le_new = le_new - 1;
                    end;
                end;
            end;
        end;
    end;
    peak_ser = zeros(len,1);
    trough_ser = zeros(len,1);
    for j = 1 : length(peak_new),
        peak_ser(peak_new(j)) = 1;
    end;
    for j = 1 : length(trough_new),
        trough_ser(trough_new(j)) = 1;
    end; 
    
    peak_spencer_ind = peak_new;
    trough_spencer_ind = trough_new;
    peak_spencer = peak_ser;
    trough_spencer = trough_ser;    
end;


num_peaks = sum(nantozero(peak_spencer));
num_troughs = sum(nantozero(trough_spencer));


%%%%%%%%%%%%%%%%%% Bry-Boschan Step IV: Determination of corresponding turns in short-term moving average of 3 to 6 months, depending on MCD

% Looking for turning points in neighbourhood +- 5 months of Spencer curve turns

MCD_LAG = MCD;
MCD_LEAD = MCD;
ser_mcd = 0*ser;
for j = MCD_LAG+1 : len - MCD_LEAD,
    ser_mcd(j) = sum(ser(j-MCD_LAG:j+MCD_LEAD))/(MCD_LAG+MCD_LEAD+1);
end;

peak_mcd = ser*0;
peak_mcd_ind = [];
trough_mcd = ser*0;
trough_mcd_ind = [];

for j = 1 : num_peaks,
    [peak,peak_ind]=max(ser_mcd(min(max(MCD_LAG+1,peak_spencer_ind(j)-5:peak_spencer_ind(j)+5),len-MCD_LEAD)));
    peak_mcd_ind(j) = peak_ind + peak_spencer_ind(j)-6;
    peak_mcd(peak_ind + peak_spencer_ind(j)-6)=1;
end;
for j = 1 : num_troughs,
    [trough,trough_ind]=min(ser_mcd(min(max(MCD_LAG+1,trough_spencer_ind(j)-5:trough_spencer_ind(j)+5),len-MCD_LEAD)));
    trough_mcd_ind(j) = trough_ind + trough_spencer_ind(j)-6;
    trough_mcd(trough_ind + trough_spencer_ind(j)-6)=1;
end;
num_peaks = sum(nantozero(peak_mcd));
num_troughs = sum(nantozero(trough_mcd));

%%%%%%%%%%%%%%%%%% Bry-Boschan Step V: Determination of turning points in unsmoothed series

% Determination of turning points in unsmoothed series
% Identification of turns within +- 4 months neighbourhood of the MCD peaks and troughs 

LAG = 4;
LEAD = 4;
peak_unsmoothed = ser*0;
peak_unsmoothed_ind = [];
trough_unsmoothed = ser*0;
trough_unsmoothed_ind = [];

for j = 1 : num_peaks,
    [peak,peak_ind]=max(ser(min(max(6,peak_mcd_ind(j)-4:peak_mcd_ind(j)+4),len-6)));
    peak_unsmoothed_ind(j) = peak_ind + peak_mcd_ind(j)-5;
    peak_unsmoothed(peak_ind + peak_mcd_ind(j)-5)=1;
end;
for j = 1 : num_troughs,
    [trough,trough_ind]=min(ser(min(max(6,trough_mcd_ind(j)-4:trough_mcd_ind(j)+4),len-6)));
    trough_unsmoothed_ind(j) = trough_ind + trough_mcd_ind(j)-5;
    trough_unsmoothed(trough_ind + trough_mcd_ind(j)-5)=1;
end;

% check for minimal cycle length of 15 months
% if cycle is shorter than 15 months, then eliminate lower of two peaks and higher of two troughs
% check for minimal cycle length of 15 months
% if cycle is shorter than 15 months, then eliminate lower of two peaks and higher of two troughs
if DO_ELIMINATE_SHORT_CYCLES,
    ser_mod = ser;
    peak_old = peak_unsmoothed_ind;
    trough_old = trough_unsmoothed_ind;
    le_old = min(length(peak_old),length(trough_old));
    peak_new = [];
    trough_new = [];
    le_new = 0;
    for j = 1 : le_old,
        if j == 1,
            peak_new = peak_old(1);
            trough_new = trough_old(1);
            le_new = 1;
        else
            if (peak_old(j) - peak_new(le_new) >= 15) & (trough_old(j) - trough_new(le_new) >= 15),
                peak_new = [peak_new,peak_old(j)];
                trough_new = [trough_new,trough_old(j)];
                le_new = le_new+1; 
            else % eliminate the lower of the two peaks and the higher of the two troughs
                % but be careful not to mess up the alterations of peaks and troughs
                if ser_mod(peak_old(j)) > ser_mod(peak_new(le_new)),
                    higher_peak = peak_old(j);
                    lower_peak  = peak_new(le_new);
                else
                    higher_peak = peak_new(le_new);
                    lower_peak  = peak_old(j);
                end;             
                if ser_mod(trough_old(j)) < ser_mod(trough_new(le_new)),
                    lower_trough = trough_old(j);
                    higher_trough = trough_new(le_new);
                else
                    lower_trough = trough_new(le_new);
                    higher_trough = trough_old(j);
                end;
                % Check: is alteration of peaks and troughs still fine?
                % Then store the new result,
                % else eliminate a LOW trough or a HIGH peak
                if ((higher_peak > lower_trough) + (peak_new(le_new) < trough_new(le_new))) == 1,
                    peak_new(le_new) = higher_peak;
                    trough_new(le_new) = lower_trough;
                else  % This case unfortunately is not clearly described in Bry-Boschan
                    % We shall keep the higher of the two peaks, if the distance between this peak and the higher trough 
                    % is larger than the distance between the lower peak and the lower trough, else keep the lower of the two troughs.
                    % It probably is better to eliminate one of the earlier or one of the latter troughs or peaks,
                    % but that necessitates even more complicated comparisons.
                    if (abs(higher_trough - higher_peak)) > (abs(lower_trough - lower_peak)),
                        peak_new(le_new) = higher_peak;
                        trough_new(le_new) = higher_trough;
                    else
                        peak_new(le_new) = lower_peak;
                        trough_new(le_new) = lower_trough;
                    end;
                end;
            end;
        end;
    end;
    % Now, take care of the last additional peak or trough, if it is there, i.e.
    % if there are not exactly as many peaks and troughs already.
    if length(peak_old) > le_old,
        if (peak_old(le_old + 1) - peak_new(le_new) >= 15),
            peak_new = [peak_new,peak_old(le_old + 1)];
        else % eliminate the lower of the two peaks and the higher of the two troughs
            % but be careful not to mess up the alterations of peaks and troughs
            if ser_mod(peak_old(le_old + 1)) > ser_mod(peak_new(le_new)), 
                % the only interesting case!
                % Question: should one eliminate the previous trough and peak?     
                if le_new == 1,
                    peak_new(le_new) = peak_old(le_old + 1);
                else
                    if ser_mod(peak_old(le_old+1))-ser_mod(peak_new(le_new)) > ...
                            ser_mod(trough_new(le_new-1))-ser_mod(trough_new(le_new)),
                        % This case unfortunately is not clearly described in Bry-Boschan
                        % We shall keep the higher of the two peaks, if the distance between the two peaks
                        % is larger than the distance between the two troughs, else keep the lower of the two troughs.
                        % It probably is better to eliminate one of the earlier or one of the latter troughs or peaks,
                        % but that necessitates even more complicated comparisons.
                        peak_new(le_new) = peak_old(le_old+1);
                        trough_new = trough_new(1:le_new-1);
                        le_new = le_new - 1;
                    end;
                end;
            end;
        end;
    end;
    if length(trough_old) > le_old,
        if (trough_old(le_old + 1) - trough_new(le_new) >= 15),
            trough_new = [trough_new,trough_old(le_old + 1)],
        else % eliminate the lower of the two peaks and the higher of the two troughs
            % but be careful not to mess up the alterations of peaks and troughs
            if ser_mod(trough_old(le_old + 1)) < ser_mod(trough_new(le_new)), 
                % the only interesting case!
                % Question: should one eliminate the previous trough and peak?     
                if le_new == 1,
                    trough_new(le_new) = trough_old(le_old + 1);
                else
                    if ser_mod(peak_new(le_new))-ser_mod(peak_new(le_new-1)) <  ...
                            ser_mod(trough_new(le_new))-ser_mod(trough_old(le_old + 1)),
                        % This case unfortunately is not clearly described in Bry-Boschan
                        % We shall keep the higher of the two peaks, if the distance between the two peaks
                        % is larger than the distance between the two troughs, else keep the lower of the two troughs.
                        % It probably is better to eliminate one of the earlier or one of the latter troughs or peaks,
                        % but that necessitates even more complicated comparisons.
                        trough_new(le_new) = trough_old(le_old+1);
                        peak_new = peak_new(1:le_new-1);
                        le_new = le_new - 1;
                    end;
                end;
            end;
        end;
    end;
    peak_ser = zeros(len,1);
    trough_ser = zeros(len,1);
    for j = 1 : length(peak_new),
        peak_ser(peak_new(j)) = 1;
    end;
    for j = 1 : length(trough_new),
        trough_ser(trough_new(j)) = 1;
    end; 
    
    peak_unsmoothed_ind = peak_new;
    trough_unsmoothed_ind = trough_new;
    peak_unsmoothed = peak_ser;
    trough_unsmoothed = trough_ser;    
end;

turn_unsmoothed = peak_unsmoothed + trough_unsmoothed;
turn_unsmoothed_ind = sort([peak_unsmoothed_ind trough_unsmoothed_ind]);


% *************************************************************************
% Combined amplitude/phase-length criterion, replacing the minimum phase length criterion in the original Bry-Boschan procedure
% Our criterion rules out expansions that are shorter than 21 months and during wihich growth is below an annual rate of 1.5 

peak_bry_boschan = peak_unsmoothed;
peak_bry_boschan_ind = peak_unsmoothed_ind;
trough_bry_boschan = trough_unsmoothed;
trough_bry_boschan_ind = trough_unsmoothed_ind;
peak_final = peak_bry_boschan;
trough_final = trough_bry_boschan;

ser_growth = diff(log(ser));
ser_growth = ser_growth(find(nantozero(ser_growth)));
ser_growth = ser_growth(1:length(ser_growth)-1);
std_neg = std(ser_growth(find(ser_growth < 0)));
std_pos = std(ser_growth(find(ser_growth >= 0)));
std_growth = std(ser_growth);

trend = ser - detrend(ser,'linear');
trend_growth = log(trend(2)) - log(trend(1));

for i = 1 : num_peaks-1,
    if (peak_bry_boschan_ind(i+1) - trough_bry_boschan_ind(i) <= 21) && ... 
        ((log(ser(peak_bry_boschan_ind(i+1)))-log(ser(trough_bry_boschan_ind(i))))/(peak_bry_boschan_ind(i+1) - trough_bry_boschan_ind(i)) <= 0.015/12),
        peak_final(peak_bry_boschan_ind(i+1)) = 0;
        trough_final(trough_bry_boschan_ind(i)) = 0;
    end;
end;
        
peak_final_ind = [];
trough_final_ind = [];
for i = 1 : length(peak_final),
    if peak_final(i) == 1; peak_final_ind = [peak_final_ind i];
    elseif trough_final(i) == 1; trough_final_ind = [trough_final_ind i];    
    end;
end;

peak_bry_boschan = peak_final;
peak_bry_boschan_ind = peak_final_ind;
trough_bry_boschan = trough_final;
trough_bry_boschan_ind = trough_final_ind;


% *************************************************************************
% Plotting the result

titlestring = 'Series with identified turning points';
done = plot_shade_with_tps(ser,startdate,peak_bry_boschan,trough_bry_boschan,peak_bry_boschan,trough_bry_boschan,titlestring);
