//============== Copyright 1999 - 2008, Sungan., All rights reserved. ==================
//
// Purpose :
//
// Programed by sungan
//
//======================================================================================

#include "StdAfx.h"
#include "EmulationVI.h"

#define VK_TO_ASCII(vk)			((vk)<<8)
#define SHIFT_VK_TO_ASCII(vk)	(0x01|((vk)<<8))
#define CTRL_VK_TO_ASCII(vk)	(0x02|((vk)<<8))
#define SHIFT_CTRL_VK_TO_ASCII(vk)	(0x03|((vk)<<8))
#define ASCII_TO_VK(asc)		((asc)>>8)
#define HAS_SHIFT_VK(asc)		((asc)>>8 && (asc)&0x1)
#define HAS_CTRL_VK(asc)		((asc)>>8 && (asc)&0x2)

static wchar_t Win32KeyTable[256][3] = 
{
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x00
	{ VK_TO_ASCII(VK_BACK), 0 , 0 },{VK_TO_ASCII(VK_TAB),0, 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{0x0D,0x0D,0},{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x08
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x10
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{27 ,27 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x18
	{' ',' ', 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{VK_TO_ASCII(0x23),0,0},{VK_TO_ASCII(0x24),0,0},{VK_TO_ASCII(0x25),0,0},{VK_TO_ASCII(0x26),0,0},{VK_TO_ASCII(0x27),0,0},//VK_TO_ASCII(0x20
	{VK_TO_ASCII(0x28),0,0},{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{VK_TO_ASCII(0x2D),0,0},{VK_TO_ASCII(0x2E),0,0},{ 0 , 0 , 0 },//0x28
	{'0',')', 0 },{'1','!', 0 },{'2','@', 0 },{'3','#', 0 },{'4','$', 0 },{'5','%', 0 },{'6','^', 0 },{'7','&', 0 },//0x30
	{'8','*', 0 },{'9','(', 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x38
	{ 0 , 0 , 0 },{'a','A', 1 },{'b','B', 2 },{'c','C', 3 },{'d','D', 4 },{'e','E', 5 },{'f','F', 6 },{'g','G', 7 },//0x40
	{'h','H', 0 },{'i','I', 9 },{'j','J',10 },{'k','K',11 },{'l','L',12 },{'m','M',13 },{'n','N',14 },{'o','O',15 },//0x48
	{'p','P',16 },{'q','Q',17 },{'r','R',18 },{'s','S', 0 },{'t','T',20 },{'u','U',21 },{'v','V',22 },{'w','W',23 },//0x50
	{'x','X',24 },{'y','Y', 0 },{'z','Z', 0 },{'[','{',27 },{'\\','|',28},{']','}',29 },{ 0 , 0 ,30 },{ 0 , 0 ,31 },//0x58
	{'0','0', 0 },{'1','1', 0 },{'2','2', 0 },{'3','3', 0 },{'4','4', 0 },{'5','5', 0 },{'6','6', 0 },{'7','7', 0 },//0x60
	{'8','8', 0 },{'9','9', 0 },{'*','*', 0 },{'+','+', 0 },{ 0 , 0 , 0 },{'-','-', 0 },{'.','.', 0 },{'/','/', 0 },//0x68
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{VK_TO_ASCII(VK_F3),0, 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x70
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x78
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x80
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x88
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x90
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0x98
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xa0
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xa8
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xb0
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{';',':', 0 },{'=','+', 0 },{',','<', 0 },{'-','_', 0 },{'.','>', 0 },{'/','?', 0 },//0xb8
	{'`','~', 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xc0
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xc8
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xd0
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{'[','{', 27},{'\\','|',0 },{ 0 , 0 , 0 },{'\'','\"', 0 },{ 0 , 0 , 0 },//0xd8
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{VK_TO_ASCII(0xE5),0,0},{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xe0
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xe8
	{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },{ 0 , 0 , 0 },//0xf0
};
namespace
{
	void InitWin32KeyTable()
	{
		for(TCHAR i = 0; i < 0xFF; ++i)
		{
			SHORT ret = VkKeyScan(i);
			BYTE vk = LOBYTE(ret);
			BYTE state = HIBYTE(ret);
			if(state <= 2)
				Win32KeyTable[vk][state] = i;
		}

	}
	int CountLastCR(std::wstring &buf)
	{
		int count = 0;
		size_t pos = buf.length()-1;
		while(pos >= 0 && (buf[pos] == '\n' || buf[pos] == '\r'))
		{
			if(buf[pos] == '\n')
				++count;
			++pos;
		}
		return count;
	}

	void EraseLastCR(std::wstring &buf)
	{
		if(!buf.empty() && buf[buf.length()-1] == '\n')
			buf.erase(buf.end()-1);
		if(!buf.empty() && buf[buf.length()-1] == '\r')
			buf.erase(buf.end()-1);
	}
	bool GetClipboardText(int idx, std::wstring &buf)
	{
		bool success = false;
		if(::OpenClipboard(NULL))
		{
			if(::IsClipboardFormatAvailable(CF_UNICODETEXT))
			{
				HANDLE hData = ::GetClipboardData(CF_UNICODETEXT);
				LPCWSTR pData = (LPCWSTR)::GlobalLock(hData);
				buf = pData;
				::GlobalUnlock(hData);
				success = true;
			}
			::CloseClipboard();
		}
		return success;
	}
}
namespace
{
	/* Editor Utils*/
	POINT_PTR SetCaret(HWND hWnd, LONG_PTR x, LONG_PTR y, bool select = false)
	{
		POINT_PTR pos = {x,y};
		POINT_PTR cur;
		Editor_GetCaretPos(hWnd, POS_LOGICAL_W, &cur);
		if(cur.x != pos.x || cur.y != pos.y)
		{
			Editor_SetCaretPosEx(hWnd, POS_LOGICAL_W, &pos, select);
			Editor_GetCaretPos(hWnd, POS_LOGICAL_W, &cur);
		}
		return cur;
	}

	UINT_PTR GetLineLength(HWND hWnd, LONG_PTR y)
	{
		GET_LINE_INFO info;
		info.cch = 0;
		info.flags = FLAG_LOGICAL;
		info.yLine = y;
		return Editor_GetLineW(hWnd, &info, NULL); 
	}

	POINT_PTR MoveCaret(HWND hWnd, LONG_PTR cx, LONG_PTR cy, bool select = false, int nLogical = POS_LOGICAL_W)
	{
		static int lastX = 0;
		static POINT_PTR lastPos = {0,0};
		POINT_PTR pos;
		Editor_GetCaretPos(hWnd, nLogical, &pos);
		if(cx || pos.x != lastPos.x || pos.y != lastPos.y)
		{
			lastX = max(0, pos.x+cx);
		}
		UINT_PTR lines = Editor_GetLines(hWnd, nLogical);
		if(pos.y+cy >= lines)
			cy = 0;
		if(cx || cy)
		{
			//pos.x+=lastX;
			pos.x = lastX;
			pos.y+=cy;
			//UINT_PTR len = GetLineLength(hWnd, pos.y);
			Editor_SetCaretPosEx(hWnd, nLogical, &pos, select);
			Editor_GetCaretPos(hWnd, nLogical, &pos);
			if(select)
				Editor_Redraw(hWnd, TRUE);
			lastPos = pos;
		}
		return pos;
	}


	BOOL IsEndOfLine(HWND hWnd)
	{
		POINT_PTR pos = MoveCaret(hWnd, 0, 0);
		return pos.x == GetLineLength(hWnd, pos.y) - 1;
	}

	BOOL IsEndOfLine(HWND hWnd, POINT_PTR &pos)
	{
		return pos.x >= GetLineLength(hWnd, pos.y) - 1;
	}

	UINT_PTR GetLineText(HWND hWnd, LONG_PTR y, std::wstring &buf)
	{
		GET_LINE_INFO info;
		info.cch = 0;
		info.flags = FLAG_LOGICAL|FLAG_WITH_CRLF;
		info.yLine = y;
		UINT_PTR lineLen = Editor_GetLineW(hWnd, &info, NULL);
		if(lineLen > 0)
		{
			buf.resize(lineLen);
			info.cch = lineLen;
			UINT_PTR recv = Editor_GetLineW(hWnd, &info, &buf[0]); 
		}
		return lineLen;
	}
	wchar_t GetCaretText(HWND hWnd, POINT_PTR &pos)
	{
		std::wstring buf;
		GetLineText(hWnd, pos.y, buf);
		if(buf.size() > pos.x)
			return buf[pos.x];
		return 0;
	}
	LONG GetRegionText(HWND hWnd, const POINT_PTR &from, const POINT_PTR &to, std::wstring &buf)
	{
		LONG len = 0;
		std::vector<wchar_t> lineBuf;
		for(LONG_PTR i = from.y; i <= to.y; ++i)
		{
			GET_LINE_INFO info;
			info.cch = 0;
			info.flags = FLAG_LOGICAL|FLAG_WITH_CRLF;
			info.yLine = i;
			UINT_PTR lineLen = Editor_GetLineW(hWnd, &info, NULL);
			if(lineLen > 0)
			{
				lineBuf.resize(lineLen);
				info.cch = lineLen;
				UINT_PTR recv = Editor_GetLineW(hWnd, &info, &lineBuf[0]); 
				UINT_PTR xFrom = (i == from.y) ? from.x : 0;
				UINT_PTR xTo = (i == to.y) ? to.x : lineLen - 1;
				while(xFrom < xTo)
				{
					buf.push_back(lineBuf[xFrom++]);
					++len;
				}
			}
		}
		return len;
	}

	UINT_PTR GetSelectionText(HWND hWnd, std::wstring &buf)
	{
		UINT_PTR len = Editor_GetSelTextW(hWnd, 0, NULL);
		buf.resize(len);
		Editor_GetSelTextW(hWnd, len, &buf[0]);
		buf.erase(buf.end()-1);
		return len-1;
	}
	SIZE_PTR GetSelectionSize(HWND hWnd)
	{
		POINT_PTR startSel, endSel;
		Editor_GetSelStart(hWnd, POS_LOGICAL_W, &startSel);
		Editor_GetSelEnd(hWnd, POS_LOGICAL_W, &endSel);
		SIZE_PTR sz;
		sz.cx = endSel.x - startSel.x;
		sz.cy = endSel.y - startSel.y;
		return sz;
	}
	BOOL HasSelection(HWND hWnd)
	{
		SIZE_PTR sz = GetSelectionSize(hWnd);
		return sz.cx || sz.cy;
	}
	BOOL MoveNextWord(HWND hWnd, bool hasSelection, bool startOfWord, bool whiteSpaceOnly, bool backward)
	{
		struct CharGroupCmp
		{
			int GetCharGroup(wchar_t c, bool whiteSpaceOnly)
			{
				int group = -1;
				if(!whiteSpaceOnly)
				{
					if((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') || (c >= '0' && c <= '9') || (c == '_'))
						group = 1;
					else if(c >= ' ' && c <= '~')
						group = 2;
				}
				if(c == ' ' || c == '\t' || c == '\r' || c == '\n')
					group = 0;
				return group;
			}
			bool IsSameCharGroup(wchar_t a, wchar_t b, bool bStrict, bool whiteSpaceOnly)
			{
				int gra = GetCharGroup(a, whiteSpaceOnly);
				if(!bStrict && !gra)
					return true;
				int grb = GetCharGroup(b, whiteSpaceOnly);
				if(bStrict && !grb)
					return true;
				return gra == grb;
			}
		} charGroupCmp;
		std::wstring buf;
		if(backward && startOfWord)
			MoveCaret(hWnd, -1, 0, hasSelection);
			//Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_LEFT : EEID_LEFT);
		else if(!backward && !startOfWord)
			MoveCaret(hWnd, 1, 0, hasSelection);
			//Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_RIGHT : EEID_RIGHT);
		POINT_PTR cur = MoveCaret(hWnd, 0, 0);
		while(true)
		{
			POINT_PTR next = cur;
			if(backward)
			{
				if(--next.x < 0)
				{
					next.y--;
					next.x = GetLineLength(hWnd, next.y)-1;
				}
			}
			else
			{
				if(IsEndOfLine(hWnd, next))
				{
					next.x = 0;
					next.y++;
				}
				else
					next.x++;
			}
			wchar_t a = GetCaretText(hWnd, cur);
			wchar_t b = GetCaretText(hWnd, next);
			if(!b)
				return false;
			if(backward ^ startOfWord)
			{
				if(!charGroupCmp.IsSameCharGroup(b, a, hasSelection, whiteSpaceOnly))
				{
					SetCaret(hWnd, next.x, next.y, hasSelection);
					break;
				}
			}
			else
			{
				if(!charGroupCmp.IsSameCharGroup(a, b, FALSE, whiteSpaceOnly))
				{
					SetCaret(hWnd, cur.x, cur.y, hasSelection);
					break;
				}
			}
			cur = next;
		}

		if(!backward && !startOfWord && hasSelection)
			Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_RIGHT : EEID_RIGHT);

		if(backward && hasSelection)
			Editor_Redraw(hWnd, TRUE);
		
		return true;
	}

	inline void InsertString( HWND hwnd, LPCWSTR szString)
	{
		BOOL bChecked = FALSE;
		Editor_QueryStatus(hwnd, EEID_INSERT, &bChecked);
		if(bChecked)
			Editor_ExecCommand(hwnd, EEID_INSERT);

		Editor_InsertStringW(hwnd, szString, true);

		if(bChecked)
			Editor_ExecCommand(hwnd, EEID_INSERT);
	}
};

//--------------------------------------------------------------------------------------
// Constructor
//--------------------------------------------------------------------------------------
EmulationVI::EmulationVI(HWND hWnd, IContext *context) : 
hWnd(hWnd)
,curMode(Invalid)
,context(context)
{
	InitWin32KeyTable();
	ChangeMode(Command);
	UpdateStatus();
	ZeroMemory(&beginInputPos, sizeof(beginInputPos));
	ZeroMemory(&endInputPos, sizeof(endInputPos));
}

//--------------------------------------------------------------------------------------
// Destructor
//--------------------------------------------------------------------------------------
EmulationVI::~EmulationVI()
{
}

bool EmulationVI::OnKeyDown(int keyCode, bool shift, bool ctrl)
{
	wchar_t c = Win32KeyTable[keyCode][0];
	if(c == VK_TO_ASCII(keyCode))
	{
		if(shift)
			c |= SHIFT_VK_TO_ASCII(keyCode);
		if(ctrl)
			c |= CTRL_VK_TO_ASCII(keyCode);
	}
	else
	{
		if(shift && ctrl)
			return false;
		c = Win32KeyTable[keyCode][ctrl ? 2 : (shift ? 1 : 0)];
	}
	if(c)
	{
		bool processed = OnChar(c);
		UpdateStatus();
		return processed;
	}
	return false;
}

void EmulationVI::OnCaretMoved(BOOL reset)
{
	if(IsInputMode())
	{
		if(!reset)
			CloseInputText();
		Editor_GetSelStart(hWnd, POS_LOGICAL_W, &beginInputPos);
	}
}

namespace
{
	int GetDefaultCommandRepeat(wchar_t c)
	{
		switch(c)
		{
		case '|': return 0;
		}
		return 1;
	}

	bool WantMotionCommand(wchar_t first)
	{
		switch(first)
		{
		case 0://before first..
		case 'd':
		case 'c':
		case 'y':
		case '<':
		case '>':
			return true;
		}
		return false;
	}
};

bool EmulationVI::OnChar(wchar_t c)
{
	Mode curMode = GetCurMode();
	SetCursorState(curMode);
	if(c == 27)//ESCAPE:
	{
		ChangeMode(Invalid);
		ChangeMode(Command);
		return false;
	}
	switch(curMode)
	{
	case Visual:
	case Command:
		{
			Cmd cmd;
			curCommands.push_back(c);
			ParseCommands(curCommands, cmd);
			if(cmd.first && !cmd.repeat)
				cmd.repeat  = GetDefaultCommandRepeat(cmd.first);
			CommandType type = ProcessCommandMode(cmd);
			if(type == CommandType_Unknown)
			{
				if(HAS_CTRL_VK(c))
					return false;
			}
			else if(type == CommandType_Motion)
			{
				curCommands.clear();
			}
			/*
			else if(type != CommandType_Motion && curMode == Visual && GetCurMode() == Visual)
				ChangeMode(Command);
				*/
		}
		return true;
	case Insert:
	case Overwrite:
		return ProcessInputMode(c);
	}
	return false;
}
EmulationVI::Mode EmulationVI::GetCurMode() const
{
	return curMode;
}

void EmulationVI::ChangeMode(Mode mode)
{
	if(curMode != mode)
	{
		OnLeaveMode(curMode);
		OnEnterMode(mode);
		curMode = mode;
	}
}
void EmulationVI::SetCursorState(Mode mode)
{
	{
		BOOL bChecked = FALSE;
		Editor_QueryStatus(hWnd, EEID_INSERT, &bChecked);
		if((mode != Insert && mode != Visual) ^ bChecked)
			Editor_ExecCommand(hWnd, EEID_INSERT);
	}
}
void EmulationVI::OnEnterMode(Mode mode)
{
	switch(mode)
	{
	case Command:
		curCommands.clear();
		Editor_ExecCommand(hWnd, EEID_ESCAPE);
		break;
	case Insert:
	case Overwrite:
		Editor_GetSelStart(hWnd, POS_LOGICAL_W, &beginInputPos);
		break;
	}
	SetCursorState(mode);
}
void EmulationVI::OnLeaveMode(Mode mode)
{
	switch(mode)
	{
	case Overwrite:
	case Insert:
		CloseInputText();
		Editor_GetSelStart(hWnd, POS_LOGICAL_W, &endInputPos);
		break;
	case Command:
		curCommands.clear();
		break;
	}
}
void EmulationVI::CloseInputText()
{
	POINT_PTR pos;
	Editor_GetSelStart(hWnd, POS_LOGICAL_W, &pos);
	GetRegionText(hWnd, beginInputPos, pos, recCmds.back().input);
	Editor_GetSelStart(hWnd, POS_LOGICAL_W, &beginInputPos);
}

bool EmulationVI::ProcessInputMode(wchar_t c)
{
	std::vector<LONG> editorCmd;
	switch(c)
	{
	case VK_TO_ASCII(VK_DELETE): editorCmd.push_back(EEID_DELETE); break;
	case SHIFT_VK_TO_ASCII(VK_DELETE): editorCmd.push_back(EEID_DELETE); break;
	case CTRL_VK_TO_ASCII(VK_DELETE): editorCmd.push_back(EEID_DELETE_RIGHT_WORD); break;
	case SHIFT_CTRL_VK_TO_ASCII(VK_DELETE): editorCmd.push_back(EEID_DELETE_LEFT_WORD); break;
	case VK_TO_ASCII(VK_BACK): editorCmd.push_back(EEID_BACK); break;
	case SHIFT_VK_TO_ASCII(VK_BACK): editorCmd.push_back(EEID_BACK); break;
	case CTRL_VK_TO_ASCII(VK_BACK): case 23:/*^W*/editorCmd.push_back(EEID_DELETE_LEFT_WORD); break;
	case SHIFT_CTRL_VK_TO_ASCII(VK_BACK): editorCmd.push_back(EEID_DELETE_RIGHT_WORD); break;
	case VK_TO_ASCII(VK_TAB): editorCmd.push_back(EEID_TAB); break;
	case SHIFT_VK_TO_ASCII(VK_TAB): editorCmd.push_back(EEID_SHIFT_TAB); break;
	case VK_TO_ASCII(VK_END): editorCmd.push_back(EEID_END); break;
	case SHIFT_VK_TO_ASCII(VK_END): editorCmd.push_back(EEID_SHIFT_END); break;
	case VK_TO_ASCII(VK_HOME): editorCmd.push_back(EEID_HOME); break;
	case SHIFT_VK_TO_ASCII(VK_HOME): editorCmd.push_back(EEID_SHIFT_HOME); break;
	case VK_TO_ASCII(VK_LEFT): editorCmd.push_back(EEID_LEFT); break;
	case SHIFT_VK_TO_ASCII(VK_LEFT): editorCmd.push_back(EEID_SHIFT_LEFT); break;
	case CTRL_VK_TO_ASCII(VK_LEFT): editorCmd.push_back(EEID_LEFT_WORD); break;
	case SHIFT_CTRL_VK_TO_ASCII(VK_LEFT): editorCmd.push_back(EEID_SHIFT_LEFT_WORD); break;
	case VK_TO_ASCII(VK_UP): editorCmd.push_back(EEID_UP); break;
	case SHIFT_VK_TO_ASCII(VK_UP): editorCmd.push_back(EEID_SHIFT_UP); break;
	case VK_TO_ASCII(VK_RIGHT): editorCmd.push_back(EEID_RIGHT); break;
	case SHIFT_VK_TO_ASCII(VK_RIGHT): editorCmd.push_back(EEID_SHIFT_RIGHT); break;
	case CTRL_VK_TO_ASCII(VK_RIGHT): editorCmd.push_back(EEID_RIGHT_WORD); break;
	case SHIFT_CTRL_VK_TO_ASCII(VK_RIGHT): editorCmd.push_back(EEID_SHIFT_RIGHT_WORD); break;
	case VK_TO_ASCII(VK_DOWN): editorCmd.push_back(EEID_DOWN); break;
	case SHIFT_VK_TO_ASCII(VK_DOWN): editorCmd.push_back(EEID_SHIFT_DOWN); break;
	case 0x01:/*^A*/editorCmd.push_back(EEID_EDIT_SELECT_ALL); break;
	case VK_TO_ASCII(VK_INSERT): ChangeMode(Command); return true;
	case 0x04:/*^D*/
	case 0x14:/*^T*/
		{
			CloseInputText();
			BOOL backward = c == 0x04;
			POINT_PTR cur = MoveCaret(hWnd, 0, 0);
			if(backward)
			{
				std::wstring buf;
				GetLineText(hWnd, cur.y, buf);
				if(buf[0] != '\t')
					return true;
				cur.x--;
			}
			SetCaret(hWnd, 0, cur.y);
			if(backward)
				editorCmd.push_back(EEID_DELETE);
			else
				InsertString(hWnd, L"\t");
			editorCmd.insert(editorCmd.end(), cur.x, EEID_RIGHT);
		}
		break;
	case 0x15:/*^U*/
		{
			POINT_PTR cur;
			Editor_GetCaretPos(hWnd, POS_LOGICAL_W, &cur);
			if(cur.x != beginInputPos.x || cur.y != beginInputPos.y)
			{
				Editor_SetCaretPosEx(hWnd, POS_LOGICAL_W, &beginInputPos, true);
				Editor_ExecCommand(hWnd, EEID_DELETE);
			}
		}
		return true;
	}
	if(!editorCmd.empty())
	{
		CloseInputText();
		recCmds.back().input.push_back(c);
		for(size_t i = 0; i < editorCmd.size(); ++i)
		{
			switch(editorCmd[i])
			{
			case EEID_RIGHT: MoveCaret(hWnd, 1, 0, FALSE); break;
			case EEID_SHIFT_RIGHT: MoveCaret(hWnd, 1, 0, TRUE); break;
			case EEID_LEFT: MoveCaret(hWnd, -1, 0, FALSE); break;
			case EEID_SHIFT_LEFT: MoveCaret(hWnd, -1, 0, TRUE); break;
			case EEID_DOWN: MoveCaret(hWnd, 0, 1, FALSE, POS_VIEW); break;
			case EEID_SHIFT_DOWN: MoveCaret(hWnd, 0, 1, TRUE, POS_VIEW); break;
			case EEID_UP: MoveCaret(hWnd, 0, -1, FALSE, POS_VIEW); break;
			case EEID_SHIFT_UP: MoveCaret(hWnd, 0, -1, TRUE, POS_VIEW); break;
			default: Editor_ExecCommand(hWnd, editorCmd[i]);break;
			}
		}
		Editor_GetSelStart(hWnd, POS_LOGICAL_W, &beginInputPos);
		return true;
	}
	return false;
}

void EmulationVI::ParseCommands(const std::vector<wchar_t> &commands, Cmd &cmd)
{
	cmd.repeat = 0;
	cmd.first = 0;
	cmd.second = 0;

	std::wstring buf;
	for(size_t i = 0; i < commands.size(); ++i)
	{
		bool isNum = 
			(!buf.empty() && commands[i] >= '0' && commands[i] <= '9')
			|| (commands[i] >= '1' && commands[i] <= '9');
		if(isNum && WantMotionCommand(cmd.first))
			buf.push_back(commands[i]);
		else
		{
			if(!cmd.first)
				cmd.first = commands[i];
			else if(!cmd.second)
				cmd.second = commands[i];
			if(!buf.empty())
			{
				int num = _wtoi(buf.c_str());
				if(!cmd.repeat)
					cmd.repeat = num;
				else
					cmd.repeat *= num;
				buf.clear();
			}
		}
	}

}
EmulationVI::CommandType EmulationVI::ProcessCommandMode(Cmd &cmd)
{
	SIZE_PTR selectionSize = GetSelectionSize(hWnd);
	bool visualMode = GetCurMode() == Visual || selectionSize.cx || selectionSize.cy;
	CommandType ret = CommandType_Unknown;
	if(cmd.first && (ret = ProcessCommandMotion(cmd.first, cmd.repeat, GetCurMode() == Visual)))
	{
		curCommands.clear();
		return ret;
	}
	else if(cmd.first && (ret = ProcessSingleCommand(cmd.first, cmd.repeat, visualMode)))
	{
	}
	else if(cmd.first && cmd.second)
	{
		ret = ProcessDoubleCommand(cmd.first, cmd.second, cmd.repeat, visualMode);
		curCommands.clear();
	}

	if(ret)
	{
		curCommands.clear();
		if(ret == CommandType_Record)
		{
			recCmds.clear();
			recCmds.push_back(RecCmd(cmd, selectionSize));
		}
		if(/*ret > CommandType_Volatile && */visualMode && GetCurMode() == Visual)
		{
			POINT_PTR pos;
			Editor_GetSelStart(hWnd, POS_LOGICAL_W, &pos);
			SetCaret(hWnd, pos.x, pos.y);
			ChangeMode(Invalid);
			ChangeMode(Command);
		}
	}

	return ret;
}
EmulationVI::CommandType EmulationVI::ProcessCommandMotion(wchar_t c, int repeat, bool hasSelection)
{

	switch(c)
	{
	case 'h':
	case VK_TO_ASCII(VK_LEFT):
	case SHIFT_VK_TO_ASCII(VK_LEFT):
		hasSelection |= HAS_SHIFT_VK(c);
		while(repeat--)
			MoveCaret(hWnd, -1, 0, hasSelection);
		return CommandType_Motion;
	case 'j':
	case VK_TO_ASCII(VK_DOWN):
	case SHIFT_VK_TO_ASCII(VK_DOWN):
	case CTRL_VK_TO_ASCII(VK_DOWN):
		hasSelection |= HAS_SHIFT_VK(c);
		while(repeat--)
			MoveCaret(hWnd, 0, 1, hasSelection, POS_VIEW);
			//Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_DOWN : EEID_DOWN);
		return CommandType_Motion;
	case 'k':
	case VK_TO_ASCII(VK_UP):
	case SHIFT_VK_TO_ASCII(VK_UP):
	case CTRL_VK_TO_ASCII(VK_UP):
		hasSelection |= HAS_SHIFT_VK(c);
		while(repeat--)
			MoveCaret(hWnd, 0, -1, hasSelection, POS_VIEW);
			//Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_UP : EEID_UP);
		return CommandType_Motion;
	case 'l':
	case VK_TO_ASCII(VK_RIGHT):
	case SHIFT_VK_TO_ASCII(VK_RIGHT):
		hasSelection |= HAS_SHIFT_VK(c);
		while(repeat--)
			if(!IsEndOfLine(hWnd))
				MoveCaret(hWnd, 1, 0, hasSelection);
		return CommandType_Motion;
	case ' ':
		hasSelection |= HAS_SHIFT_VK(c);
		while(repeat--)
			MoveCaret(hWnd, 1, 0, hasSelection, POS_VIEW);
			//Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_RIGHT : EEID_RIGHT);
		return CommandType_Motion;
	case 'W':
	case 'w':
		while(repeat--)
			MoveNextWord(hWnd, hasSelection, true, c == 'W', false);
		return CommandType_Motion;
	case 'B':
	case 'b':
		while(repeat--)
			MoveNextWord(hWnd, hasSelection, true, c == 'B', true);
		return CommandType_Motion;
	case 'E':
	case 'e':
		while(repeat--)
			MoveNextWord(hWnd, hasSelection, false, c == 'E', false);
		return CommandType_Motion;
	case 'G':
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_BOTTOM : EEID_BOTTOM);
		return CommandType_Motion;
	case '0':
	case VK_TO_ASCII(VK_HOME):
	case SHIFT_VK_TO_ASCII(VK_HOME):
		hasSelection |= HAS_SHIFT_VK(c);
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_HOME : EEID_HOME);
		return CommandType_Motion;
	case '^':
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_HOME : EEID_HOME);
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_HOME_TEXT : EEID_HOME_TEXT);
		return CommandType_Motion;
	case '$':
	case VK_TO_ASCII(VK_END):
	case SHIFT_VK_TO_ASCII(VK_END):
		hasSelection |= HAS_SHIFT_VK(c);
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_END : EEID_END);
		return CommandType_Motion;
	case 0x04://^d
	case 0x15:/*^U*/
		{
			bool backward = c == 0x15;
			POINT_PTR pos;
			Editor_GetScrollPos(hWnd, &pos);
			SIZE_PTR size;
			Editor_GetPageSize(hWnd, &size);
			LONG_PTR diff = backward ? -size.cy/2 : size.cy/2; 
			if(pos.y + diff < 0)
				diff = -pos.y;
			pos.y += diff;
			MoveCaret(hWnd, 0, diff, hasSelection);
			Editor_SetScrollPos(hWnd, &pos);
		}
		return CommandType_Motion;
	case 6://^f
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_PAGEDOWN : EEID_PAGEDOWN);
		return CommandType_Motion;
	case 1://^a
		Editor_ExecCommand(hWnd, EEID_EDIT_SELECT_ALL);
		return CommandType_Motion;
	case 2://^b
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_PAGEUP : EEID_PAGEUP);
		return CommandType_Motion;
	case VK_TO_ASCII(VK_BACK):
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_LEFT : EEID_LEFT);
		return CommandType_Motion;
	case VK_TO_ASCII(VK_DELETE):
	case SHIFT_VK_TO_ASCII(VK_DELETE):
		hasSelection |= HAS_SHIFT_VK(c);
		Editor_ExecCommand(hWnd, EEID_DELETE);
		return CommandType_Motion;
	case 0x0D://ENTER
	case '+':
	case '-':
	case '_':
		{
			BOOL backward = c == '-';
			if(c == '_')
				repeat--;
			while(repeat--)
				if(!backward)
					Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_DOWN : EEID_DOWN);
				else
					Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_UP : EEID_UP);
			Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_HOME : EEID_HOME);
			Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_HOME_TEXT : EEID_HOME_TEXT);
		}
		return CommandType_Motion;
	case '|':
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_HOME : EEID_HOME);
		{
			POINT_PTR pos;
			Editor_GetCaretPos(hWnd, POS_VIEW, &pos);
			pos.x = repeat;
			Editor_SetCaretPosEx(hWnd, POS_VIEW, &pos, hasSelection);
		}
		return CommandType_Motion;
	case 'n':
	case VK_TO_ASCII(VK_F3):
	case 'N':
	case SHIFT_VK_TO_ASCII(VK_F3):
		{
			POINT_PTR cur, next;
			bool backward = (c == 'N' || c == SHIFT_VK_TO_ASCII(VK_F3));
			backward ^= !setting.findForward;
			Editor_GetSelStart(hWnd, POS_LOGICAL_W, &cur);
			Editor_ExecCommand(hWnd, EEID_ESCAPE);
			if(backward)
				Editor_ExecCommand(hWnd, EEID_LEFT);
			Editor_ExecCommand(hWnd, backward ? EEID_EDIT_REPEAT_BACK : EEID_EDIT_REPEAT);
			Editor_GetSelEnd(hWnd, POS_LOGICAL_W, &next);
			//Editor_GetSelStart(hWnd, POS_LOGICAL_W, &next);
			if(hasSelection)
			{
				Editor_SetCaretPos(hWnd, POS_LOGICAL_W, &cur);
				Editor_SetCaretPosEx(hWnd, POS_LOGICAL_W, &next, hasSelection);
			}	
		}
		return CommandType_Motion;
	case '*':
	case '#':
		{
			POINT_PTR cur, next;
			//Editor_GetSelStart(hWnd, POS_LOGICAL_W, &cur);
			//Editor_ExecCommand(hWnd, EEID_ESCAPE);
			Editor_ExecCommand(hWnd, c == '*' ? EEID_FIND_NEXT_WORD : EEID_FIND_PREV_WORD);
			if(hasSelection)
			{
				Editor_GetSelEnd(hWnd, POS_LOGICAL_W, &next);
				Editor_SetCaretPos(hWnd, POS_LOGICAL_W, &cur);
				Editor_SetCaretPosEx(hWnd, POS_LOGICAL_W, &next, true);
			}
		}
		return CommandType_Motion;
	case 'H':
	case 'M':
	case 'L':
		{
			POINT_PTR pos;
			Editor_GetScrollPos(hWnd, &pos);
			SIZE_PTR size;
			Editor_GetPageSize(hWnd, &size);
			if(c == 'M')
				pos.y += size.cy/2;
			else if(c == 'L')
				pos.y += size.cy-1;
			SetCaret(hWnd, 0, pos.y, hasSelection);
			Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_HOME_TEXT : EEID_HOME_TEXT);
			Editor_Redraw(hWnd, TRUE);
		}
		return CommandType_Motion;
	case '%':
		Editor_ExecCommand(hWnd, hasSelection ? EEID_SHIFT_NEXT_PAREN : EEID_NEXT_PAREN);
		return CommandType_Motion;
	case 5://^E
		Editor_ExecCommand(hWnd, EEID_SCROLL_DOWN);
		return CommandType_Motion;
	case 25://^Y
		Editor_ExecCommand(hWnd, EEID_SCROLL_UP);
		return CommandType_Motion;
	case 26://^Z
		Editor_ExecCommand(hWnd, EEID_EDIT_UNDO);
		return CommandType_Motion;
	case ';':
	case ',':
		{
			bool backward = c == ',';
			if(lastFindCmd.first)
			{
				Cmd cmd = lastFindCmd;
				wchar_t first = cmd.first;
				if(backward)
				{
					if(first >= 'a' && first <= 'z')
						first += 'A' - 'a';
					else
						first += 'a' - 'A';
				}
				EmulationVI::CommandType ret = ProcessDoubleCommand(first, cmd.second, cmd.repeat, hasSelection);
				lastFindCmd = cmd;//restore
				return ret;
			}
		}
		break;
	case CTRL_VK_TO_ASCII(VK_TAB):
		Editor_ExecCommand(hWnd, EEID_NEXT_WINDOW);
		return CommandType_Motion;
	case SHIFT_CTRL_VK_TO_ASCII(VK_TAB):
		Editor_ExecCommand(hWnd, EEID_PREV_WINDOW);
		return CommandType_Motion;
	}
	return CommandType_Unknown;
}

EmulationVI::CommandType EmulationVI::ProcessSingleCommand(wchar_t c, int repeat, bool hasSelection)
{
	switch(c)
	{
	case '.':
		while(repeat--)
		{
			std::vector<RecCmd> cmds = recCmds;//save rec
			for(size_t i = 0; i < cmds.size(); ++i)
			{
				RecCmd &cmd = cmds[i];
				switch(GetCurMode())
				{
				case Command:
					if(cmd.selection.cx || cmd.selection.cy)
						curMode = Visual;
					MoveCaret(hWnd, cmd.selection.cx, cmd.selection.cy, true);
					ProcessCommandMode(cmd.cmd);
					switch(GetCurMode())
					{
					case Overwrite:
					case Insert:
						if(!cmd.input.empty())
						{
							wchar_t *ptr = &cmd.input[0];
							while(*ptr)
							{
								if(Overwrite == GetCurMode() && !IsEndOfLine(hWnd))
								{
									MoveCaret(hWnd, 1, 0);
									Editor_ExecCommand(hWnd, EEID_BACK);
								}
								if(!ProcessInputMode(*ptr))
								{
									if(*ptr == '\r')
									{
									}
									else
									{
										wchar_t buf[2] = {*ptr, 0};
										InsertString(hWnd, buf);
									}
								}
								++ptr;
							}
						}
						break;
					case Visual:
						//Input Movement̴
						break;
					}
				}
			}
			ChangeMode(Command);
			recCmds = cmds;//restore rec
		}
		return CommandType_Volatile;
	case 'u':
		while(repeat--)
			Editor_ExecCommand(hWnd, EEID_EDIT_UNDO);
		return CommandType_Volatile;
	case 18://^r
		while(repeat--)
			Editor_ExecCommand(hWnd, EEID_EDIT_REDO);
		return CommandType_Volatile;
	case 'v':
		while(--repeat)
			Editor_ExecCommand(hWnd, EEID_SHIFT_RIGHT);
		ChangeMode(Visual);
		return CommandType_Volatile;
	case 'V':
		ChangeMode(Visual);
		Editor_ExecCommand(hWnd, EEID_HOME);
		while(repeat--)
			Editor_ExecCommand(hWnd, EEID_SHIFT_DOWN);
		return CommandType_Volatile;
	case 'i':
	case VK_TO_ASCII(VK_INSERT):
		ChangeMode(Insert);
		return CommandType_Record;
	case 'I':
		Editor_ExecCommand(hWnd, EEID_HOME);
		Editor_ExecCommand(hWnd, EEID_HOME_TEXT);
		ChangeMode(Insert);
		return CommandType_Record;
	case 'a':
		MoveCaret(hWnd, 1, 0);
		ChangeMode(Insert);
		return CommandType_Record;
	case 'A':
		Editor_ExecCommand(hWnd, EEID_LOGICAL_END);
		ChangeMode(Insert);
		return CommandType_Record;
	case 'x':
		if(!hasSelection)
			while(repeat--)
				if(!IsEndOfLine(hWnd))
					MoveCaret(hWnd, 1, 0, true);
		if(HasSelection(hWnd))
			Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
		return CommandType_Record;
	case 'X':
		while(repeat--)
		{
			POINT_PTR pos = MoveCaret(hWnd, 0, 0);
			if(pos.x != 0)
				Editor_ExecCommand(hWnd, EEID_SHIFT_LEFT);
		}
		if(HasSelection(hWnd))
			Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
		return CommandType_Record;
	case 's':
		if(!hasSelection)
			while(repeat--)
				if(!IsEndOfLine(hWnd))
					MoveCaret(hWnd, 1, 0, true);
		if(HasSelection(hWnd))
			Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
		ChangeMode(Insert);
		return CommandType_Record;
	case 'S':
		Editor_ExecCommand(hWnd, EEID_HOME);
		Editor_ExecCommand(hWnd, EEID_SHIFT_END);
		while(--repeat)
			Editor_ExecCommand(hWnd, EEID_SHIFT_DOWN);
		Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
		ChangeMode(Insert);
		return CommandType_Record;
	case 'd':
		if(hasSelection)
		{
			if(HasSelection(hWnd))
				Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
			return CommandType_Record;
		}
		return CommandType_Unknown;
	case 'D':
		while(--repeat)
			Editor_ExecCommand(hWnd, EEID_SHIFT_DOWN);
		Editor_ExecCommand(hWnd, EEID_SHIFT_END);
		Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
		return CommandType_Record;
	case 'c':
		if(hasSelection)
		{
			Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
			ChangeMode(Insert);
			return CommandType_Record;
		}
		return CommandType_Unknown;
	case 'C':
		while(--repeat)
			Editor_ExecCommand(hWnd, EEID_SHIFT_DOWN);
		Editor_ExecCommand(hWnd, EEID_SHIFT_END);
		Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
		ChangeMode(Insert);
		return CommandType_Record;
	case 'J':
		Editor_ExecCommand(hWnd, EEID_SELECT_LINE);
		while(--repeat)
			Editor_ExecCommand(hWnd, EEID_SHIFT_DOWN);
		Editor_ExecCommand(hWnd, EEID_JOIN_LINES);
		Editor_ExecCommand(hWnd, EEID_ESCAPE);
		return CommandType_Record;
	case 'o':
		Editor_ExecCommand(hWnd, EEID_LINE_OPEN_BELOW);
		ChangeMode(Insert);
		return CommandType_Record;
	case 'O':
		Editor_ExecCommand(hWnd, EEID_LINE_OPEN_ABOVE);
		ChangeMode(Insert);
		return CommandType_Record;
	case 'p':
		if(hasSelection)
			Editor_ExecCommand(hWnd, EEID_DELETE);
		else
		{
			if(!IsEndOfLine(hWnd))
				MoveCaret(hWnd, 1, 0); 
		}

		{
			std::wstring buf;
			if(GetClipboardText(0, buf))
			{
				int crCnt = CountLastCR(buf);
				if(crCnt > 0) //Insert Line
				{
					POINT_PTR pt = MoveCaret(hWnd, 0, 0); 
					while(repeat--)
					{
						SetCaret(hWnd, 0, pt.y + 1);
						InsertString(hWnd, buf.c_str());
						SetCaret(hWnd, 0, pt.y + 1);
					}
					Editor_ExecCommand(hWnd, EEID_HOME_TEXT);
				}
				else
				{
					while(repeat--)
						InsertString(hWnd, buf.c_str());
				}
			}
		}
		return CommandType_Record;
	case 'P':
		if(hasSelection)
			Editor_ExecCommand(hWnd, EEID_DELETE);
		{
			std::wstring buf;
			if(GetClipboardText(0, buf))
			{
				int crCnt = CountLastCR(buf);
				if(crCnt > 0) //Insert Line
				{
					POINT_PTR pt = MoveCaret(hWnd, 0, 0); 
					while(repeat--)
					{
						SetCaret(hWnd, 0, pt.y);
						InsertString(hWnd, buf.c_str());
						SetCaret(hWnd, 0, pt.y);
					}
					Editor_ExecCommand(hWnd, EEID_HOME_TEXT);
				}
				else
				{
					while(repeat--)
						InsertString(hWnd, buf.c_str());
				}
			}
		}
		return CommandType_Record;
	case 'y':
		if(hasSelection)
		{
			Editor_ExecCommand(hWnd, EEID_EDIT_COPY);
			return CommandType_Volatile;
		}
		return CommandType_Unknown;
	case 'Y':
		{
			POINT_PTR pt = MoveCaret(hWnd, 0, 0); 
			Editor_ExecCommand(hWnd, EEID_SELECT_LINE);
			while(--repeat)
				Editor_ExecCommand(hWnd, EEID_SHIFT_DOWN);
			Editor_ExecCommand(hWnd, EEID_EDIT_COPY);
			Editor_ExecCommand(hWnd, EEID_ESCAPE);
			SetCaret(hWnd, pt.x, pt.y);
		}
		return CommandType_Volatile;
	case 'R':
		ChangeMode(Overwrite);
		return CommandType_Record;
	case '>':
		if(hasSelection)
		{
			while(repeat--)
				Editor_ExecCommand(hWnd, EEID_TAB);
			return CommandType_Record;
		}
		return CommandType_Unknown;
	case '<':
		if(hasSelection)
		{
			while(repeat--)
				Editor_ExecCommand(hWnd, EEID_SHIFT_TAB);
			return CommandType_Record;
		}
		return CommandType_Unknown;
	case '~':
		if(!hasSelection)
			while(repeat--)
				if(!IsEndOfLine(hWnd))
					MoveCaret(hWnd, 1, 0, true);

		{
			std::wstring buf;
			UINT_PTR len = GetSelectionText(hWnd, buf);
			if(len)
			{
				Editor_ExecCommand(hWnd, EEID_DELETE);
				for(size_t i = 0; i < buf.length(); ++i)
				{
					if(buf[i] >= 'a' && buf[i] <= 'z')
						buf[i] += 'A' - 'a';
					else if(buf[i] >= 'A' && buf[i] <= 'Z')
						buf[i] += 'a' - 'A';
				}
				InsertString(hWnd, buf.c_str());
			}
		}
		return CommandType_Record;
	case 'm':
		Editor_ExecCommand(hWnd, EEID_BOOKMARK_TOGGLE); 
		return CommandType_Volatile;
	case '\'':
		Editor_ExecCommand(hWnd, EEID_BOOKMARK_NEXT); 
		return CommandType_Volatile;
	case '\"':
		Editor_ExecCommand(hWnd, EEID_BOOKMARK_PREV); 
		return CommandType_Volatile;
	case 3://^C
		Editor_ExecCommand(hWnd, EEID_EDIT_COPY);
		return CommandType_Volatile;
	case 22://^V
		Editor_ExecCommand(hWnd, EEID_EDIT_PASTE);
		return CommandType_Record;
	case 24://^X
		Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
		return CommandType_Volatile;
	}
	return CommandType_Unknown;
}
EmulationVI::CommandType EmulationVI::ProcessDoubleCommand(wchar_t first, wchar_t second, int repeat, bool hasSelection)
{
	switch(first)
	{
	case 'd':
	case 'c':
	case 'y':
	case '<':
	case '>':
		{
			POINT_PTR pt = MoveCaret(hWnd, 0, 0); 
			POINT_PTR scrollPos;
			Editor_GetScrollPos(hWnd, &scrollPos);
			if(first == second)
			{
				Editor_ExecCommand(hWnd, EEID_SELECT_LINE);
				while(--repeat)
					Editor_ExecCommand(hWnd, EEID_SHIFT_DOWN);
			}
			else
			{
				switch(second)
				{
				case 'j':
				case 'k':
				case 0x0D:
					Editor_ExecCommand(hWnd, EEID_SELECT_LINE);
					break;
				}

				if(!ProcessCommandMotion(second, repeat, true))
					return CommandType_Unknown;
			}
			if(first == 'c')
			{
				Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
				ChangeMode(Insert);
			}
			else if(first == 'y')
				Editor_ExecCommand(hWnd, EEID_EDIT_COPY);
			else if(first == 'd')
				Editor_ExecCommand(hWnd, EEID_EDIT_CUT);
			else if(first == '>')
			{
				pt.x++;
				Editor_ExecCommand(hWnd, EEID_TAB);
			}
			else if(first == '<')
			{
				if(pt.x > 0) pt.x--;
				Editor_ExecCommand(hWnd, EEID_SHIFT_TAB);
			}

			Editor_ExecCommand(hWnd, EEID_ESCAPE);
			SetCaret(hWnd, pt.x, pt.y);
			Editor_SetScrollPos(hWnd, &scrollPos);
		}
		return first == 'y' ? CommandType_Volatile : CommandType_Record;
	case 'r':
		if(!hasSelection)
			while(repeat--)
				if(!IsEndOfLine(hWnd))
					MoveCaret(hWnd, 1, 0, true);

		{
			std::wstring buf;
			UINT_PTR len = GetSelectionText(hWnd, buf);
			if(len)
			{
				Editor_ExecCommand(hWnd, EEID_DELETE);
				for(size_t i = 0; i < buf.length(); ++i)
				{
					if(buf[i] != '\r' && buf[i] != '\n')
						buf[i] = second;
				}
				InsertString(hWnd, buf.c_str());
				//Editor_OverwriteW(hWnd, buf.c_str(), true);
			}
		}
		return CommandType_Record;
	case 'g':
		switch(second)
		{
		case 'g':
			Editor_ExecCommand(hWnd, EEID_TOP);
			return CommandType_Motion;
		case 'u':
			Editor_ExecCommand(hWnd, EEID_MAKE_LOWER);
			return CommandType_Record;
		case 'U':
			Editor_ExecCommand(hWnd, EEID_MAKE_UPPER);
			return CommandType_Record;
		case 'c':
			Editor_ExecCommand(hWnd, EEID_EDIT_COMMENT);
			return CommandType_Record;
		case 'C':
			Editor_ExecCommand(hWnd, EEID_EDIT_UNCOMMENT);
			return CommandType_Record;
		case 'E':
		case 'e':
			while(repeat--)
				MoveNextWord(hWnd, hasSelection, false, second == 'E', true);
			return CommandType_Motion;
		case 'i':
			SetCaret(hWnd, endInputPos.x, endInputPos.y);
			ChangeMode(Insert);
			return CommandType_Record;
		case 'I':
			Editor_ExecCommand(hWnd, EEID_HOME);
			ChangeMode(Insert);
			return CommandType_Record;
		}
	case 'f':
	case 'F':
	case 't':
	case 'T':
		{
			lastFindCmd.first = first;
			lastFindCmd.second = second;
			lastFindCmd.repeat = repeat;
			bool backward = (first == 'F' || first == 'T');
			bool before = (first == 't' || first == 'T');
			POINT_PTR pos = MoveCaret(hWnd, 0, 0);
			std::wstring buf;
			GetLineText(hWnd, pos.y, buf); 
			for(INT_PTR i = (INT_PTR)pos.x + (backward ? -1 : 1); (i >= 0 && i < (INT_PTR)buf.size());)
			{
				if(buf[i] == second)
				{
					if(before && i > 0)
						i += backward ? 1 : -1;
					SetCaret(hWnd, i, pos.y, hasSelection);
					break;
				}
				i += backward ? -1 : 1;
			}
		}
		return CommandType_Volatile;
	case 'z':
		switch(second)
		{
		case 'z':
		case 0x0D://CR
		case '.':
		case '-':
		case 't':
		case 'b':
			{
				POINT_PTR spos;
				Editor_GetScrollPos(hWnd, &spos);
				POINT_PTR cpos;
				Editor_GetCaretPos(hWnd, POS_LOGICAL_W, &cpos);
				SIZE_PTR size;
				Editor_GetPageSize(hWnd, &size);
				if(second == 0x0D || second == 't')
					spos.y += cpos.y - spos.y;
				else if(second == '-' || second == 'b')
					spos.y += cpos.y - (spos.y+size.cy);
				else
					spos.y += cpos.y - (spos.y+size.cy/2);
				if(spos.y < 0)
					spos.y = 0;
				Editor_SetScrollPos(hWnd, &spos);
				return CommandType_Motion;
			}
			break;
		}
		break;
	case 'Z':
		switch(second)
		{
		case 'Z':
			Editor_ExecCommand(hWnd, EEID_FILE_SAVE_EXIT);
			return CommandType_Record;
		}
		break;
	}
	return CommandType_Unknown;
}

std::wstring EmulationVI::GetStatus() const
{
	std::wstring buf;
	{
		switch(GetCurMode())
		{
		case Command:
			buf += L"-- NORMAL --";
			break;
		case Visual:
			buf += L"-- VISUAL --";
			break;
		case Insert:
			buf += L"-- INSERT --";
			break;
		case Overwrite:
			buf += L"-- REPLACE --";
			break;
		}
	}
	buf += L"\t";
	if(!curCommands.empty())
	{
		buf.insert(buf.end(), curCommands.begin(), curCommands.end());
	}

	buf += L"\t";
	{
		SIZE_PTR sz = GetSelectionSize(hWnd);
		if(sz.cx || sz.cy)
		{
			wchar_t msg[256];
			wsprintf(msg, L"(%d, %d)", sz.cx, sz.cy+1);
			buf += msg;
		}
	}
	return buf;
}
void EmulationVI::UpdateStatus()
{
	Editor_SetStatusW(hWnd, GetStatus().c_str());
}