﻿#include "config.h"
#include "CaretImeColor.h"
#include "tools.h"
#include "thread_hook.h"
#include "choice_color.h"
#include "smart_color.h"
#include "option.h"
#include "plugin.h"
#include "resource.h"
#include <commctrl.h>
#include <utility>
#include <vector>

#undef min
#undef max

namespace CaretImeColor
{
	static_assert( sizeof(::COLORREF) == sizeof(::DWORD), "::COLORREF size error." );

	//	キャレットサイズ設定ダイアログ表示
	extern bool	ShowChangeCaretSize( ::HWND editor, ::HWND parent );

//----------------------------------------------------------------------------

	//	排他
	static tools::Mutex mutex;

	//	各スレッドに対するフック
	static tools::ThreadHook hook( mutex );

	//	色情報
	struct Colors {
		bool		fixed;		//	キャレット色固定モード
		::COLORREF	caret;		//	キャレット色
		::COLORREF	back;		//	合成元背景色
		Colors( ::COLORREF col, bool fix ): fixed(fix), caret(col), back(RGB(255,255,255)) {}
	};

	//	複数選択対応バージョンか？
	static bool	hasMultiSelect;

	//	色情報
	static const Colors	 openDefaultColors( RGB(255,0,0), true	);
	static const Colors	closeDefaultColors( RGB(  0,0,0), false	);
	static Colors		 openColors(  openDefaultColors );
	static Colors		closeColors( closeDefaultColors );

	//	アクティブドキュメント
	static ::HWND		latestActiveDocument;
	//	最後に設定したキャレット色
	static ::COLORREF	latestStoreCaretColor = -1;

//----------------------------------------------------------------------------

	//	設定を読み取る
	void	LoadOptions( ::HWND hwnd )
	{
		option::Reader reader(hwnd);
		 openColors.caret = reader.UInt32(	L"openColor",	 openColors.caret );
		closeColors.caret = reader.UInt32(	L"closeColor",	closeColors.caret );
		 openColors.fixed = reader.Bool(	L"openFixed",	 openColors.fixed );
		closeColors.fixed = reader.Bool(	L"closeFixed",	closeColors.fixed );
	}

	//	設定を保存する
	void	SaveOptions( ::HWND hwnd )
	{
		option::Writer writer(hwnd);
		writer.UInt32(	L"openColor",	 openColors.caret );
		writer.UInt32(	L"closeColor",	closeColors.caret );
		writer.Bool(	L"openFixed",	 openColors.fixed );
		writer.Bool(	L"closeFixed",	closeColors.fixed );
	}

	//	今の設定のプロパティの「表示」「カーソルのある行」の背景色を取得
	::COLORREF	GetCurrentBackColor( ::HWND hwnd )
	{
		//	14.4 以降は組み込み関数で取得可能
		static const bool builtin = []( ::HWND editor ){
			return	Editor_GetVersion(editor) >= 14400;
		}( hwnd );

		//	カーソル行の背景色取得
		::COLORREF	color;
		tools::OptionalSmartColor	smart_color;
		if( !builtin || !Editor_GetColor( hwnd, false, SMART_COLOR_CURLINE, nullptr, &color, nullptr ) ){
			smart_color.Activate(hwnd);
			color = smart_color->GetBackground( SMART_COLOR_CURLINE );
		}

		//	システム色処理
		switch( color ){
		case DEFAULT_COLOR:
			//	標準指定なので、ウィンドウ色取得
			color = ::GetSysColor( COLOR_WINDOW );
			break;
		case TRANSPARENT_COLOR:
			//	透過指定なので、通常の背景色取得
			if( !builtin || !Editor_GetColor( hwnd, false, SMART_COLOR_NORMAL, nullptr, &color, nullptr ) ){
				smart_color.Activate(hwnd);
				color = smart_color->GetBackground( SMART_COLOR_NORMAL );
			}
			switch( color ){
			case TRANSPARENT_COLOR:
			case DEFAULT_COLOR:
				color = ::GetSysColor( COLOR_WINDOW );
				break;
			}
			break;
		}
		return	color;
	}

//----------------------------------------------------------------------------
	namespace Property
	{
		//	ダイアログボックスへ現行の設定から入力
		void	Import( ::HWND dlg )
		{
			auto ImportColorBox = []( ::HWND dlg, ::UINT box_id, ::COLORREF color ){
				if( auto ctrl = ::GetDlgItem(dlg,box_id) )
					::SetWindowLongPtr( ctrl, GWLP_USERDATA, color );
			};

			ImportColorBox(	  dlg, IDC_COLOR_BOX_OPEN ,	 openColors.caret );
			ImportColorBox(	  dlg, IDC_COLOR_BOX_CLOSE,	closeColors.caret );
			::CheckDlgButton( dlg, IDC_AUTO_OPEN,		 openColors.fixed ? 0 : 1 );
			::CheckDlgButton( dlg, IDC_AUTO_CLOSE,		closeColors.fixed ? 0 : 1 );
		}

		//	ダイアログボックスから現行の設定へ出力
		void	Export( ::HWND dlg )
		{
			auto ExportColorBox = []( ::HWND dlg, ::UINT box_id, ::COLORREF& color ){
				if( auto ctrl = ::GetDlgItem(dlg,box_id) )
					color = static_cast<::COLORREF>(::GetWindowLongPtr( ctrl, GWLP_USERDATA ));
			};

			ExportColorBox( dlg, IDC_COLOR_BOX_OPEN ,  openColors.caret );
			ExportColorBox( dlg, IDC_COLOR_BOX_CLOSE, closeColors.caret );
			 openColors.fixed = ::IsDlgButtonChecked( dlg, IDC_AUTO_OPEN  ) == 0;
			closeColors.fixed = ::IsDlgButtonChecked( dlg, IDC_AUTO_CLOSE ) == 0;
		}

		//	現行の設定をリセット
		void	Reset( ::HWND dlg )
		{
			 openColors =  openDefaultColors;
			closeColors = closeDefaultColors;
			Import( dlg );
			::InvalidateRect( GetDlgItem( dlg, IDC_COLOR_BOX_OPEN  ), nullptr, false );
			::InvalidateRect( GetDlgItem( dlg, IDC_COLOR_BOX_CLOSE ), nullptr, false );
		}

		//	色を選択
		bool	ChoiceColor( ::HWND dlg, ::UINT box_id, ::UINT auto_id = 0 )
		{
			if( auto const box = ::GetDlgItem( dlg, box_id ) ){
				//	背景関係
				auto const enable_xor( ::IsDlgButtonChecked( dlg, auto_id ) );
				auto const back( static_cast<::COLORREF>(::GetWindowLongPtr( dlg, DWLP_USER )) );

				//	色選択
				auto color( static_cast<::COLORREF>(::GetWindowLongPtr( box, GWLP_USERDATA )) );
				if( enable_xor ) color = 0x00ffffff & (color ^ back ^ RGB(255,255,255));
				if( tools::ShowChoiceColor( reinterpret_cast<::HWND>(::GetWindowLongPtr(dlg, GWLP_USERDATA )), dlg, color ) ){
					if( enable_xor ) color = 0x00ffffff & (color ^ back ^ RGB(255,255,255));
					::SetWindowLongPtr(	box, GWLP_USERDATA, color );

					::InvalidateRect(	box, nullptr, false );
					return	true;
				}
			}else{
				::MessageBeep( MB_ICONERROR );
			}
			return	false;
		}

		//	色サンプル描画
		void	DrawColorBox( ::HWND dlg, ::DRAWITEMSTRUCT const& dat, ::UINT auto_id = 0 )
		{
			auto color( ::GetWindowLongPtr( dat.hwndItem, GWLP_USERDATA ) );
			if( auto_id && ::IsDlgButtonChecked( dlg, auto_id ) ){
				auto const back( static_cast<::COLORREF>(::GetWindowLongPtr( dlg, DWLP_USER )) );
				color = 0x00ffffff & (color ^ back ^ RGB(255,255,255));
			}

			auto const brush( ::CreateSolidBrush( static_cast<::COLORREF>(color) ) );
			::FillRect( dat.hDC, &dat.rcItem, brush );
			::DeleteObject( brush );
		}


		//	プロパティダイアログプロシージャ
		::INT_PTR CALLBACK Procedure( ::HWND dlg, ::UINT msg, ::WPARAM wParam, ::LPARAM lParam )
		{
			switch( msg ){

			//	初期化
			case WM_INITDIALOG:
				{
					::SetWindowLongPtr(	dlg, GWLP_USERDATA, lParam );
					::SetWindowLongPtr(	dlg, DWLP_USER, GetCurrentBackColor(reinterpret_cast<::HWND>(lParam)) );
					::SetWindowText( dlg, tools::GetString( IDS_NAME ).c_str() );
					Import( dlg );
				}
				break;

			case WM_COMMAND:
				switch( HIWORD(wParam) ){
				case BN_CLICKED:
					switch( LOWORD(wParam) ){
				
					//	新たな色設定を保存し閉じる
					case IDOK:
						Export( dlg );
						SaveOptions( reinterpret_cast<::HWND>(::GetWindowLongPtr( dlg, GWLP_USERDATA )) );
						::EndDialog( dlg, IDOK );
						return	true;

					//	変更せずに閉じる
					case IDCANCEL:
						::EndDialog( dlg, IDCANCEL );
						return	true;

					//	色情報更新
					case IDC_AUTO_OPEN:
						::InvalidateRect( GetDlgItem( dlg, IDC_COLOR_BOX_OPEN  ), nullptr, false );
						break;
					case IDC_AUTO_CLOSE:
						::InvalidateRect( GetDlgItem( dlg, IDC_COLOR_BOX_CLOSE ), nullptr, false );
						break;

					//	色を選択する
					case IDC_COLOR_BOX_OPEN:
					case IDC_COLOR_BTN_OPEN:
						ChoiceColor( dlg, IDC_COLOR_BOX_OPEN,  IDC_AUTO_OPEN  );
						return	true;
					case IDC_COLOR_BOX_CLOSE:
					case IDC_COLOR_BTN_CLOSE:
						ChoiceColor( dlg, IDC_COLOR_BOX_CLOSE, IDC_AUTO_CLOSE );
						return	true;

					//	プロパティリセット
					case IDC_RESET:
						if( ::MessageBox( dlg, tools::GetString( IDS_INQUIRE_RESET ).c_str(), tools::GetString( IDS_NAME ).c_str(), MB_YESNO ) == IDYES ){
							Reset( dlg );
							::MessageBox( dlg, tools::GetString( IDS_RESET ).c_str(), tools::GetString( IDS_NAME ).c_str(), MB_OK );
						}
						return	true;

					}
					break;
				}
				break;

			case WM_NOTIFY:
				switch( reinterpret_cast<::LPNMHDR>(lParam)->code ){
				case NM_CLICK:
					switch( reinterpret_cast<::LPNMHDR>(lParam)->idFrom ){
					case IDC_CARET_SIZE:
						// カスタマイズの表示を出したいが、親ウィンドウの指定ができない。あげくフリーズすることがあるので、代わりに自前のを出す
						ShowChangeCaretSize( reinterpret_cast<::HWND>(::GetWindowLongPtr( dlg, GWLP_USERDATA )), dlg );
						return	true;
					}
					break;
				}
				break;

			case WM_DRAWITEM:
				switch( wParam ){
				//	色サンプル描画
				case IDC_COLOR_BOX_OPEN:
					return	lParam && (DrawColorBox( dlg, *reinterpret_cast<DRAWITEMSTRUCT const*>(lParam), IDC_AUTO_OPEN  ), true);
				case IDC_COLOR_BOX_CLOSE:
					return	lParam && (DrawColorBox( dlg, *reinterpret_cast<DRAWITEMSTRUCT const*>(lParam), IDC_AUTO_CLOSE ), true);
				}
				break;
			}
			return	false;
		}
	}

	//	プロパティ開く
	//		hwnd	EmEditorウィンドウハンドル
	bool	ShowProperty( ::HWND hwnd )
	{
		auto const em( tools::FindMostParentWindow(hwnd) );
		if( ::DialogBoxParam( tools::Instance(), MAKEINTRESOURCE(IDD_PROPERTY), hwnd, Property::Procedure, reinterpret_cast<::LPARAM>(em) ) != -1 ){
			latestActiveDocument  = nullptr;
			latestStoreCaretColor = -1;
			return	true;
		}
		return	false;
	}

//----------------------------------------------------------------------------

	//	IMMコンテキスト
	struct imm_context_t {
	public:
		imm_context_t(::HWND wnd) : window_(wnd), context_(::ImmGetContext(window_)) {}
		~imm_context_t() { ::ImmReleaseContext(window_,context_); }
		operator bool () const { return context_ != nullptr; }
		bool operator!() const { return context_ == nullptr; }
		::HIMC	get() const { return context_; }
	private:
		::HWND	window_;
		::HIMC	context_;
		imm_context_t(imm_context_t const&);
		imm_context_t& operator=(imm_context_t const&);
	};

	//	IMEが開かれた状態か？
	inline bool IsOpenIme(::HWND wnd)
	{
		imm_context_t const context(wnd);
		return context && ::ImmGetOpenStatus(context.get());
	}

	//	アクティブドキュメントのカーソル行の背景色から、合成用色情報更新
	void	UpdateBackgroundColor( ::HWND hwnd, ::HWND active_doc )
	{
		//	キャレット色固定が含まれてたら、現設定のプロパティの「表示」「通常」の背景色を取得
		::COLORREF const color_for_fixed( (openColors.fixed || closeColors.fixed) ? GetCurrentBackColor( hwnd ) : RGB(255,255,255) );

		 openColors.back =  openColors.fixed ? color_for_fixed : RGB(255,255,255);
		closeColors.back = closeColors.fixed ? color_for_fixed : RGB(255,255,255);

		latestActiveDocument  = active_doc;
		latestStoreCaretColor = -1;
	}

	//	キャレットの色を変更
	bool	ChangeCaretColor( ::HWND hwnd )
	{
		//	排他制御
		tools::Mutex::Lock lock( mutex );

		//	無限ループ防止
		static bool busy = false;
		if( busy ) return false;
		struct auto_busy_t {
			 auto_busy_t( bool& busy ): busy_(busy) { busy_ = true; }
			~auto_busy_t(){ busy_ = false; }
		private:
			bool&	busy_;
		};
		auto_busy_t const auto_busy( busy );

		//	新色
		Colors&			colors(	IsOpenIme(hwnd) ? openColors : closeColors );
		::DWORD const	new_color( (colors.back ^ colors.caret) & 0x00ffffff );

		//	キャレット色設定更新
		if( latestStoreCaretColor != new_color ){
			latestStoreCaretColor  = new_color;
			if( Editor_RegSetValue( hwnd, EEREG_COMMON, nullptr, L"CaretColor", REG_DWORD, reinterpret_cast<::BYTE const*>(&new_color), sizeof(new_color), 0 ) != ERROR_SUCCESS ){
				return false;
			}

			//	フォーカス再設定でキャレット更新
			if( auto prev = ::GetFocus() ){
				if( prev == hwnd && tools::IsViewWindow(hwnd) ){
					::PostMessage( prev, WM_KILLFOCUS, 0, 0 );
					::PostMessage( prev, WM_SETFOCUS, reinterpret_cast<::WPARAM>(prev), 0 );
				}

				//	複数選択時は追加処理
				if( hasMultiSelect && Editor_GetMultiSel( hwnd, -1, nullptr ) ){
					if( Editor_GetRedraw( hwnd ) ){
						Editor_Redraw( hwnd, false );
						Editor_Redraw( hwnd, true  );
					}
				}
			}
		}
		return	true;
	}

//----------------------------------------------------------------------------

	// スレッドへのメッセージのフック関数
	::LRESULT CALLBACK SendMsgHookProc( int code, ::WPARAM wParam, ::LPARAM lParam )
	{
		if( code >= 0 && code == HC_ACTION ){
			if( auto const msg = reinterpret_cast<::CWPRETSTRUCT const*>(lParam) ){
				//	IMEの状態が変化したら、キャレット色変更
				if( msg->message == WM_IME_NOTIFY ){
					if( msg->wParam == IMN_SETOPENSTATUS || msg->wParam == IMN_OPENSTATUSWINDOW ){
						if( tools::IsViewWindow(msg->hwnd) ){
							ChangeCaretColor( msg->hwnd );
						}
					}
				}
			}
		}
		return CallNextHookEx( hook.Get(), code, wParam, lParam );
	}

//----------------------------------------------------------------------------

	//	イベント応答
	void	OnEvents( ::HWND hwnd, ::UINT nEvent, ::LPARAM lParam ) 
	{
		//	作成された
		if( nEvent & EVENT_CREATE ){
			//	設定を読み取る
			LoadOptions( hwnd );

			//	複数選択があるバージョンか？
			hasMultiSelect = (Editor_GetVersion(hwnd) / 1000) >= 13;

			//	キャレット色をシステム設定を使わない設定にしておく
			::DWORD value(FALSE);
			Editor_RegSetValue( hwnd, EEREG_COMMON, nullptr, L"UseSysColorCaret", REG_DWORD, reinterpret_cast<::BYTE const*>(&value), sizeof(value), 0 );

			//	(未フックなら)フックする
			hook.Hook( WH_CALLWNDPROCRET, SendMsgHookProc );
		}
		//	フレーム作成された
		if( nEvent & EVENT_CREATE_FRAME ){
			//	(未フックなら)フックする
			hook.Hook( WH_CALLWNDPROCRET, SendMsgHookProc );
		}

		//	必要なら合成用色情報更新し、キャレット色更新
		if( nEvent & EVENT_CONFIG_CHANGED ){
			tools::Mutex::Lock lock( mutex );
			auto const active_doc( reinterpret_cast<::HWND>(Editor_Info( hwnd, EI_GET_ACTIVE_DOC, 0 )) );
			UpdateBackgroundColor( hwnd, active_doc );
			ChangeCaretColor( hwnd );
		}else if( nEvent & (EVENT_SET_FOCUS | EVENT_DOC_SEL_CHANGED) ){
			tools::Mutex::Lock lock( mutex );
			auto const active_doc( reinterpret_cast<::HWND>(Editor_Info( hwnd, EI_GET_ACTIVE_DOC, 0 )) );
			if( latestActiveDocument != active_doc )
				UpdateBackgroundColor( hwnd, active_doc );
			else if( nEvent & EVENT_DOC_SEL_CHANGED )
				latestStoreCaretColor = -1;
			ChangeCaretColor( hwnd );
		}

		//	フレーム閉じる前
		if( nEvent & EVENT_CLOSE_FRAME ){
			//	フック解除
			hook.Unhook();
		}
		//	プラグイン解放前
		if( nEvent & EVENT_CLOSE ){
			//	全フック解除
			hook.AllUnhook();
		}
	}

	//	プロセスにアタッチされた
	void	OnAttachProcess()
	{
	}
	
	//	プロセスからデタッチされる
	void	OnDetachProcess()
	{
		//	全フック解除
		hook.AllUnhook();
	}
	
	//	スレッドにアタッチされた
	void	OnAttachThread()
	{
	}

	//	スレッドからデタッチされる
	void	OnDetachThread()
	{
		//	フック解除
		hook.Unhook();
	}

//----------------------------------------------------------------------------

	//	コマンド実行可能か？
	bool	CanCommand( ::HWND hwnd )
	{
		return	true;
	}

	//	チェックが付いた状態か？
	bool	IsChecked( ::HWND hwnd )
	{
		return	false;
	}

	//	コマンド実行
	void	OnCommand( ::HWND hwnd )
	{
		ShowProperty( hwnd );
	}

//----------------------------------------------------------------------------
}

