#include "registry.h"
#include <assert.h>

#undef min
#undef max

using namespace std;

static_assert( sizeof(INT32) == sizeof(float),	 "float size mismatch." );
static_assert( sizeof(INT64) == sizeof(double), "double size mismatch." );

namespace registry
{
////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//	J
	HKEY	Reader::Open( HKEY key, wchar_t const* subkey )
	{
		HKEY	new_key(nullptr);
		return	(RegOpenKeyEx( key, subkey, 0, KEY_READ, &new_key ) == ERROR_SUCCESS) ? new_key : nullptr;
	}

	//	f[^ǂݎ
	LONG	Reader::QueryValue( wchar_t const* name, DWORD type, void* data, DWORD& size ) const
	{
		return	RegQueryValueEx( key_, name, nullptr, &type, reinterpret_cast<BYTE*>(data), &size );
	}

	//	擾
	auto	Reader::GetInfo( wchar_t const* name ) const -> Info
	{
		Info	info;
		info.result = RegQueryValueEx( key_, name, nullptr, &info.type, nullptr, &info.size );
		return	info;
	}

	//	32bitǂݎ
	INT32	Reader::Int32(	wchar_t const* name, INT32 defaultValue ) const
	{
		auto info( GetInfo( name ) );
		if( info && info.type == REG_DWORD && info.size == sizeof(INT32) ){
			INT32 value( defaultValue );
			if( QueryValue( name, info.type, &value, info.size ) == ERROR_SUCCESS ){
				assert( info.size == sizeof(INT32) );
				return value;
			}
		}
		return	defaultValue;
	}

	//	64bitǂݎ
	INT64	Reader::Int64(	wchar_t const* name, INT64 defaultValue ) const
	{
		auto info( GetInfo( name ) );
		if( info && info.type == REG_QWORD && info.size == sizeof(INT64) ){
			INT64 value( defaultValue );
			if( QueryValue( name, info.type, &value, info.size ) == ERROR_SUCCESS ){
				assert( info.size == sizeof(INT64) );
				return value;
			}
		}
		return	defaultValue;
	}

	//	_lǂݎ
	bool	Reader::Bool( wchar_t const* name, bool defaultValue ) const
	{
		return	Int32( name, defaultValue ? 1 : 0 ) != 0;
	}

	//	Px_ǂݎ
	float	Reader::Float( wchar_t const* name, float defaultValue ) const
	{
		INT32 value( Int32( name, *reinterpret_cast<INT32*>(&defaultValue) ) );
		return	*reinterpret_cast<float*>(&value);
	}

	//	{x_ǂݎ
	double	Reader::Double( wchar_t const* name, double defaultValue ) const
	{
		INT64 value( Int64( name, *reinterpret_cast<INT64*>(&defaultValue) ) );
		return	*reinterpret_cast<double*>(&value);
	}

	//	ǂݎ
	wstring	Reader::String(	wchar_t const* name, wchar_t const*	defaultValue ) const
	{
		wstring	result;
		auto info( GetInfo( name ) );
		if( info && info.type == REG_SZ ){
			result.resize( info.size / sizeof(wchar_t) + 1 );
			DWORD size( static_cast<DWORD>(result.size() * sizeof(wchar_t)) );
			if( QueryValue( name, info.type, &result[0], size ) == ERROR_SUCCESS ){
				size /= sizeof(wchar_t);
				assert( size < result.size() );
				result.resize( size );
				goto Exit;
			}
		}
		if( defaultValue )	result = defaultValue;
		else				result.clear();
	Exit:
		return	result;
	}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	//	J
	HKEY Writer::Open( HKEY key, wchar_t const* subkey )
	{
		HKEY	new_key(nullptr);
		DWORD	disposition(0);
		return	(RegCreateKeyEx( key, subkey, 0, nullptr, REG_OPTION_NON_VOLATILE, KEY_WRITE, nullptr, &new_key, &disposition ) == ERROR_SUCCESS) ? new_key : nullptr;
	}

	//	f[^
	LONG	Writer::SetValue( wchar_t const* name, DWORD type, void const* data, DWORD size )
	{
		return	RegSetValueEx( key_, name, 0, type, reinterpret_cast<BYTE const*>(data), size );
	}

	//	32bit
	bool	Writer::Int32( wchar_t const* name, INT32 value )
	{
		return	SetValue( name, REG_DWORD, &value, sizeof(UINT32) ) != ERROR_SUCCESS;
	}

	//	64bit
	bool	Writer::Int64( wchar_t const* name, INT64 value )
	{
		return	SetValue( name, REG_QWORD, &value, sizeof(UINT64) ) != ERROR_SUCCESS;
	}

	//	_l
	bool	Writer::Bool( wchar_t const* name, bool value )
	{
		return	Int32( name, value ? 1 : 0 );
	}

	//	Px_
	bool	Writer::Float( wchar_t const* name, float value )
	{
		return	Int32( name, *reinterpret_cast<INT32*>(&value) );
	}

	//	{x_
	bool	Writer::Double( wchar_t const* name, double value )
	{
		return	Int64( name, *reinterpret_cast<INT64*>(&value) );
	}

	//	񏑂
	bool	Writer::String( wchar_t const* name, wchar_t const*	value )
	{
		return	SetValue( name, REG_SZ, value ? value : L"", static_cast<DWORD>((value ? (wcslen(value) + 1) : 1) * sizeof(wchar_t)) ) != ERROR_SUCCESS;
	}

	//	l폜
	bool	Writer::Erase( wchar_t const* name )
	{
		return	RegDeleteValue( key_, name ) == ERROR_SUCCESS;
	}

	//	TuL[폜
	bool	Writer::EraseKey( wchar_t const* name )
	{
		return	RegDeleteKey( key_, name ) == ERROR_SUCCESS;
	}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}
