﻿#include <windows.h>
#include <shlwapi.h>
#include <commctrl.h>
#include <tchar.h>
#include <vector>
#include <map>
#include <string>
#include <regex>
#include <type_traits>
#include <assert.h>
#include "tools.h"
#include "option.h"
#include "registry.h"
#include "plugin.h"
#include "resource.h"

#undef min
#undef max

namespace docNavi {
	using namespace std;

//////////////////////////////////////////////////////////////////////////////

	wchar_t const*const	keyHitTop(		L"HIT_TOP"		);
	wchar_t const*const	keyHitInclude(	L"HIT_INCLUDE"	);
	wchar_t const*const	keyRelative(	L"RELATIVE"		);
	wchar_t const*const	keyTraceBack(	L"TRACE_BACK"	);

	size_t const maxTraceBack  = 99; // 遡る時に辿るフォルダ数の上限の最大値
	size_t const initTraceBack =  1; // 遡る時に辿るフォルダ数の上限の初期値

//////////////////////////////////////////////////////////////////////////////

	//	プロパティダイアログプロシージャ
	INT_PTR CALLBACK PropertyProc( HWND dlg, UINT msg, WPARAM wParam, LPARAM lParam )
	{
		switch( msg ){
		case WM_INITDIALOG:{	//	オプションダイアログ初期化
				auto const em( reinterpret_cast<HWND>(lParam) );
				SetWindowLongPtr( dlg, GWLP_USERDATA, reinterpret_cast<LONG_PTR>(em) );

				SendDlgItemMessage( dlg, IDC_TRACEBACK, EM_LIMITTEXT, static_cast<int>(log10(static_cast<double>(maxTraceBack))) + 1, 0 );
				SendDlgItemMessage( dlg, IDC_TRACEBACK_SPIN, UDM_SETRANGE32, 0, maxTraceBack );

				option::Reader reader( em );
				CheckDlgButton(	dlg, IDC_HIT_TOP,	  reader.Bool(  keyHitTop,		true  ) );
				CheckDlgButton(	dlg, IDC_HIT_INCLUDE, reader.Bool(  keyHitInclude,	false ) );
				CheckDlgButton(	dlg, IDC_RELATIVE,	  reader.Bool(  keyRelative,	false ) );
				SetDlgItemInt(	dlg, IDC_TRACEBACK,	  reader.Int32( keyTraceBack,	initTraceBack ), true );

				SetWindowText(  dlg, tools::GetString(IDS_NAME).c_str() );
				SendMessage( dlg, WM_COMMAND, MAKELONG(IDC_RELATIVE,BN_CLICKED), reinterpret_cast<LPARAM>(GetDlgItem(dlg,IDC_RELATIVE)) );
			}
			return	true;
		case WM_COMMAND:
			if( HIWORD(wParam) == BN_CLICKED ){
				switch( LOWORD(wParam) ){
				case IDOK:{			//	新たな設定保存
						auto const  em( reinterpret_cast<HWND>(GetWindowLongPtr(dlg, GWLP_USERDATA )) );
						option::Writer writer( em );
						writer.Bool(  keyHitTop,		IsDlgButtonChecked(	dlg, IDC_HIT_TOP	 ) != 0 );
						writer.Bool(  keyHitInclude,	IsDlgButtonChecked(	dlg, IDC_HIT_INCLUDE ) != 0 );
						writer.Bool(  keyRelative,		IsDlgButtonChecked(	dlg, IDC_RELATIVE	 ) != 0 );
						writer.Int32( keyTraceBack,		GetDlgItemInt(		dlg, IDC_TRACEBACK, NULL, true ) );
						EndDialog( dlg, 1 );
					}
					return	true;
				case IDCANCEL:		//	キャンセル
					EndDialog( dlg, 0 );
					return	true;
				case IDC_HIT_TOP:
					if( !IsDlgButtonChecked( dlg, IDC_HIT_TOP ) ){
						CheckDlgButton( dlg, IDC_HIT_INCLUDE, 1 );
						return	true;
					}
					break;
				case IDC_HIT_INCLUDE:
					if( !IsDlgButtonChecked( dlg, IDC_HIT_INCLUDE ) ){
						CheckDlgButton( dlg, IDC_HIT_TOP, 1 );
						return	true;
					}
					break;
				case IDC_RELATIVE:{
						bool const enable( IsDlgButtonChecked( dlg, IDC_RELATIVE ) != 0 );
						EnableWindow( GetDlgItem( dlg, IDC_RELATIVE_FRAME	), enable );
						EnableWindow( GetDlgItem( dlg, IDC_TRACEBACK_LABEL	), enable );
						EnableWindow( GetDlgItem( dlg, IDC_TRACEBACK		), enable );
						EnableWindow( GetDlgItem( dlg, IDC_TRACEBACK_SPIN	), enable );
					}
					return	true;
				}
			}
			break;
		}
		return	false;
	}

	//	オプションダイアログ表示
	bool	ShowProperty( HWND parent )
	{
		parent = tools::GetTopParent(parent);
		return	DialogBoxParam( tools::Instance(), MAKEINTRESOURCE(IDD_OPTION), parent, PropertyProc, reinterpret_cast<LPARAM>(parent) ) != -1;
	}

//////////////////////////////////////////////////////////////////////////////

	//	スクロールバー再描画要求
	const UINT WM_REDRAW_SCROLLBAR = WM_USER + 1;

	//	リストボックスの指定項目を選択
	//		listBox	リストボックスのハンドル
	//		index	選択項目のインデックス
	//		send	選択項目設定メッセージを send するか
	void	SelectItemForListBox( HWND listBox, LONG_PTR index, bool send = false )
	{
		SendMessage(		listBox, WM_SETREDRAW,		false,	0 );
		if( send )	SendMessage( listBox, LB_SETCURSEL, index,	0 );
		else		PostMessage( listBox, LB_SETCURSEL, index,	0 );
		PostMessage(		listBox, WM_SETREDRAW,		true,	0 );
		InvalidateRect(		listBox, nullptr, false );
		PostMessage( GetParent(listBox), WM_REDRAW_SCROLLBAR, SB_VERT, reinterpret_cast<LPARAM>(listBox) );
	}

	//	リストボックスの次の項目を選択
	//		listBox	リストボックスのハンドル
	//		next	真なら次へ	偽なら前へ
	//		send	選択項目設定メッセージを send するか
	//		result	新たに選択したインデックス	エラーなら LB_ERR
	LONG_PTR	SelectNextItemForListBox( HWND listBox, bool next, bool send = false )
	{
		LONG_PTR const now_index( SendMessage( listBox, LB_GETCURSEL, 0, 0 ) );	//	今の選択項目
		LONG_PTR const count(	  SendMessage( listBox, LB_GETCOUNT,  0, 0 ) );	//	項目数
		if( now_index != LB_ERR && count != LB_ERR && count > 0 )
		{
			//	新たな選択項目を選ぶ
			LONG_PTR new_index( now_index + (next ? 1 : -1) );
			while( new_index < 0 ) new_index += count;
			new_index %= count;

			//	適用
			SelectItemForListBox(listBox, new_index, send );
			return	new_index;
		}
		return	LB_ERR;
	}

	//	リストボックスの次のページへ移動
	//		listBox	リストボックスのハンドル
	//		next	真なら次へ	偽なら前へ
	//		send	選択項目設定メッセージを send するか
	//		result	新たに選択したインデックス	エラーなら LB_ERR
	LONG_PTR	MoveNextPageForListBox( HWND listBox, bool next, bool send = false )
	{
		LONG_PTR const now_index( SendMessage( listBox, LB_GETCURSEL,	  0, 0 ) );	//	今の選択項目
		LONG_PTR const count(	  SendMessage( listBox, LB_GETCOUNT,	  0, 0 ) );	//	項目数
		LONG_PTR const height(	  SendMessage( listBox, LB_GETITEMHEIGHT, 0, 0 ) );	//	一項目の高さ
		if( now_index != LB_ERR && count != LB_ERR && count > 0 && height > 0 )
		{
			//	新たな選択項目を選ぶ
			RECT rc;
			GetClientRect( listBox, &rc );
			LONG_PTR const line_count( (rc.bottom - rc.top) / height );
			LONG_PTR const offset( next ? line_count : -line_count );
			LONG_PTR const new_index( max( min( now_index + offset, count - LONG_PTR(1) ), LONG_PTR() ) );

			//	適用
			SelectItemForListBox( listBox, new_index, send );
			return	new_index;
		}
		return	LB_ERR;
	}


	//	リストボックスの端っこへ移動
	//		listBox	リストボックスのハンドル
	//		bottom	真なら末端へ	偽なら先端へ
	//		send	選択項目設定メッセージを send するか
	//		result	新たに選択したインデックス	エラーなら LB_ERR
	LONG_PTR	MoveVergeForListBox( HWND listBox, bool bottom, bool send = false )
	{
		LONG_PTR const now_index( SendMessage( listBox, LB_GETCURSEL, 0, 0 ) );	//	今の選択項目
		LONG_PTR const count(	  SendMessage( listBox, LB_GETCOUNT,  0, 0 ) );	//	項目数
		if( now_index != LB_ERR && count != LB_ERR && count > 0 )
		{
			//	新たな選択項目を選ぶ
			LONG_PTR const new_index( bottom ? (count - 1) : 0 );

			//	適用
			SelectItemForListBox( listBox, new_index, send );
			return	new_index;
		}
		return	LB_ERR;
	}

//////////////////////////////////////////////////////////////////////////////

	//	インクリメンタルサーチ
	class INCREMENTALSEARCH {
	public:
		//	コンストラクタ・デストラクタ
		 INCREMENTALSEARCH(): output(NULL), listBox(NULL), originIndex(-1), hitTop(true), hitInclude(false) {}
		~INCREMENTALSEARCH(){}

		//	初期化
		//		out		経過を表示するウィンドウ
		//		lbox	処理対象リストボックスコントロール
		//		em		エムエディタ
		void	Initialize( HWND out, HWND lbox, HWND em ){
			option::Reader reader( em );
			output		= out;
			listBox		= lbox;
			originIndex	= -1;
			hitTop		= reader.Bool( keyHitTop,	  true  );
			hitInclude	= reader.Bool( keyHitInclude, false );
			buffer.clear();
			files .clear();
		}

		//	一文字押し込む
		//		key	押し込む文字
		void	Push( wchar_t key )
		{
			if( key ){
				if( buffer.empty() )
					originIndex = SendMessage( listBox, LB_GETCURSEL, 0, 0 );
				buffer.push_back( towlower( key ) );
				SetWindowText( output, buffer.c_str() );
				Update();
			}
		}

		//	一文字取り除く
		void	Pop()
		{
			if(!buffer.empty() ){
				buffer.resize( buffer.size() - 1 );
				SetWindowText( output, buffer.c_str() );
				Update();
			}
		}

		//	クリア
		void	Clear()
		{
			if(!buffer.empty() ){
				buffer.clear();
				SetWindowText( output, buffer.c_str() );
			}
		}
	private:
		HWND					output;			//	経過を表示するウィンドウ
		HWND					listBox;		//	処理対象リストボックスコントロール
		LRESULT					originIndex;	//	検索開始時点で選択されているインデックス
		wstring					buffer;			//	検索文字列
		map<LRESULT,wstring>	files;			//	ファイルリスト
		bool					hitTop;			//	先頭から一致するものにヒット
		bool					hitInclude;		//	含むものにヒット

		//	検索し、選択状態更新
		void	Update()
		{
			//	まだリストボックスから情報を取得してなければ取得
			if( files.empty() ){
				vector<wchar_t>	buf;
				for( LRESULT i(0), num(SendMessage( listBox, LB_GETCOUNT,   0, 0 ) ) ; i < num ; ++i ){
					LRESULT const len( SendMessage( listBox, LB_GETTEXTLEN, i, 0 ) + 32 );
					if( static_cast<LRESULT>(buf.size()) < len ) buf.resize( len + 64 );
					if( SendMessage( listBox, LB_GETTEXT, i, reinterpret_cast<LPARAM>(&buf[0]) ) != LB_ERR ){
						wchar_t* const	top( &buf[0] );
						wchar_t*		ptr(  wcschr( top, L'\t' ) );
						ptr = ptr ? _wcsinc( ptr ) : top;
						_wcslwr_s( ptr, buf.size() - (ptr - top) );
						wchar_t const*const	ptr2( wcschr( ptr, L'\t' ) );
						if( ptr2 )
							files[i] = wstring( ptr, ptr2 );
						else
							files[i] = ptr;
					}
				}
			}

			//	検索
			LRESULT	index( originIndex );
			if( !buffer.empty() ){
				bool			undiscovery( true );
				wchar_t const*	key( buffer.c_str() );
				size_t			len( buffer.size()  );

				//	先頭から一致するものにヒット
				if( hitTop ){
					//	初期位置から末尾まで先頭から検索
					if( undiscovery ){
						bool	mode( true );
						for( auto it(files.begin()), e(files.end()) ; it != e ; ++it ){
							if( mode ) mode = it->first < originIndex;
							if(!mode && wcsncmp( it->second.c_str(), key, min( len, it->second.size() ) ) == 0 ){
								index		= it->first;
								undiscovery	= false;
								break;
							}
						}
					}
					//	先頭から初期位置目前まで先頭から検索
					if( undiscovery ){
						for( auto it(files.begin()), e(files.end()) ; it != e && it->first < originIndex ; ++it ){
							if( wcsncmp( it->second.c_str(), key, min( len, it->second.size() ) ) == 0 ){
								index = it->first;
								undiscovery	= false;
								break;
							}
						}
					}
				}

				//	含むものにヒット
				if( hitInclude ){
					//	初期位置から末尾まで含むものを検索
					if( undiscovery ){
						bool	mode( true );
						for( auto it(files.begin()), e(files.end()) ; it != e ; ++it ){
							if( mode ) mode = it->first < originIndex;
							if(!mode && wcsstr( it->second.c_str(), key ) ){
								index		= it->first;
								undiscovery	= false;
								break;
							}
						}
					}
					//	先頭から初期位置目前まで含むものを検索
					if( undiscovery ){
						for( auto it(files.begin()), e(files.end()) ; it != e && it->first < originIndex ; ++it ){
							if( wcsstr( it->second.c_str(), key ) ){
								index = it->first;
								undiscovery	= false;
								break;
							}
						}
					}
				}
			}

			//	選択位置更新
			SelectItemForListBox( listBox, index );
		}
	};

//////////////////////////////////////////////////////////////////////////////

	//	リストボックスに項目追加
	//		emEditor		EmEditorのハンドル
	//		listBox			リストボックスのハンドル
	//		emIndex			ドキュメントのインデックス
	//		emActiveIndex	アクティブなドキュメントのインデックス
	//		activeIndex		アクティブなドキュメントを登録したらそのリストボックスでのインデックスを格納
	//		activeDir		アクティブなドキュメントのあるフォルダ
	//		relative		相対パス表示するか
	//		traceBack		相対パス表示で遡る時に辿るフォルダ数の上限
	bool	AddItemForListBox( HWND emEditor, HWND listBox, LONG_PTR emIndex, LONG_PTR emActiveIndex, LONG_PTR& activeIndex, wchar_t const* activeDir, bool relative, int traceBack )
	{
		//	タイトル取得
		wchar_t	shortTitle[4096], fullTitle[4096];
		Editor_DocInfo( emEditor, (int)emIndex, EI_GET_SHORT_TITLEW, reinterpret_cast<LPARAM>(shortTitle) );
		Editor_DocInfo( emEditor, (int)emIndex, EI_GET_FULL_TITLEW,  reinterpret_cast<LPARAM>( fullTitle) );

		//	短いタイトルは、「登録されている拡張子は表示しない」の影響を受けるので、拡張子を長いほうから移す
		if( registry::Reader( HKEY_CURRENT_USER, L"Software\\microsoft\\Windows\\CurrentVersion\\Explorer\\Advanced" ).Bool( L"HideFileExt" ) ){
			auto const fullExt( PathFindExtension( fullTitle ) );
			if( fullExt && wcscmp( fullExt, L"." ) != 0 ){
				PathAddExtension( shortTitle, fullExt );
			}
		}

		//	アクティブなドキュメントと同じフォルダにあるなら、それがわかるようにする
		wchar_t	flag[64] = L"";
		if( activeDir && *activeDir ){
			wchar_t	dir[4096];
			Editor_DocInfo( emEditor, static_cast<int>(emIndex), EI_GET_FILE_NAMEW, reinterpret_cast<LPARAM>(dir) );
			PathRemoveFileSpec(  dir );
			PathRemoveBackslash( dir );
			if( _wcsicmp( dir, activeDir ) == 0 )
				wcscat_s( flag, L"f" );
		}

		//	登録する文字列準備
		wchar_t const*	useTitle;
		wchar_t			title[4096];
		if( *shortTitle && *fullTitle ){
			//	おそらく一度でも保存されたドキュメントかGREPの結果
			wchar_t	 relativePath[4096];
			wchar_t* showTitle( fullTitle );

			//	普通のドキュメントなら短いのと長いのは末尾が重複してるので、長い方から重複部分を取り除いてしまう
			size_t sl( wcslen( shortTitle ) ), fl( wcslen( fullTitle ) );
			if( sl < fl && _wcsicmp( &fullTitle[fl - sl], shortTitle ) == 0 ){
				fullTitle[fl - sl] = NULL;

				//	相対パスに変換
				if( relative && emIndex != emActiveIndex ){
					if( PathRelativePathTo( relativePath, activeDir, FILE_ATTRIBUTE_DIRECTORY, fullTitle, FILE_ATTRIBUTE_DIRECTORY ) ){
						showTitle = relativePath;

						//	必要であれば、遡る時に辿るフォルダ数に制限をかける
						if( traceBack < maxTraceBack ){
							PathAddBackslash( relativePath );
							static const wregex regex_prev( L"^\\.\\.\\\\", regex_constants::ECMAScript | regex_constants::optimize );
							if( regex_search( showTitle, regex_prev ) ){
								if( traceBack <= 0 ){
									showTitle = fullTitle;
								}else{
									static const wregex regex_name( L"[^\\\\]+\\\\", regex_constants::ECMAScript | regex_constants::optimize );
									wcmatch	match;
									int		num(0);
									wchar_t const*	str( showTitle );
									while( regex_search( str, match, regex_name, regex_constants::format_first_only ) ){ 
										++num;
										if( traceBack < num ) break;
										auto const& m( match[match.size() - 1] );
										if( m.second <= str ) break;
										str = m.second;
									}
									//	限界を超えてたら、絶対パスに差し戻す
									if( traceBack < num )
										showTitle = fullTitle;
								}
							}
						}
					}
				}
			}

			//	連結
			PathRemoveBackslash( showTitle );
			useTitle = shortTitle;
			swprintf_s( title, L"%s\t%s\t%s", flag, useTitle, showTitle );
		}else{
			//	おそらくまだ一度も保存されてないドキュメント
			useTitle = *shortTitle ? shortTitle : tools::GetString(IDS_NONAME).c_str();
			swprintf_s( title, L"%s\t%s\t%s", flag, useTitle, *fullTitle ? fullTitle : tools::GetString(IDS_NONAME).c_str() );
		}

		//	リストボックスに追加
		LONG_PTR const lbIndex( SendMessage( listBox, LB_ADDSTRING, 0, reinterpret_cast<LPARAM>(title) ) );
		if( lbIndex != LB_ERR ){
			SendMessage( listBox, LB_SETITEMDATA, lbIndex, static_cast<LPARAM>(emIndex) );
			//	アクティブなドキュメントだったなら、記録
			if( emIndex == emActiveIndex ) activeIndex = lbIndex;
			return	true;
		}
		return	false;
	}

//////////////////////////////////////////////////////////////////////////////

	struct ChoiceProcData {
		HWND		parent;
		LONG_PTR	select;
		ChoiceProcData( HWND parent_ ): parent(parent_), select(LB_ERR) {}
	};

	//	ドキュメントリストのメッセージプロシージャ
	INT_PTR CALLBACK ChoiceProc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
	{
	#if !defined(_DEBUG)
		static UINT_PTR				timerID;
	#endif
		static HWND					parent;
		static INCREMENTALSEARCH	incrementalSearch;

		switch( msg ){
		//	ダイアログ開始
		case WM_INITDIALOG:{
			assert( lparam != NULL );
			bool const go( !(GetAsyncKeyState(VK_SHIFT) & 0x80000000) );
			parent = reinterpret_cast<ChoiceProcData*>(lparam)->parent;
			SetWindowLongPtr( dlg, GWLP_USERDATA, lparam );

			//	リストボックス初期化
			HWND const listBox( GetDlgItem( dlg, IDL_FILES ) );
			int	 const tabStop[2] = { 16, 128 };
			SendMessage( listBox, LB_SETTABSTOPS, 2, reinterpret_cast<LPARAM>(tabStop) );
			LONG_PTR lbHeight( SendMessage( listBox, LB_GETITEMHEIGHT, 0, 0 ) );
			if( lbHeight != LB_ERR )
				SendMessage( listBox, LB_SETITEMHEIGHT, 0, MAKELPARAM(lbHeight + 4,0) );

			//	リストボックスのサイズに応じて、ダイアログボックスのサイズを微調整
			RECT rc;
			if( GetWindowRect( listBox, &rc ) ){
				if( AdjustWindowRectEx( &rc, GetWindowLong(dlg,GWL_STYLE), false, GetWindowLong(dlg,GWL_EXSTYLE) ) ){
					SetWindowPos( dlg, nullptr, 0, 0, rc.right - rc.left, rc.bottom - rc.top, SWP_NOMOVE | SWP_NOZORDER );
				}
			}

			//	アクティブなドキュメント
			LONG_PTR const active( Editor_Info( parent, EI_GET_ACTIVE_INDEX, 0 ) );
			wchar_t	activeDir[4096];
			Editor_DocInfo( parent, static_cast<int>(active), EI_GET_FILE_NAMEW, reinterpret_cast<LPARAM>(activeDir) );
			PathRemoveFileSpec(  activeDir );
			PathRemoveBackslash( activeDir );

			//	ドキュメントリスト登録
			map<LONG_PTR,LONG_PTR>	docs;
			for( LONG_PTR index(0), count( Editor_Info( parent, EI_GET_DOC_COUNT, 0 ) ) ; index < count ; ++index ){
				HEEDOC const doc( (HEEDOC)Editor_Info( parent, EI_INDEX_TO_DOC, static_cast<LPARAM>(index) ) );
				if( doc ){
					LONG_PTR const z( Editor_Info( parent, EI_DOC_TO_ZORDER, reinterpret_cast<LPARAM>(doc) ) );
					docs[z] = index;
				}
			}
			option::Reader reader( parent );
			bool const relative(  reader.Bool(  keyRelative,  false			) );
			int  const traceBack( reader.Int32( keyTraceBack, initTraceBack	) );
			LONG_PTR lbActive(0);
			for( auto it(docs.begin()), e(docs.end()) ; it != e ; ++it ){
				AddItemForListBox( parent, listBox, it->second, active, lbActive, activeDir, relative, traceBack );
			}
			incrementalSearch.Initialize( dlg, GetDlgItem(dlg,IDL_FILES), parent );

			//	今のをいったん選択
			SendMessage( listBox, LB_SETCURSEL, lbActive, 0 );
			//	次を選択
			SelectNextItemForListBox( listBox, go, true );

	#if !defined(_DEBUG)
			//	タイマー開始
			timerID = SetTimer( dlg, reinterpret_cast<UINT_PTR>(dlg), 100, NULL );
			PostMessage( dlg, WM_TIMER, timerID, NULL );
	#endif

			EnableWindow( dlg, true );
			EnableWindow( reinterpret_cast<HWND>(GetWindowLongPtr(dlg,GWLP_HWNDPARENT)), false );
			SetFocus( listBox );
			}break;
		//	キー押された
		case WM_VKEYTOITEM:{
				auto const listBox(	reinterpret_cast<HWND>(lparam) );
				auto const key( LOWORD(wparam) );
				switch( key ){
				case VK_UP:		//	前へ
					incrementalSearch.Clear();
					SelectNextItemForListBox( listBox, false );
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					return	true;
				case VK_DOWN:	//	次へ
					incrementalSearch.Clear();
					SelectNextItemForListBox( listBox, true );
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					return	true;
				case VK_PRIOR:	//	PageUp
					incrementalSearch.Clear();
					MoveNextPageForListBox( listBox, false );
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					return	true;
				case VK_NEXT:	//	PageDown
					incrementalSearch.Clear();
					MoveNextPageForListBox( listBox, true );
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					return	true;
				case VK_HOME:	//	Home:
					incrementalSearch.Clear();
					MoveVergeForListBox( listBox, false );
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					return	true;
				case VK_END:	//	End:
					incrementalSearch.Clear();
					MoveVergeForListBox( listBox, true );
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					return	true;
				case VK_RETURN:	//	決定
					incrementalSearch.Clear();
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					SendMessage( dlg, WM_CLOSE, 0, 0 );
					return	true;
				case VK_TAB:	//	次か前を選択
					incrementalSearch.Clear();
					SelectNextItemForListBox( listBox, !(GetAsyncKeyState( VK_SHIFT ) & 0x80000000) );
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					return	true;
				case VK_BACK:	//	バックスペース
					incrementalSearch.Pop();
					SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					return	true;
				default:{		//	インクリメンタルサーチに追加
						BYTE	dummy[256] = {};
						if( GetAsyncKeyState( VK_SHIFT ) & 0x80000000 ) dummy[VK_SHIFT] = 0xff;
						wchar_t	work[64];
						if( ToUnicode( key, 0x80000000u, dummy, work, static_cast<int>(extent<decltype(work)>::value), 0u ) == 1 )
							incrementalSearch.Push( work[0] );
						SetWindowLongPtr( dlg, DWLP_MSGRESULT, -2 );
					}
					return	true;
				}
			}
			break;
		//	スクロールバー再描画
		case WM_REDRAW_SCROLLBAR:
			if( auto const wnd = reinterpret_cast<HWND>(lparam) ){
				auto const bar( static_cast<int>(wparam) );
				SetScrollPos( wnd, bar, 
				GetScrollPos( wnd, bar ), true );
			}
			return	true;
	#if !defined(_DEBUG)
		//	タイマー
		case WM_TIMER:
			if( wparam == timerID ){
				if( !(GetAsyncKeyState(VK_CONTROL) & 0x80000000) ){
					SendMessage( dlg, WM_CLOSE, 0, 0 );
					return	true;
				}
			}
			return	true;
	#endif
		//	閉じる
		case WM_CLOSE:{
			HWND	 const listBox(	GetDlgItem( dlg, IDL_FILES ) );
			LONG_PTR const lbIndex(	SendMessage( listBox, LB_GETCURSEL,	  0,	   0 ) );
			LONG_PTR const index(	SendMessage( listBox, LB_GETITEMDATA, lbIndex, 0 ) );
			if( index != LB_ERR ){
				if( auto data = reinterpret_cast<ChoiceProcData*>(GetWindowLongPtr( dlg, GWLP_USERDATA )) ){
					data->select = index;
				}else{
					Editor_DocInfo( parent, static_cast<int>(index), EI_SET_ACTIVE_INDEX, 0 );
				}
			}
			EnableWindow(  reinterpret_cast<HWND>(GetWindowLongPtr(dlg,GWLP_HWNDPARENT)), true );
			EnableWindow(  dlg, false );
			DestroyWindow( dlg );
			}return	true;
		//	破棄
		case WM_DESTROY:
	#if !defined(_DEBUG)
			if( timerID ){
				KillTimer( dlg, timerID );
				timerID = 0;
			}
	#endif
		
			EnableWindow( reinterpret_cast<HWND>(GetWindowLongPtr(dlg,GWLP_HWNDPARENT)), true );
			EnableWindow( dlg, false );
			break;
		}
		return	false;
	}

	//	ドキュメントリスト開く
	//		hwnd	EmEditorウィンドウハンドル
	void	ShowDocumentList( HWND hwnd )
	{
		ChoiceProcData	data(		tools::GetTopParent( hwnd ) );
		BOOL const		prevEnable(	IsWindowEnabled( data.parent ) );
		HWND const		dlg(		CreateDialogParam( tools::Instance(), MAKEINTRESOURCE(IDD_MAIN), data.parent, ChoiceProc, reinterpret_cast<LPARAM>(&data) ) );
		MSG				msg;
		while( GetMessage( &msg, dlg, NULL, NULL ) > 0 ){
			TranslateMessage( &msg );
			SendMessage( msg.hwnd, msg.message, msg.wParam, msg.lParam );
			if( msg.message == WM_DESTROY && msg.hwnd == dlg ) break;
		}
		EnableWindow( data.parent, prevEnable );
		if( data.select != LB_ERR ){
			Editor_DocInfo( data.parent, static_cast<int>(data.select), EI_SET_ACTIVE_INDEX, 0 );
		}
	}

};

