﻿#include "stdafx.h"
#include <cstring>
#include <cwchar>
#include <cmath>
#include <cassert>
#include "ScrollMargin.h"
#include "tools.h"
#include "option.h"
#include "resource.h"
#include <commctrl.h>

#undef min
#undef max

namespace ScrollMargin {
//----------------------------------------------------------------------------
	namespace
	{
		//	最大再描画遅延時間
		::DWORD const MaxRedrawDelay = 1000;

		//	ヘディング対応バージョンか
		bool	hedingable	= false;

		//	スクロールマージン行数
		::INT32	scrollMarginUp	 = 2;
		::INT32	scrollMarginDown = 2;

		//	マウスボタン押下中無視する
		bool	mouseCancel	= true;

		//	再描画遅延時間
		std::pair<::DWORD,bool>	redrawDelay;

		//	再描画要求
		__declspec(thread)	bool		redrawRequest = false;
		__declspec(thread)	::UINT_PTR	redrawTimer = 0;
		__declspec(thread)	::HWND		redrawWindow;
	}
//----------------------------------------------------------------------------

	//	プラグインの情報取得
	::LRESULT	GetPluginInfo( ::HWND const em, ::UINT_PTR const flag )
	{
		switch( flag ){
		case EPGI_ALLOW_OPEN_SAME_GROUP:	return	TRUE;		//	ファイルを同じグループ内で開くことを許す場合に TRUE を返します。 
		case EPGI_ALLOW_MULTIPLE_INSTANCES:	return	TRUE;		//	プラグインが複数インスタンスをサポートする場合に TRUE を返します。プラグインが 2 個以上のフレームで同時に動作することが許される場合には、このメッセージは TRUE を返す必要があります。複数インスタンスが実行している間、グローバル変数は共有されることに注意してください。 
		case EPGI_MAX_EE_VERSION:			return	99 * 1000;	//	対応するもっとも新しい EmEditor のバージョン番号 * 1000 を返します。	とりあえず・・・
		case EPGI_MIN_EE_VERSION:			return	 7 * 1000;	//	対応するもっとも古い EmEditor のバージョン番号 * 1000 を返します。 
		case EPGI_SUPPORT_EE_PRO:			return	TRUE;		//	EmEditor Professional をサポートする場合に TRUE を返します。 
		case EPGI_SUPPORT_EE_STD:			return	FALSE;		//	EmEditor Standard をサポートする場合に TRUE を返します。 
		}
		return	0;
	}

//----------------------------------------------------------------------------

	//	既定の再描画遅延時間
	inline ::DWORD	GetDefaultRedrawDelay()
	{
		::DWORD time(0);
		if( ::SystemParametersInfo( SPI_GETKEYBOARDSPEED, 0, &time, 0 ) ){
			return	(std::min)( (std::max)( static_cast<::DWORD>(100 / (time * ((30 - 2.5) / 31) + 2.5) * 1.20 + 1) * 10ul, 1ul ), MaxRedrawDelay );
		}else{
			return	100ul;
		}
	}

	//	設定を読み取る
	void	LoadOptions( ::HWND hwnd )
	{
		option::Reader reader(hwnd);
		scrollMarginUp		= reader.Int32(	L"scrollMarginUp",	 scrollMarginUp	  );
		scrollMarginDown	= reader.Int32(	L"scrollMarginDown", scrollMarginDown );
		mouseCancel			= reader.Bool(	L"mouseCancel",		 mouseCancel	  );
		redrawDelay.first	= reader.UInt32(L"redrawDelay",		 0 );
		if( redrawDelay.first ){
			redrawDelay.second = true;
			redrawDelay.first  = (std::min)( (std::max)( redrawDelay.first, 1ul ), MaxRedrawDelay );
		}else{
			redrawDelay.second = false;
			redrawDelay.first  = GetDefaultRedrawDelay();
		}
	}

	//	設定を保存する
	void	SaveOptions( ::HWND hwnd )
	{
		option::Writer writer(hwnd);
		writer.Int32( L"scrollMarginUp",	scrollMarginUp		);
		writer.Int32( L"scrollMarginDown",	scrollMarginDown	);
		writer.Bool(  L"mouseCancel",		mouseCancel			);
		if( redrawDelay.second )
			writer.UInt32( L"redrawDelay",	redrawDelay.first	);
	}

//----------------------------------------------------------------------------

	//	スクロール量算出
	::LONG_PTR	ComputeScrollLines( ::HWND hwnd )
	{
		//	ヘディング
		auto const heding( hedingable ? Editor_Info( hwnd, EI_GET_HEADING_LINES, 0 ) : 0 );

		//	一画面の行数
		::SIZE_PTR page_size;
		Editor_GetPageSize( hwnd, &page_size );

		//	左上からの相対表示座標
		::POINT_PTR caret, scroll;
		Editor_GetCaretPos(  hwnd, POS_VIEW, &caret );
		Editor_GetScrollPos( hwnd, &scroll );
		auto const relative_caret( caret.y - scroll.y - heding );

		//	スクロール量算出
		if( relative_caret < page_size.cy / 2 ){
			auto const margin( (std::min)( static_cast<::LONG_PTR>(scrollMarginUp),	  page_size.cy / 3 ) );
			return	(std::max)(	static_cast<::LONG_PTR>(0),	margin - relative_caret					   );	//	上部
		}else{
			auto const margin( (std::min)( static_cast<::LONG_PTR>(scrollMarginDown), page_size.cy / 3 ) );
			return -(std::max)(	static_cast<::LONG_PTR>(0),	margin + relative_caret	- page_size.cy + 1 );	//	下部
		}
	}

	//	マージン適用
	void	ApplyMargin( ::HWND hwnd )
	{
		//	マウスボタン押下中は無視
		if( mouseCancel ){
			if( (::GetAsyncKeyState(VK_LBUTTON) & 0x8000) ) return;
			if( (::GetAsyncKeyState(VK_RBUTTON) & 0x8000) ) return;
			if( (::GetAsyncKeyState(VK_MBUTTON) & 0x8000) ) return;
		}

		//	スクロール
		auto const count( ComputeScrollLines( hwnd ) );
		if( count ){
			::POINT_PTR	prev_pos;
			Editor_GetScrollPos( hwnd, &prev_pos );
			::POINT_PTR	new_pos = {
				prev_pos.x,
				(std::max)( prev_pos.y - count, static_cast<::LONG_PTR>(0) )
			};
			if( prev_pos.y != new_pos.y ){
				Editor_SetScrollPos( hwnd, &new_pos );
				Editor_GetScrollPos( hwnd, &new_pos );

				//	一度に２行以上スクロールさせると、表示が乱れることがあるのでその対策
				if( std::abs(new_pos.y - prev_pos.y) > 1 ){
					redrawRequest = true;
				}

				//	フリーカーソル時に、横スクロール位置がおかしくなるので補正（不完全）
				if( new_pos.x < prev_pos.x ){
					::POINT_PTR caret;
					Editor_GetCaretPos(   hwnd, POS_VIEW, &caret );
					Editor_SetCaretPosEx( hwnd, POS_VIEW, &caret, TRUE );
				}
			}
		}
	}

	//	再描画
	void	DelayRedraw( ::HWND hwnd )
	{
		//	一定時間後に全再描画
		if(	redrawTimer )
			::KillTimer( nullptr, redrawTimer );
		redrawWindow = hwnd;
		redrawTimer	 = ::SetTimer( nullptr, 1, redrawDelay.first, []( ::HWND, ::UINT, ::UINT_PTR, ::DWORD ){
			if( redrawTimer ){
				::KillTimer( nullptr, redrawTimer );
				redrawTimer = 0;
			}
			if( !redrawRequest && ::IsWindow(redrawWindow) ){
				::InvalidateRect( redrawWindow, nullptr, false );
#if defined(_DEBUG)
				::MessageBeep(MB_OK);
#endif
			}
		} );
	}

//----------------------------------------------------------------------------

	//	イベント応答
	void	OnEvents( ::HWND hwnd, ::UINT nEvent, ::LPARAM lParam ) 
	{
		//	EmEditorが起動した直後、またはプラグインを追加した
		if( nEvent & EVENT_CREATE ){
			hedingable = Editor_GetVersion( hwnd ) >= 14600;	// 14.6.0以降
			LoadOptions( hwnd );
		}

		//	カーソル位置が移動した
		if( nEvent & EVENT_CARET_MOVED ){
			ApplyMargin( hwnd );
		}

		//	アイドルになった
		if( nEvent & EVENT_IDLE ){
			if(	redrawRequest ){
				redrawRequest = false;
				DelayRedraw( hwnd );
			}
		}

		//	終了する
		if( nEvent & (EVENT_CLOSE | EVENT_CLOSE_FRAME) ){
			redrawRequest = false;
			if(	redrawTimer ){
				::KillTimer( nullptr, redrawTimer );
				redrawTimer = 0;
			}
		}
	}

	//	コマンド実行可能か？
	bool	CanCommand( ::HWND hwnd )
	{
		return	true;
	}

	//	チェックが付いた状態か？
	bool	IsChecked( ::HWND hwnd )
	{
		return	false;
	}

	//	コマンド実行
	void	OnCommand( ::HWND hwnd )
	{
		ShowProperty( hwnd );
	}

//----------------------------------------------------------------------------
#pragma region Property
	namespace Property
	{
		//	プロパティ
		::INT_PTR CALLBACK Proc( ::HWND dlg, ::UINT msg, ::WPARAM wParam, ::LPARAM lParam )
		{
			const int limit = 20; // マージン行数
			switch( msg ){

			//	初期化
			case WM_INITDIALOG:
				::SetWindowLongPtr(	dlg, DWLP_USER, lParam );
				::SetWindowText(	dlg, tools::GetString( IDS_NAME ).c_str() );
				::SetDlgItemText(	dlg, IDC_COMMENT, tools::GetString( IDS_COMMENT ).c_str() );

				::SendDlgItemMessage( dlg, IDC_MARGIN_UP,	CB_RESETCONTENT, 0, 0 );
				::SendDlgItemMessage( dlg, IDC_MARGIN_DOWN,	CB_RESETCONTENT, 0, 0 );
				for( auto i(0) ; i <= limit ; ++i ){
					wchar_t	str[64];
					::swprintf_s( str, L"%d", i );
					::SendDlgItemMessage( dlg, IDC_MARGIN_UP,	CB_ADDSTRING, 0, reinterpret_cast<::LPARAM>(str) );
					::SendDlgItemMessage( dlg, IDC_MARGIN_DOWN,	CB_ADDSTRING, 0, reinterpret_cast<::LPARAM>(str) );
				}
				::SendDlgItemMessage( dlg, IDC_MARGIN_UP,	CB_SETCURSEL, (std::min)( (std::max)( scrollMarginUp,	0 ), limit ), 0 );
				::SendDlgItemMessage( dlg, IDC_MARGIN_DOWN,	CB_SETCURSEL, (std::min)( (std::max)( scrollMarginDown,	0 ), limit ), 0 );
				::CheckDlgButton(	  dlg, IDC_MOUSE_CANCEL,	mouseCancel ? 1 : 0 );

				::SendDlgItemMessage( dlg, IDC_DELAY_SPIN, UDM_SETRANGE32, 1, 1000 );
				::SendDlgItemMessage( dlg, IDC_DELAY_SPIN, UDM_SETPOS32, 0, redrawDelay.first );
				::SendDlgItemMessage( dlg, IDC_DELAY, EM_LIMITTEXT, 4, 0 );
				::SetDlgItemInt(	  dlg, IDC_DELAY, redrawDelay.first, false );

				break;

			case WM_COMMAND:
				if( HIWORD(wParam) == BN_CLICKED ){
					switch( LOWORD(wParam) ){

					//	遅延時間をデフォルトに戻す
					case IDC_RESET_DELAY:{
						auto const value( GetDefaultRedrawDelay() );
						::SendDlgItemMessage( dlg, IDC_DELAY_SPIN, UDM_SETPOS32, 0, value );
						::SetDlgItemInt(	  dlg, IDC_DELAY, value, false );
						::SetFocus( ::GetDlgItem( dlg, IDC_DELAY ) );
						}return TRUE;
				
					//	新たな設定を保存し閉じる
					case IDOK:{
							::LRESULT value;

							::BOOL success(FALSE);
							auto new_delay( ::GetDlgItemInt( dlg, IDC_DELAY, &success, FALSE ) );
							if( !success ){
								::MessageBox( dlg, tools::GetString(IDS_ERROR_DELAY).c_str(), nullptr, MB_ICONERROR | MB_OK );
								return	true;
							}

							value = ::SendDlgItemMessage( dlg, IDC_MARGIN_UP,	CB_GETCURSEL, 0, 0 );
							if( value != CB_ERR ) scrollMarginUp	= (std::min)( (std::max)( static_cast<::INT32>(value), 0 ), limit );
						
							value = ::SendDlgItemMessage( dlg, IDC_MARGIN_DOWN,	CB_GETCURSEL, 0, 0 );
							if( value != CB_ERR ) scrollMarginDown	= (std::min)( (std::max)( static_cast<::INT32>(value), 0 ), limit );

							mouseCancel	= ::IsDlgButtonChecked( dlg, IDC_MOUSE_CANCEL ) != 0;

							new_delay = (std::min)( (std::max)( static_cast<::DWORD>(new_delay), 1ul ), MaxRedrawDelay );
							if( redrawDelay.first != new_delay || redrawDelay.second ){
								redrawDelay.first  = new_delay;
								redrawDelay.second = true;
							}
						}
						SaveOptions( reinterpret_cast<::HWND>(::GetWindowLongPtr( dlg, DWLP_USER )) );
						::EndDialog( dlg, IDOK );
						return TRUE;

					//	変更せずに閉じる
					case IDCANCEL:
						::EndDialog( dlg, IDCANCEL );
						return TRUE;
	
					}
				}
				break;

			case WM_NOTIFY:
				if( auto const hed = reinterpret_cast<::NMHDR const*>(lParam) ){
					if( hed->code == UDN_DELTAPOS ){
						auto const& dat( *reinterpret_cast<::NMUPDOWN const*>(lParam) );
						auto value( static_cast<int>(::GetDlgItemInt( dlg, IDC_DELAY, nullptr, FALSE )) );
						if( dat.iDelta > 0 ){
							value = (value + 10) / 10 * 10;
						}else if( dat.iDelta < 0 ){
							value = (value -  1) / 10 * 10;
						}else break;
						value = (std::min)( (std::max)( value, 1 ), static_cast<int>(MaxRedrawDelay) );
						::SetDlgItemInt( dlg, IDC_DELAY, value, false );
						return	TRUE;
					}
				}
				break;
			}
			return	FALSE;
		}
	}

	//	プロパティ開く
	//		hwnd	EmEditorウィンドウハンドル
	bool	ShowProperty( ::HWND em )
	{
		return	::DialogBoxParam( tools::Instance(), MAKEINTRESOURCE(IDD_PROPERTY), em, Property::Proc, reinterpret_cast<::LPARAM>(em) ) != -1;
	}

	//	プロパティあるか？
	bool	HasProperty()
	{
		return	true;
	}

#pragma endregion Property
//----------------------------------------------------------------------------
}

