﻿#include "stdafx.h"
#include "SplitBox.h"
#include "tools.h"
#include "option.h"
#include "customize.h"
#include "resource.h"

#include <sstream>
#include <ios>

#undef min
#undef max

namespace SplitBox
{
	//	オプションキー　幅
	wchar_t const*const option_width{ L"width" };

	//	幅下限値
	int	const width_lower_limit{ 8 };

	//	幅割合
	int	width_rate{ 7 };

//----------------------------------------------------------------------------

	//	プラグインの情報取得
	::LRESULT	GetPluginInfo(::HWND const em, ::UINT_PTR const flag)
	{
		switch (flag) {
		case EPGI_ALLOW_OPEN_SAME_GROUP:	return	TRUE;		//	ファイルを同じグループ内で開くことを許す場合に TRUE を返します。 
		case EPGI_ALLOW_MULTIPLE_INSTANCES:	return	TRUE;		//	プラグインが複数インスタンスをサポートする場合に TRUE を返します。プラグインが 2 個以上のフレームで同時に動作することが許される場合には、このメッセージは TRUE を返す必要があります。複数インスタンスが実行している間、グローバル変数は共有されることに注意してください。 
		case EPGI_MAX_EE_VERSION:			return	99 * 1000;	//	対応するもっとも新しい EmEditor のバージョン番号 * 1000 を返します。	とりあえず・・・
		case EPGI_MIN_EE_VERSION:			return	143 * 100;	//	対応するもっとも古い EmEditor のバージョン番号 * 1000 を返します。 
		case EPGI_SUPPORT_EE_PRO:			return	TRUE;		//	EmEditor Professional をサポートする場合に TRUE を返します。 
		case EPGI_SUPPORT_EE_STD:			return	FALSE;		//	EmEditor Standard をサポートする場合に TRUE を返します。 
		}
		return	0;
	}

//----------------------------------------------------------------------------

	struct unhooker_t
	{
		void	operator()(::HHOOK const hook) const
		{
			if (hook)
				::UnhookWindowsHookEx(hook);
		}
	};
	typedef std::unique_ptr<std::remove_pointer<::HHOOK>::type, unhooker_t> hook_ptr;

	struct window_destroyer_t
	{
		void	operator()(::HWND const wnd) const
		{
			if (wnd)
				::DestroyWindow(wnd);
		}
	};
	typedef std::unique_ptr<std::remove_pointer<::HWND>::type, window_destroyer_t> window_ptr;

//----------------------------------------------------------------------------

	//	タイマー
	class timer_t {
	public:
		timer_t() : id_{} {}
		~timer_t() { reset(); }
		void	reset() {
			if (id_) {
				::KillTimer(nullptr, id_);
				id_ = 0;
			}
		}
		void	reset(::UINT time, ::TIMERPROC procedure) {
			reset();
			id_ = ::SetTimer(nullptr, 0, time, procedure);
		}
	private:
		::UINT_PTR	id_;
		timer_t(timer_t const&) = delete;
		timer_t& operator=(timer_t const&) = delete;
	};

	//	マウスキャプチャ
	class capture_t {
	public:
		capture_t() : hwnd_{} {}
		capture_t(::HWND const hwnd) { operator()(hwnd); }
		~capture_t() { reset(); }
		void	reset() {
			if (hwnd_) {
				if (::GetCapture() == hwnd_)
					::SetCapture(nullptr);
				hwnd_ = nullptr;
			}
		}
		void	operator()(::HWND hwnd)
		{
			::SetCapture(hwnd_ = hwnd);
		}
	private:
		::HWND	hwnd_;
		capture_t(capture_t const&) = delete;
		capture_t& operator=(capture_t const&) = delete;
	};

//----------------------------------------------------------------------------

	//	ペイン進める、自動復元
	struct auto_pain_t
	{
		auto_pain_t(::HWND em) : em_(em) {
			Editor_ExecCommand(em_, EEID_NEXT_PANE);
		}
		~auto_pain_t() {
			Editor_ExecCommand(em_, EEID_PREV_PANE);
		}
	private:
		::HWND const em_;

		auto_pain_t(auto_pain_t&&) = delete;
		auto_pain_t(auto_pain_t const&) = delete;
		auto_pain_t& operator=(auto_pain_t&&) = delete;
		auto_pain_t& operator=(auto_pain_t const&) = delete;
	};

//----------------------------------------------------------------------------

	//	方向ID
	enum class orientation_id
	{
		horizontal,
		vertical
	};

	//	方向ごとの値
	template<typename T>
	struct orientation_t
	{
		T	horizontal;
		T	vertical;

		auto	operator[](orientation_id i) const->T const&;
		auto	operator[](orientation_id i)->T&;

		orientation_t() : horizontal{}, vertical{} {}
		template<typename H, typename V>
		orientation_t(H&& h, V&& v) : horizontal{ std::forward<H>(h) }, vertical{ std::forward<V>(v) } {}
	};

	template<typename T>
	inline auto	orientation_t<T>::operator[](orientation_id i) -> T&
	{
		switch (i) {
		case orientation_id::horizontal: return	horizontal;
		case orientation_id::vertical:	 return	vertical;
		default:						 throw	std::exception("unknown orientation");
		}
	}
	template<typename T>
	inline auto	orientation_t<T>::operator[](orientation_id i) const-> T const&
	{
		return	const_cast<orientation_t<T>*>(this)->operator[](i);
	}

//----------------------------------------------------------------------------

	typedef std::vector<::HWND> views_t;

	//	分割状態
	struct split_state_t : private orientation_t<bool>
	{
		using orientation_t<bool>::horizontal;	//	水平分割状態	─
		using orientation_t<bool>::vertical;	//	垂直分割状態	│

		split_state_t() {}
		explicit split_state_t(::HWND em);	// キャプチャ

		bool	operator==(split_state_t const& r) const { return horizontal == r.horizontal && vertical == r.vertical; }
		bool	operator!=(split_state_t const& r) const { return !operator==(r); }
	};

	//	分割開始ドラッグ
	struct split_drag_t
	{
		split_drag_t(::HWND hwnd, ::HWND em, orientation_id orientation, int x, int y);
		void	on_move(::WPARAM wParam, int x, int y);
	private:
		capture_t				capture_;
		::HWND			const	em_;
		::POINT			const	gap_;
		::POINT			const	first_;
		orientation_id	const	orientation_;
		bool					active_;

		void	split(::WPARAM wParam, int x, int y);
	};
	typedef std::unique_ptr<split_drag_t> split_drag_ptr;

	//	分割ボックス
	struct split_boxes_t;
	struct split_box_t
	{
		explicit split_box_t(orientation_id orientation__) : orientation_{ orientation__ }, view_{}, vscroll_{}, hscroll_{} {}
		static auto	find_scrollbar(::HWND view)->orientation_t<::HWND>;

		void	reset();
		void	reset(::HWND view);
		void	reset(split_box_t const& r);

		void	show(int x, int y, int w, int h);
		void	hide();
	private:
		orientation_id const	orientation_;
	public:
		::HWND					view_;
		::HWND					vscroll_;
		::HWND					hscroll_;
	private:
		window_ptr				box_;
		split_drag_ptr			drag_;
		static ::LRESULT CALLBACK	on_message(::HWND hwnd, ::UINT msg, ::WPARAM wParam, ::LPARAM lParam);
		friend split_boxes_t;
	};
	struct split_boxes_t : private orientation_t<split_box_t>
	{
		using	orientation_t<split_box_t>::vertical;
		using	orientation_t<split_box_t>::horizontal;

		split_boxes_t() : orientation_t<split_box_t>(orientation_id::horizontal, orientation_id::vertical) {}

		void	reset(views_t const& views, bool show_scroll_only_active);
		void	show(split_state_t const& split_state);

		void	reset() { horizontal.reset(); vertical.reset(); }
		void	hide() { horizontal.hide(); vertical.hide(); }

		bool	is_resize_target(::HWND hwnd) const {
			return hwnd == vertical.view_ || hwnd == horizontal.view_;
		}
		bool	is_visible_target(::HWND hwnd) const {
			return hwnd == vertical.vscroll_ || hwnd == horizontal.hscroll_;
		}
	};

	enum request_t
	{
		light,
		middle,
		heavy
	};

	//	分割ボックス制御
	class controller_t {
	public:
		controller_t();
		~controller_t() { finalize(); }

		void	initialize(::HWND em);
		void	finalize();
		void	update(::HWND em);
		void	request(request_t weight);
		void	redraw(::HWND em, request_t weight = middle);
	private:
		bool			active_;
		hook_ptr		hook_;
		split_state_t	split_state_;
		::HWND			em_;
		::HWND			frame_;
		views_t			views_;
		split_boxes_t	split_box_;
		timer_t			timer_;
		request_t		weight_;
		bool			show_scroll_only_active_;

		void	drop_ones_request();
		void	do_request(request_t weight);
		void	raw_update(::HWND em);
		static ::LRESULT CALLBACK on_hook(int nCode, ::WPARAM wParam, ::LPARAM lParam);
	};
	thread_local controller_t	controller;

//----------------------------------------------------------------------------

	//	分割状態キャプチャ
	split_state_t::split_state_t(::HWND const em)
	{
		::BOOL	horizontal_{};
		::BOOL	vertical_{};
		Editor_QueryStatus(em, EEID_WINDOW_SPLIT_HORZ_TOGGLE, &horizontal_);
		Editor_QueryStatus(em, EEID_WINDOW_SPLIT_VERT_TOGGLE, &vertical_);
		horizontal = !!horizontal_;
		vertical = !!vertical_;
	}

//----------------------------------------------------------------------------

	//	分割ボックスを取り除き、ビューへの参照もやめる
	void	split_box_t::reset()
	{
		view_ = nullptr;
		vscroll_ = nullptr;
		hscroll_ = nullptr;
		box_.reset();
	}

	auto	split_box_t::find_scrollbar(::HWND view)->orientation_t<::HWND>
	{
		orientation_t<::HWND>	scrollbar;
		::EnumChildWindows(view,
			[](::HWND wnd, ::LPARAM lParam) {
			auto& scrollbar{ *reinterpret_cast<orientation_t<::HWND>*>(lParam) };
			auto const style{ ::GetWindowLongPtr(wnd, GWL_STYLE) };
			if (style & 0x04) {
				if (tools::IsClassedWindow(wnd, L"EmEditorScrollBar")) {
					((style & 0x01) ? scrollbar.vertical : scrollbar.horizontal) = wnd;
					if (scrollbar.vertical && scrollbar.horizontal) return FALSE;
				}
			}
			return TRUE;
		}, reinterpret_cast<::LPARAM>(&scrollbar));
		return scrollbar;
	}

	//	ビューを参照し、スクロールバーとかの情報も取得しておく
	void	split_box_t::reset(::HWND const view)
	{
		reset();
		view_ = view;
		auto const scrollbar(find_scrollbar(view));
		vscroll_ = scrollbar.vertical;
		hscroll_ = scrollbar.horizontal;
	}

	//	他のをもとにリセットする
	void	split_box_t::reset(split_box_t const& r)
	{
		if (&r != this) {
			view_ = r.view_;
			vscroll_ = r.vscroll_;
			hscroll_ = r.hscroll_;
			box_.reset();
		}
	}

	//	分割ボックス表示
	void	split_box_t::show(int const x, int const y, int const w, int const h)
	{
		if (box_) {
			//	あるなら指定場所に表示
			::SetWindowPos(box_.get(), nullptr, x, y, w, h, SWP_NOZORDER | SWP_SHOWWINDOW);
		}
		else {
			//	分割ボックスのウィンドウクラス
			static const auto register_class{ [](bool const vertical) {
				WNDCLASSEXW wc{};
				wc.cbSize = sizeof(WNDCLASSEX);
				wc.style = CS_DBLCLKS | CS_NOCLOSE;
				wc.hInstance = tools::Instance();
				wc.hCursor = ::LoadCursor(nullptr, vertical ? IDC_SIZENS : IDC_SIZEWE);
				wc.hbrBackground = ::GetSysColorBrush(COLOR_3DFACE);
				wc.lpszClassName = vertical ? L"VERTICAL_SPLITTER" : L"HORIZONTAL_SPLITTER";
				wc.lpfnWndProc = on_message;
				return	::RegisterClassExW(&wc);
			} };
			static const orientation_t<::ATOM> window_class(register_class(false), register_class(true));

			//	ないので新規作成
			box_.reset(::CreateWindowExW(
				WS_EX_DLGMODALFRAME,
				MAKEINTATOM(window_class[orientation_]), L"",
				WS_CHILD | WS_VISIBLE,
				x, y, w, h, view_,
				nullptr, tools::Instance(), reinterpret_cast<::LPVOID>(this)));
		}
	}

	//	分割ボックス非表示
	void	split_box_t::hide()
	{
		if (box_)
			::ShowWindow(box_.get(), SW_HIDE);
	}

	//	分割ボックスへのメッセージ応答
	::LRESULT CALLBACK	split_box_t::on_message(::HWND const hwnd, ::UINT const msg, ::WPARAM const wParam, ::LPARAM const lParam)
	{
		switch (msg) {
		case WM_CREATE:
			if (auto const prm = reinterpret_cast<::CREATESTRUCT const*>(lParam))
				::SetWindowLongPtr(hwnd, GWLP_USERDATA, reinterpret_cast<::LONG_PTR>(prm->lpCreateParams));
			return TRUE;

		case WM_LBUTTONDBLCLK:
			//	分割状態変更
			if (auto const self = reinterpret_cast<split_box_t const*>(::GetWindowLongPtr(hwnd, GWLP_USERDATA))) {
				split_state_t const state(self->view_);
				static const orientation_t<int> command{ EEID_WINDOW_SPLIT_VERT_FIX, EEID_WINDOW_SPLIT_HORZ_FIX };
				Editor_ExecCommand(self->view_, command[self->orientation_]);
			}
			return TRUE;

		case WM_LBUTTONDOWN:
			//	ドラッグ開始
			if (auto const self = reinterpret_cast<split_box_t*>(::GetWindowLongPtr(hwnd, GWLP_USERDATA))) {
				self->drag_ = std::make_unique<split_drag_t>(hwnd, self->view_, self->orientation_, static_cast<short>(LOWORD(lParam)), static_cast<short>(HIWORD(lParam)));
				return TRUE;
			}
			break;
		case WM_MOUSEMOVE:
			//	マウス移動応答
			if (auto const self = reinterpret_cast<split_box_t*>(::GetWindowLongPtr(hwnd, GWLP_USERDATA)))
				if (self->drag_) self->drag_->on_move(wParam, static_cast<short>(LOWORD(lParam)), static_cast<short>(HIWORD(lParam)));
			break;
		case WM_LBUTTONUP:
			//	ドラッグ終了
			if (auto const self = reinterpret_cast<split_box_t*>(::GetWindowLongPtr(hwnd, GWLP_USERDATA))) {
				self->drag_.reset();
				return TRUE;
			}
			break;
		}
		return  ::DefWindowProc(hwnd, msg, wParam, lParam);
	}

//----------------------------------------------------------------------------

	//	分割開始ドラッグ
	split_drag_t::split_drag_t(::HWND const hwnd, ::HWND const em, orientation_id orientation, int x, int y) :
		capture_{ hwnd }, em_{ em },
		gap_{ ::GetSystemMetrics(SM_CXDRAG), ::GetSystemMetrics(SM_CYDRAG) }, first_{ x, y },
		orientation_{ orientation }, active_{ true }
	{
	}

	//	マウスカーソルが移動した
	void	split_drag_t::on_move(::WPARAM wParam, int x, int y)
	{
		if (active_) {
			if (std::abs(first_.x - x) > gap_.x || std::abs(first_.y - y) > gap_.y) {
				active_ = false;
				capture_.reset();
				split(wParam, x, y);
			}
		}
	}

	//	分割
	void	split_drag_t::split(::WPARAM wParam, int x, int y)
	{
		::POINT pos;
		::GetCursorPos(&pos);

		split_state_t const state(em_);
		if (!state.horizontal && !state.vertical) {
			static const orientation_t<int> command{ EEID_WINDOW_SPLIT_VERT_FIX, EEID_WINDOW_SPLIT_HORZ_FIX };
			Editor_ExecCommand(em_, command[orientation_]);

			POINT_PTR scroll{}, caret{};
			Editor_GetCaretPos(em_, POS_VIEW, &caret);
			Editor_GetScrollPos(em_, &scroll);
			auto_pain_t const auto_pain(em_);
			if (auto wnd = GetFocus()) {
				if (auto view = tools::FindViewWindow(wnd)) {
					Editor_SetCaretPos(view, POS_VIEW, &caret);
					Editor_SetScrollPos(view, &scroll);
				}
			}
		}

		static const orientation_t<int> command{ EEID_WINDOW_SPLIT_VERT, EEID_WINDOW_SPLIT_HORZ };
		Editor_ExecCommand(em_, command[orientation_]);

		::SetCursorPos(pos.x, pos.y);
		if (auto const hwnd = ::GetCapture()) {
			::ScreenToClient(hwnd, &pos);
			::PostMessage(hwnd, WM_MOUSEMOVE, wParam, MAKELONG(pos.x, pos.y));
		}
	}

//----------------------------------------------------------------------------

	//	分割ボックスリセット
	void	split_boxes_t::reset(views_t const& views, bool const show_scroll_only_active)
	{
		reset();

		//	対象ビュー検索
		if (views.size() == 1u) {
			vertical.view_ = horizontal.view_ = views.back();
		}
		else {
			::HWND has{};
			if (show_scroll_only_active &&
				std::count_if(views.begin(), views.end(),
					[&](::HWND const view) {
				auto const scrollbar(split_box_t::find_scrollbar(view));
				if ((scrollbar.vertical   && ::IsWindowVisible(scrollbar.vertical)) ||
					(scrollbar.horizontal && ::IsWindowVisible(scrollbar.horizontal))) {
					has = view;
					return true;
				}
				return false;
			}) == 1u) {
				assert(has);
				vertical.view_ = horizontal.view_ = has;
			}
			else {
				for (auto view : views) {
					if (vertical.view_ || horizontal.view_) {
						::RECT area, varea, harea;
						GetWindowRect(view, &area);
						GetWindowRect(vertical.view_, &varea);
						GetWindowRect(horizontal.view_, &harea);
						if (area.top < varea.top || (area.top == varea.top && area.left > varea.left)) vertical.view_ = view;
						if (area.top > harea.top || (area.top == harea.top && area.left > harea.left)) horizontal.view_ = view;
					}
					else
						vertical.view_ = horizontal.view_ = view;
				}
			}
		}

		//	スクロールバー検索
		vertical.reset(vertical.view_);
		if (vertical.view_ != horizontal.view_)
			horizontal.reset(horizontal.view_);
		else
			horizontal.reset(vertical);
	}

	//	幅算出
	inline int compute_width(int const bar_width, int const rate)
	{
		return bar_width * rate / 10;
	}
	inline int compute_width(int const bar_width)
	{
		return compute_width(bar_width,width_rate);
	}

	//	分割ボックス表示
	void	split_boxes_t::show(split_state_t const& split_state)
	{
		auto const use_vscroll{ !split_state.horizontal && vertical  .vscroll_ && ::IsWindowVisible(vertical  .vscroll_) };
		auto const use_hscroll{ !split_state.vertical   && horizontal.hscroll_ && ::IsWindowVisible(horizontal.hscroll_) };

		if (!use_hscroll)
			horizontal.box_.reset();
		if (!use_vscroll)
			vertical.box_.reset();

		if (use_vscroll || use_hscroll) {
			auto	both{ use_vscroll && use_hscroll };
			if (!both) {
				if (use_vscroll)
					both = vertical.hscroll_ && ::IsWindowVisible(vertical.hscroll_);
				else if (use_hscroll)
					both = horizontal.vscroll_ && ::IsWindowVisible(horizontal.vscroll_);
			}

			int const vwidth { ::GetSystemMetrics(SM_CXVSCROLL) };
			int const hheight{ ::GetSystemMetrics(SM_CYHSCROLL) };
			int bwidth{}, bheight{};
			if (both) {
				bwidth = vwidth;
				bheight = hheight;
			}
			if (use_vscroll) {
				::RECT	area;
				if (::GetClientRect(vertical.view_, &area)) {
					auto const h{ (std::max)(compute_width(hheight), width_lower_limit) };
					auto const x{ area.right - vwidth };
					vertical.show(x, 0, vwidth, h);
					::SetWindowPos(vertical.vscroll_, nullptr, x, h, vwidth, area.bottom - h - bheight, SWP_NOZORDER);
					::InvalidateRect(vertical.vscroll_, nullptr, false);
				}
				else
					vertical.box_.reset();
			}
			if (use_hscroll) {
				::RECT	area;
				if (::GetClientRect(horizontal.view_, &area)) {
					auto const w{ (std::max)(compute_width(vwidth), width_lower_limit) };
					auto const x{ area.right - w - bwidth };
					auto const y{ area.bottom - hheight };
					horizontal.show(x, y, w, hheight);
					::SetWindowPos(horizontal.hscroll_, nullptr, 0, y, x, hheight, SWP_NOZORDER);
					::InvalidateRect(horizontal.hscroll_, nullptr, false);
				}
				else
					horizontal.box_.reset();
			}
		}
	}

//----------------------------------------------------------------------------

	//	分割ボックス
	controller_t::controller_t() : active_{}, em_{}, frame_{}, weight_{}, show_scroll_only_active_{ true }
	{
	}

	//	初期化
	void	controller_t::initialize(::HWND const em)
	{
		if (!active_) {
			if (!hook_)
				hook_.reset(::SetWindowsHookEx(WH_CALLWNDPROCRET, on_hook, 0, ::GetCurrentThreadId()));
			active_ = true;
			frame_ = tools::FindFrameWindow(em);
			option::Reader const reader{ em };
			width_rate = reader.Int32(option_width, width_rate);
			request(heavy);
		}
	}

	//	後始末
	void	controller_t::finalize()
	{
		drop_ones_request();
		views_.clear();
		split_box_.reset();
		frame_ = nullptr;
		em_ = nullptr;
		active_ = false;
		hook_.reset();
	}

	//	分割ボックス周りを再描画させる
	void	controller_t::redraw(::HWND em, request_t weight)
	{
		auto const frame(tools::FindFrameWindow(em));
		::EnumChildWindows(frame,
			[](::HWND hwnd, ::LPARAM lParam)->::BOOL {
			if (tools::IsViewWindow(hwnd)) {
				::RECT area;
				if (::GetClientRect(hwnd, &area))
					::PostMessage(hwnd, WM_SIZE, SIZE_RESTORED, MAKELONG(area.right - area.left, area.bottom - area.top));
			}
			return TRUE;
		}, 0);
		request(weight);
	}

	//	フレーム変更を処理する　おそらく過剰
	inline void	controller_t::update(::HWND const em)
	{
		if (active_ && em_ != em) raw_update(em);
	}
	void	controller_t::raw_update(::HWND const em)
	{
		em_ = em;
		if (auto const frame = tools::FindFrameWindow(em)) {
			if (frame_ != frame) {
				frame_ = frame;
				split_state_ = split_state_t(frame_);
				request(middle);
			}
		}
		else if (frame_) {
			if (!::IsWindow(frame_)) {
				views_.clear();
				split_box_.reset();
				frame_ = nullptr;
				request(light);
			}
		}
	}

	//	分割ボックス更新リクエストを出す
	void	controller_t::request(request_t const weight)
	{
		timer_.reset();
		split_box_.hide();

		if (weight_ < weight)
			weight_ = weight;
		timer_.reset(250,
			[](HWND, UINT, UINT_PTR, DWORD) {
			auto const weight(controller.weight_);
			controller.drop_ones_request();
			controller.do_request(weight);
		});
	}

	//	分割ボックス更新リクエストを取り下げる
	void	controller_t::drop_ones_request()
	{
		timer_.reset();
		weight_ = light;
	}

	//	分割ボックス更新リクエストを片づける
	void	controller_t::do_request(request_t const weight)
	{
		if (weight >= heavy) {
			//	今の文書の設定を読み取る
			try {
				wchar_t	config[MAX_CONFIG_NAME]{};
				Editor_DocGetConfigW(frame_, -1, config);
				customize::customize_info_t const inf{ frame_, config };
				show_scroll_only_active_ = inf(&CCustomizeInfo::m_bShowScrollOnlyActive, true);
			}
			catch (...) {
				show_scroll_only_active_ = true;
			}
		}

		if (weight >= middle) {
			//	ビュー収集
			views_.clear();
			::EnumChildWindows(frame_,
				[](::HWND wnd, ::LPARAM lParam) {
				if (tools::IsViewWindow(wnd))
					reinterpret_cast<views_t*>(lParam)->emplace_back(wnd);
				return TRUE;
			}, reinterpret_cast<::LPARAM>(&views_));
			std::sort(views_.begin(), views_.end());

			//	分割ボックスリセット
			split_box_.reset(views_, show_scroll_only_active_);
		}

		//	分割ボックス表示
		split_state_ = split_state_t(frame_);
		split_box_.show(split_state_);

		//	EmEditor本体にサムの長さがおかしくなることがあるバグがあるので、スクロールバーを再描画させ、正しい長さで表示されるようにする
		POINT_PTR scroll{};
		Editor_GetScrollPos(frame_, &scroll);
		Editor_SetScrollPos(frame_, &scroll);
	}

	//	フック応答
	::LRESULT CALLBACK controller_t::on_hook(int const nCode, ::WPARAM const wParam, ::LPARAM const lParam)
	{
		auto const hook{ controller.hook_.get() };
		if (nCode == HC_ACTION) {
			auto const& msg(*reinterpret_cast<::CWPRETSTRUCT const*>(lParam));
			switch (msg.message) {
			case WM_SIZE:
				//	分割状態が前と変わったら、全更新
				if (std::binary_search(controller.views_.begin(), controller.views_.end(), msg.hwnd)) {
					split_state_t state(controller.frame_);
					if (controller.split_state_ != state) {
						controller.split_state_ = state;
						controller.request(middle);
						break;
					}
				}
				//	分割状態が前と同じなら、リサイズ系
				if (controller.split_box_.is_resize_target(msg.hwnd))
					controller.request(light);
				break;
			case WM_SHOWWINDOW:
				//	自動でスクロールバーの表示状態が変わる設定の時、リサイズ系
				if (controller.split_box_.is_visible_target(msg.hwnd))
					controller.request(light);
				break;
			case WM_SETFOCUS:
				//	「アクティブペインの中だけバーを表示」対策
				if (controller.show_scroll_only_active_)
					if (std::binary_search(controller.views_.begin(), controller.views_.end(), msg.hwnd))
						controller.request(middle);
				break;
			case WM_KILLFOCUS:
				//	「アクティブペインの中だけバーを表示」対策
				if (controller.show_scroll_only_active_)
					if (std::binary_search(controller.views_.begin(), controller.views_.end(), msg.hwnd))
						controller.redraw(msg.hwnd);
				break;
			}
		}
		return	::CallNextHookEx(hook, nCode, wParam, lParam);
	}

//----------------------------------------------------------------------------

	//	イベント応答
	void	OnEvents(::HWND const em, ::UINT const nEvent, ::LPARAM const lParam)
	{
		::UINT const flags{ ~static_cast<::UINT>(EVENT_CARET_MOVED | EVENT_CHANGE | EVENT_CHAR | EVENT_IDLE | EVENT_KILL_FOCUS | EVENT_MODIFIED | EVENT_SCROLL | EVENT_SEL_CHANGED | EVENT_SET_FOCUS | EVENT_HISTORY) };
		if (nEvent & flags)
			controller.update(em);

		//	エディタフレーム作成時に作成
		if (nEvent & EVENT_CREATE_FRAME)
			controller.initialize(em);

		//	設定が変わった
		if (nEvent & (EVENT_CONFIG_CHANGED | EVENT_FILE_OPENED))
			controller.request(heavy);

		//	設定が変わった/アクティブドキュメントが変わった
		if (nEvent & (EVENT_CONFIG_CHANGED | EVENT_DOC_SEL_CHANGED))
			controller.redraw(em);

		//	エディタフレーム閉じる/エディタ閉じる/アンインストールなので破棄
		if (nEvent & (EVENT_CLOSE | EVENT_CLOSE_FRAME))
			controller.finalize();
	}

	//	コマンド実行
	void	OnCommand(::HWND const em)
	{
		controller.redraw(em, heavy);
	}


	//	コマンド実行可能か？
	bool	CanCommand(::HWND const em)
	{
		return	true;
	}

	//	チェックが付いた状態か？
	bool	IsChecked(::HWND const em)
	{
		return	false;
	}

//----------------------------------------------------------------------------
#pragma region Property

	::INT_PTR CALLBACK property_procedure(::HWND const dlg, ::UINT const msg, ::WPARAM const wParam, ::LPARAM const lParam)
	{
		switch (msg) {
		case WM_INITDIALOG:
			if (auto const ctrl = ::GetDlgItem(dlg, IDC_WIDTH)) {
				auto const base_width{ (std::max)(::GetSystemMetrics(SM_CXVSCROLL), ::GetSystemMetrics(SM_CYHSCROLL)) };
				auto const threshold{ (std::min)(width_lower_limit, compute_width(base_width)) };
				auto add{ [&](int const rate, wchar_t const*const str) {
					if (threshold <= compute_width(base_width,rate)) {
						auto const index{ ::SendMessage(ctrl, CB_ADDSTRING, 0, reinterpret_cast<::LPARAM>(str)) };
						if (index != CB_ERR) {
							::SendMessage(ctrl, CB_SETITEMDATA, index, rate);
							if (rate == width_rate)
								::SendMessage(ctrl, CB_SETCURSEL, index, 0);
						}
					}
				} };
				for (auto id{ IDS_WIDTH_BEGIN };id <= IDS_WIDTH_END;++id) {
					auto const str{ tools::GetString(id) };
					if (!str.empty()) add(id - IDS_WIDTH_BEGIN, str.c_str());
				}
			}
			return TRUE;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
			case IDOK:
				if (auto const ctrl = ::GetDlgItem(dlg, IDC_WIDTH)) {
					auto const index{ ::SendMessage(ctrl, CB_GETCURSEL, 0, 0) };
					if (index != CB_ERR) {
						auto const rate{ ::SendMessage(ctrl, CB_GETITEMDATA, index, 0) };
						if (rate != CB_ERR) {
							width_rate = static_cast<int>(rate);
							option::Writer writer{ ::GetParent(dlg) };
							writer.Int32(option_width, width_rate);
							controller.request(light);
						}
					}
				}
				EndDialog(dlg, IDOK);
				return TRUE;
			case IDCANCEL:
				EndDialog(dlg, IDCANCEL);
				return TRUE;
			}
			break;
		}
		return FALSE;
	}
	//	プロパティ開く
	bool	ShowProperty(::HWND const em)
	{
		return DialogBox(tools::Instance(), MAKEINTRESOURCE(IDD_PROPERTY), em, property_procedure) > 0;
	}

	//	プロパティあるか？
	bool	HasProperty()
	{
		return	true;
	}

#pragma endregion Property
//----------------------------------------------------------------------------
}

