/* eslint-disable no-console */
const path = require("path");
const { spawn } = require("child_process");
const Datastore = require("nedb-promises");
const {
  pathExists,
  rmdir,
  mkdir,
  copy,
  rename,
  stat,
  readdir,
  rm,
} = require("fs-extra");

const EXTEND_DB = "extend.db";

class UBPluginInstaller {
  parseArguments() {
    this._mode = "install";
    process.argv.forEach((val, index) => {
      if (val === "--7z") {
        this._7z = process.argv[index + 1];
      } else if (val === "--project") {
        this._project = process.argv[index + 1];
      } else if (val === "--plugin") {
        this._plugin = process.argv[index + 1];
      } else if (val === "--install") {
        this._mode = "install";
      } else if (val === "--uninstall") {
        this._mode = "uninstall";
      } else if (val === "--list") {
        this._mode = "list";
      } else if (val === "--name") {
        this._name = process.argv[index + 1];
      }
    });
  }

  async checkArguments(needExtract, needPlugin, needName) {
    if (!this._project) {
      console.info("项目路径未提供\r\n");
      return false;
    }
    const existsProject = await pathExists(this._project);
    if (!existsProject) {
      console.info("项目不存在\r\n");
      return false;
    }

    if (needExtract) {
      if (!this._7z) {
        console.info("解压工具路径未提供\r\n");
        return false;
      }
      const exists7z = await pathExists(this._7z);
      if (!exists7z) {
        console.info("解压工具不存在\r\n");
        return false;
      }
    }

    if (needPlugin)
    {
      if (!this._plugin) {
        console.info("命令库文件路径未提供\r\n");
        return false;
      }
      const existsPlugin = await pathExists(this._plugin);
      if (!existsPlugin) {
        console.info("命令库文件不存在\r\n");
        return false;
      }
    }

    if (needName) {
      if (!this._name) {
        console.info("命令库名称未提供\r\n");
        return false;
      }
    }
    return true;
  }

  decompress(src, dest) {
    return new Promise((resolve, reject) => {
      try {
        const child = spawn(this._7z, ["x", src, `-o${dest}`]);
        child.on("close", (code) => {
          if (code !== 0) {
            reject(new Error("命令库文件文件解压失败\r\n"));
          } else {
            resolve("");
          }
        });
        child.stderr.on("data", (data) => {
          reject(new Error("命令库文件文件解压失败\r\n"));
        });
      } catch (error) {
        reject(error);
      }
    });
  }

  async withExtractPlugin(callback) {
    const tempPath = path.join(this._project, "temp");
    try {
      await this.decompress(this._plugin, tempPath);
      const exists = await pathExists(tempPath);
      if (!exists) {
        console.info("命令库文件解压失败\r\n");
        return false;
      }
      return await callback(tempPath);
    } catch (error) {
      console.info(`${error.message}\r\n`);
      return false;
    } finally {
      await rmdir(tempPath, { recursive: true });
    }
  }

  convertCommand(command) {
    const { sourceTemplate } = command;
    const extendCommand = {
      args: [],
      targetAction: "",
      commandDesc: command.visualTemplate,
      commandName: command.caption,
      function: command.namespace + "." + command.functionName,
      srcCode: sourceTemplate,
      translateType: command.result ? 3 : 32, // 3代表有返回值，32代表没有
      visible: command.visible,
    };
    const args = [];
    const requireArgs = command.groups[0];
    if (requireArgs) {
      const parameters = requireArgs.parameters;
      for (const param of parameters) {
        const item = {
          dataType: "variable",
          defaultValue: param.default || '""',
          desc: param.desc,
          name: param.name,
          propName: param.caption,
        };
        args.push(item);
      }
    }
    const optionsArgs = command.groups[1];
    if (optionsArgs) {
      const parameters = optionsArgs.parameters;
      const options = [];
      const item = {
        dataType: "variable",
        defaultValue: "",
        desc: "",
        name: "optionArgs",
        propName: optionsArgs.caption,
        options: [],
      };
      for (const param of parameters) {
        const item = {
          dataType: "variable",
          defaultValue: param.default || '""',
          desc: param.desc,
          name: param.name,
          propName: param.caption,
        };
        options.push(item);
      }
      item.options = options;
      args.push(item);
    }
    const { result } = command;
    if (result) {
      const item = {
        name: "result",
        propName: "输出到",
        desc: result.desc || "",
        dataType: "variable",
        defaultValue: result.default || '""',
      };
      args.unshift(item);
    }
    extendCommand.args = args;
    return extendCommand;
  }

  async updateToDb(config) {
    const { commands } = config;
    const dbPath = path.join(this._project, "extend", EXTEND_DB);
    const db = new Datastore({ filename: dbPath });
    try {
      const commandsData = commands.map((command) =>
        this.convertCommand(command)
      );
      const record = {
        bigIcon: "",
        className: config.caption,
        creatorType: 0,
        name: config.name,
        isLib: 1,
        items: commandsData,
        namespace: config.namespace,
        version: config.version,
        author: config.author,
      };
      const result = await db.count({ name: config.name });
      if (result > 0) {
        console.info(`已存在命令库[${config.name}]，正在更新\r\n`);
        await db.update({ name: config.name }, record);
      } else {
        await db.insert(record);
      }
      return true;
    } catch (error) {
      console.info(`${error.message}\r\n`);
      return false;
    }
  }

  async copyFiles(config, pluginFolder) {
    const folder = {
      python: "python",
      dotnet: "DotNet",
      "bot-script": "lib",
    };
    try {
      const { type, namespace } = config;
      const configPath = path.join(pluginFolder, "package.json");
      const newConfigPath = path.join(
        pluginFolder,
        `${namespace}.package.json`
      );
      if (type === "bot-script") {
        const dirs = ["extend", "res", "ui"];
        for (const dir of dirs) {
          const oldPath = path.join(pluginFolder, dir);
          const exists = await pathExists(oldPath);
          if (!exists) {
            continue;
          }
          const stats = await stat(oldPath);
          if (stats.isDirectory()) {
            const newPath = path.join(pluginFolder, namespace + "." + dir);
            await rename(oldPath, newPath);
          }
        }
      }
      await rename(configPath, newConfigPath);
      const dest = path.join(
        this._project,
        "extend",
        folder[type.toLocaleLowerCase()]
      );
      const exists = await pathExists(dest);
      if (!exists) {
        await mkdir(dest, { recursive: true });
      }
      await copy(pluginFolder, dest, { recursive: true });
      return true;
    } catch (error) {
      console.info(`${error.message}\r\n`);
      return false;
    }
  }

  async install() {
    let success = await this.checkArguments(true, true, false);
    if (!success) {
      process.exit(-1);
    }
    let name = "";
    success = await this.withExtractPlugin(async (pluginFolder) => {
      console.info("命令库文件解压完成\r\n");
      const configPath = path.join(pluginFolder, "package.json");
      const exists = await pathExists(configPath);
      if (!exists) {
        console.info("命令库文件配置文件不存在\r\n");
        return false;
      }
      const config = require(configPath);
      const { commands } = config;
      name = config.name;
      if (!commands) {
        console.info("命令库文件命令信息不存在\r\n");
        return false;
      }
      console.info("命令库文件检查完成\r\n");
      let success = await this.updateToDb(config);
      if (!success) {
        console.info("命令库文件命令信息转换失败\r\n");
        return false;
      }
      success = await this.copyFiles(config, pluginFolder);
      if (!success) {
        console.info("命令库文件复制失败\r\n");
        return false;
      }
      return true;
    });
    if (!success) {
      console.info(name ? `命令库[${name}]安装失败\r\n`: "命令库安装失败\r\n");
      process.exit(-1);
    } else {
      console.info(`命令库[${name}]安装成功\r\n`);
      process.exit(0);
    }
  }

  async removeFiles(namespace) {
    const folder = {
      python: "python",
      dotnet: "DotNet",
      "bot-script": "lib",
    };
    for (const type of Object.keys(folder)) {
      const dest = path.join(this._project, "extend", folder[type]);
      const configPath = path.join(dest, `${namespace}.package.json`);
      const exists = await pathExists(configPath);
      if (!exists) {
        continue;
      }
      const list = await readdir(dest, { withFileTypes: true });
      for (const item of list) {
        const ext = path.extname(item.name);
        const base = path.basename(item.name, ext);
        if (base !== namespace) {
          continue;
        }
        const target = path.join(dest, item.name);
        await rm(target, { recursive: true });
        console.info(`删除[${item.name}]成功\r\n`);
      }
      await rm(configPath);
    }
  }

  async removeFromDb(name, callback) {
    const dbPath = path.join(this._project, "extend", EXTEND_DB);
    const exists = await pathExists(dbPath);
    if (!exists) {
      console.info("命令库文件不存在\r\n");
      process.exit(-1);
    }
    const db = new Datastore({ filename: dbPath });
    const [record] = await db.find({ name });
    if (!record) {
      return false;
    }
    try {
      await callback(record.namespace);
    } catch (error) {
      console.info(`${error.message}\r\n`);
      console.info(`删除命令库文件失败，可能文件被占用，请尝试退出Creator后重试\r\n`);
      return false;
    }
    const count = await db.remove({ name });
    return count > 0;
  }

  async uninstall(name) {
    let success = await this.checkArguments(false, false, true);
    if (!success) {
      process.exit(-1);
    }
    try {
      const success = await this.removeFromDb(name, (namespace) => this.removeFiles(namespace));
      if (success) {
        console.info(`命令库[${name}]卸载成功\r\n`);
      } else {
        console.info(`未找到命令库[${name}]\r\n`);
      }
      process.exit(0);
    } catch (error) {
      console.info(`${error.message}\r\n`);
      console.info(`命令库[${name}]卸载失败\r\n`);
      process.exit(-1);
    }
  }

  async list() {
    let success = await this.checkArguments(false, false, false);
    if (!success) {
      process.exit(-1);
    }
    try {
      const dbPath = path.join(this._project, "extend", EXTEND_DB);
      const exists = await pathExists(dbPath);
      if (!exists) {
        console.info([]);
        return;
      }
      const db = new Datastore({ filename: dbPath });
      const result = await db.find({});
      console.info(JSON.stringify(result, null, 2));
      process.exit(0);
    } catch (error) {
      console.info(`${error.message}\r\n`);
      process.exit(-1);
    }
  }

  async run() {
    this.parseArguments();
    switch (this._mode) {
      case "install":
        await this.install();
        break;
      case "uninstall":
        await this.uninstall(this._name);
        break;
      case "list":
        await this.list();
        break;
      default:
        console.info("未知命令\r\n");
        process.exit(-1);
    }
  }
}

const installer = new UBPluginInstaller();
installer.run();
