/*
Company:
Microchip Technology Inc.

File Name:
XNODE.cpp

Summary:
This is a library that will be used to set up a serial communication between an XNODE 
controller and an I/O type XNODE.

Description:
This header file provides implementations for driver APIs for all modules selected in the GUI.
Generation Information :
Product Revision  :  
Device            :  Any device that has a serial port
Driver Version    :  3.00
*/

/* Include of the application */
#include "XNODE.h"                                                                 //Reference to the file header

/* Include of the system */
#include "Arduino.h" 

HardwareSerial *TRANSMITTING_SERIAL = NULL;                                        //Declaration of the global variable to define the serial port through which the communication with the node will be established
HardwareSerial *RECEIVING_SERIAL = NULL;                                           //Declaration of the global variable to define the serial port through which all the instructions sent to the node will be displayed
int waitPeriod = 10000;                                                            //Defines a wait time in milliseconds in case of there being an error in the connection
unsigned long currentTime = 0;
bool bufferException = true;                                                       //Variable used to check if an error has been detected between the microcontroller and the node

    /*
    Constructor:
        XNODE::XNODE(HardwareSerial *_TRANSMITTING_SERIAL)
    Overview:
        Defines the serial ports that will be used to establish 
        communication with the node as global variables.
    Precondition:
        None.
    Input:
        HardwareSerial *_TRANSMITTING_SERIAL: Serial port that communicates with the node; e.g. Serial
    Output:
        None.
    */
XNODE::XNODE(HardwareSerial *_TRANSMITTING_SERIAL) {  
    TRANSMITTING_SERIAL = _TRANSMITTING_SERIAL;                                    //Assigns the serial port through which the commands will be sent to the nodes
}

    /*
    Constructor:
        XNODE::XNODE(HardwareSerial* _TRANSMITTING_SERIAL, HardwareSerial* _RECEIVING_SERIAL)
    Overview:
        Defines the serial ports that will be used to establish 
        communication with the node as global variables.
    Precondition:
        None.
    Input:
        HardwareSerial* _TRANSMITTING_SERIAL: Serial port that communicates with the node; e.g. Serial.
        HardwareSerial* _RECEIVING_SERIAL: Serial port where the instructions between nodes are displayed; e.g. Serial1, Serial2, Serial3.
    Output:
        None.
    */
XNODE::XNODE(HardwareSerial* _TRANSMITTING_SERIAL, HardwareSerial* _RECEIVING_SERIAL) {
    TRANSMITTING_SERIAL = _TRANSMITTING_SERIAL;                                    //Assigns the serial port through which the commands will be sent to the nodes 
    RECEIVING_SERIAL = _RECEIVING_SERIAL;                                          //Assigns the serial port through which all the commands sent to the nodes will be displayed 
}

    /*
    Function:
        void XNODE::SendCommand(char* node, char* command, char* value)
    Overview:
        Sends a command with a numeric value through the serial port to the XNODE.
    Precondition:
        None.
    Input:
        char* node: X-NODE's ID; e.g. 016A, 007E.
        char* command: Commands for the X-NODEs; R, G, F, S.
        char* value: Command's value; e.g. 101010, 00001111.
    Output:
        None.
    */
void XNODE::SendCommand(char* node, char* command, char* value) {
    if (TRANSMITTING_SERIAL->available()) {                                        //Evaluates if the transmitter serial port is enabled
        while (TRANSMITTING_SERIAL->available() > 0) {                             //Evaluates if the transmitter serial port is transmitting information
            char c = TRANSMITTING_SERIAL->read();                                  //Stores the last character obtained
        }
    }
    SerialDebugSet(node, command, value);                                          //Send the commands to visualize them through the receiving serial port
    TRANSMITTING_SERIAL->write("XN");
    TRANSMITTING_SERIAL->write(node);                                              //Node's ID
    TRANSMITTING_SERIAL->write("+");
    TRANSMITTING_SERIAL->write(command);                                           //Command that defines the action to be performed
    TRANSMITTING_SERIAL->write("=");
    TRANSMITTING_SERIAL->write(value);                                             //Value of the action that the node will perform
    TRANSMITTING_SERIAL->write("\r\n");                                            //An "Enter" is written on the serial port to send the previously defined command
}


void XNODE::SendCommandWithRange(char* node, char* command, int value1, int  value2, int value3) {
    if (TRANSMITTING_SERIAL->available()) {                                        //Evaluates if the transmitter serial port is enabled
        while (TRANSMITTING_SERIAL->available() > 0) {                             //Evaluates if the transmitter serial port is transmitting information
            char c = TRANSMITTING_SERIAL->read();                                  //Stores the last character obtained
        }
    }
    //SerialDebugSet(node, command, value);                                          //Send the commands to visualize them through the receiving serial port
    TRANSMITTING_SERIAL->write("XN");
    TRANSMITTING_SERIAL->write(node);                                              //Node's ID
    TRANSMITTING_SERIAL->write("+");
    TRANSMITTING_SERIAL->write(command);
    TRANSMITTING_SERIAL->write("=");                                           //Command that defines the action to be performed
    TRANSMITTING_SERIAL->print(value1);
    TRANSMITTING_SERIAL->write(",");
    TRANSMITTING_SERIAL->print(value2);
    TRANSMITTING_SERIAL->write(",");
    TRANSMITTING_SERIAL->print(value3);                                             //Value of the action that the node will perform
    TRANSMITTING_SERIAL->write("\r\n");                                            //An "Enter" is written on the serial port to send the previously defined command
}


void XNODE::SendCommandWithRange(char* node, char* command, int value) {
    if (TRANSMITTING_SERIAL->available()) {                                        //Evaluates if the transmitter serial port is enabled
        while (TRANSMITTING_SERIAL->available() > 0) {                             //Evaluates if the transmitter serial port is transmitting information
            char c = TRANSMITTING_SERIAL->read();                                  //Stores the last character obtained
        }
    }
    //SerialDebugSet(node, command, value);                                          //Send the commands to visualize them through the receiving serial port
    TRANSMITTING_SERIAL->write("XN");
    TRANSMITTING_SERIAL->write(node);                                              //Node's ID
    TRANSMITTING_SERIAL->write("+S=");
    TRANSMITTING_SERIAL->write(command);                                           //Command that defines the action to be performed
    TRANSMITTING_SERIAL->write(",");
    TRANSMITTING_SERIAL->print(value);                                             //Value of the action that the node will perform
    TRANSMITTING_SERIAL->write("\r\n");                                            //An "Enter" is written on the serial port to send the previously defined command
}
    /*
    Function:
        void XNODE::SendAsk(char* node)
    Overview:
        Sends the ID and the command '?' through the serial port to the X-NODE.
    Precondition:
        None.
    Input:
        char* node: X-NODE's ID; e.g. 016A, 007E.
    Output:
        None.
    */
void XNODE::SendAsk(char* node) {
    TRANSMITTING_SERIAL->write("XN");
    TRANSMITTING_SERIAL->write(node);                                              //Node's ID
    TRANSMITTING_SERIAL->write("?");
    TRANSMITTING_SERIAL->write("\r\n");                                            //An "Enter" is written on the serial port to send the previously defined command
    
}

    /*
    Function:
        float XNODE::GetValueAsFloat(char* node, char* command)
    Overview:
        Sends the command to the X-NODE through the serial port, waits for the 
        response and then processes it.
    Precondition:
        None.
    Input:
        char* node: X-NODE's ID; e.g. 016A, 007E.
        char* command: Commands for the X-NODEs; e.g GH, GT, GP.
    Output:
        Float: The X-NODE's response as a float.
    */
float XNODE::GetValueAsFloat(char* node, char* command) {
    char Buffer[20];                                                               //The function's own Buffer array is declared
    PrintCommand(node, command);                                                   //The PrintCommand() function is executed
    ReadBufferSerial(Buffer, bufferException);                                     //The ReadBufferSerial() function is executed
    if (CheckModel(Buffer, node)) {                                                //Determines if the array's format is correct
        SerialDebugGet(node, command, Buffer);                                     //Sends the instructions to visualize that they were received from the nodes to the receiving serial port
        bufferException = false;                                                   //Defines that the buffer was read correctly
        return (StringtoFloat(Buffer));                                            //Returns the conversion of the value in numeric format of type String to float
    } else {
        SerialDebugGet(node, command, Buffer);                                     //Sends the instructions to visualize that they were received from the nodes to the receiving serial port
        bufferException = true;                                                    //Defines that the buffer was not read correctly
        return -1;                                                                 //Returns a value to notify that the process has failed
    }
}


float XNODE::GetValueAsFloat(char* node, char* command, float val1, float val2) {
    char Buffer[20];                                                               //The function's own Buffer array is declared
    PrintCommand(node, command, val1, val2);                                                   //The PrintCommand() function is executed
    ReadBufferSerial(Buffer, bufferException);                                     //The ReadBufferSerial() function is executed
    if (CheckModel(Buffer, node)) {                                                //Determines if the array's format is correct
        SerialDebugGet(node, command, Buffer);                                     //Sends the instructions to visualize that they were received from the nodes to the receiving serial port
        bufferException = false;                                                   //Defines that the buffer was read correctly
        return (StringtoFloat(Buffer));                                            //Returns the conversion of the value in numeric format of type String to float
    } else {
        SerialDebugGet(node, command, Buffer);                                     //Sends the instructions to visualize that they were received from the nodes to the receiving serial port
        bufferException = true;                                                    //Defines that the buffer was not read correctly
        return -1;                                                                 //Returns a value to notify that the process has failed
    }
}
    /*
    Function:
        int XNODE::GetValueAsInt(char* node, char* command)  
    Overview:
        Sends the command to the X-NODE through the serial port, waits for the 
        response and then processes it.
    Precondition:
        None.
    Input:
        char* node: X-NODE's ID; e.g. 016A, 007E.
        char* command: Commands for the X-NODEs; e.g GH, GT, GP.
    Output:
        Int: The X-NODE's response as an Integer.
    */
int XNODE::GetValueAsInt(char* node, char* command) {
    char Buffer[20];                                                               //The function's own Buffer array is declared
    PrintCommand(node, command);                                                   //The PrintCommand() function is executed
    ReadBufferSerial(Buffer, bufferException);                                      //The ReadBufferSerial() function is executed
    if (CheckModel(Buffer, node)) {                                                //Determines if the array's format is correct
        SerialDebugGet(node, command, Buffer);                                     //Sends the instructions to visualize that they were received from the nodes to the receiving serial port
        bufferException = false;                                                   //Defines that the buffer was read correctly
        return (StringtoInt(Buffer));                                              //Returns the conversion of the value in numeric format of type String to Int
    } else {
        SerialDebugGet(node, command, Buffer);                                     //Sends the instructions to visualize that they were received from the nodes to the receiving serial port
        bufferException = true;                                                    //Defines that the buffer was not read correctly
        return -1;                                                                 //Returns a value to notify that the process has failed
    }
}

    /*
    Function:
        void XNODE::PrintCommand(char* node, char* command) 
    Overview:
        Sends the ID and the command through the serial port.
    Precondition:
        None.
    Input:
        char* node: X-NODE's ID; e.g. 016A, 007E.
        char* command: Commands for the X-NODEs; e.g GH, GT, GP.
    Output:
        none.
    */
void XNODE::PrintCommand(char* node, char* command) {
    if (TRANSMITTING_SERIAL->available()) {                                        //Evaluates if the transmitter serial port is enabled
        while (TRANSMITTING_SERIAL->available() > 0) {                             //Evaluates if the transmitter serial port is transmitting information
            char c = TRANSMITTING_SERIAL->read();                                  //Stores the last character obtained
        }
    }
    TRANSMITTING_SERIAL->write("XN");                                    
    TRANSMITTING_SERIAL->write(node);                                              //Node's ID
    TRANSMITTING_SERIAL->write("+");
    TRANSMITTING_SERIAL->write(command);                                           //Command that defines the action to be performed
    TRANSMITTING_SERIAL->write("\r\n");                                            //An "Enter" is written on the serial port to send the previously defined command
}


void XNODE::PrintCommand(char* node, char* command, float val1, float val2) {
    if (TRANSMITTING_SERIAL->available()) {                                        //Evaluates if the transmitter serial port is enabled
        while (TRANSMITTING_SERIAL->available() > 0) {                             //Evaluates if the transmitter serial port is transmitting information
            char c = TRANSMITTING_SERIAL->read();                                  //Stores the last character obtained
        }
    }
    TRANSMITTING_SERIAL->write("XN");                                    
    TRANSMITTING_SERIAL->write(node);                                              //Node's ID
    TRANSMITTING_SERIAL->write("+");
    TRANSMITTING_SERIAL->write("SG=");
    TRANSMITTING_SERIAL->print(val1); 
    TRANSMITTING_SERIAL->write(",");
    TRANSMITTING_SERIAL->print(val2);                                         //Command that defines the action to be performed
    TRANSMITTING_SERIAL->write("\r\n");                                            //An "Enter" is written on the serial port to send the previously defined command
}

    /*
    Function:
        int XNODE::ReadBufferSerial(char* Buffer, bool _bufferException)
    Overview:
        Reads the serial port and stores the characters in "Buffer".
        Returns the number of characters read.
    Precondition:
        None.
    Input:
        char* Buffer: XNODE´s Answer from the serial port.
        bool _bufferException: Checks if the connection meets with the correct conditions.
    Output:
        Int: Number of characters read.
    */


int XNODE::ReadBufferSerial(char* Buffer, bool _bufferException) {

    int ContBuff = 0;
    bool bufferWait = true;                                                        //Control variable for the buffer reading
    char bufferExtractor;

    while (TRANSMITTING_SERIAL->available() > 0) {                                 //Verifies that the transmitter serial port is transmitting information
        bufferExtractor = TRANSMITTING_SERIAL->read();                             //Stores the last character transmitted by the transmitter serial port
    }

    while (bufferWait) {                                                           //Cycle that runs while there is information being received through the serial port
        if (((unsigned long)(millis() - currentTime) > waitPeriod) && _bufferException) { //Periodically checks if the connection was established
            currentTime = millis();
            errorMessage();                                                        //Function that sends an error message
            bufferWait = false;
        }

        if (TRANSMITTING_SERIAL->available()) {                                    //Checks if the transmitter serial port is available
            char c = TRANSMITTING_SERIAL->read();                                  //Stores the character that is currently transmitted through the transmitter serial port
            if (c == '\n' || c == '\r') {                                          //Checks if the current array's index character is different from '\ n' or '\ r'
                Buffer[ContBuff] = '\0';                                           //Makes the character at the following index equal to '\ r' and '\ n' to '\ 0' so that the program knows where the expression ends.
                bufferWait = false;
            } else {
                bufferWait = true;
                Buffer[ContBuff++] = c;                                            //Stores the last character obtained from the transmitter serial port
            }
        }
    }
    return ContBuff;                                                               //Returns the final size of the arrangement
}

    /*
    Function:
        bool XNODE::CheckModel(char* Buffer, char* node)
    Overview:
        Returns a bool value when deciding if the evaluation is 
        true or false.
    Precondition:
        None.
    Input:
        char* Buffer: XNODE´s Answer from the serial port.
        char* node: X-NODE's ID; e.g. 016A, 007E
    Output:
        Bool: Comparison result.
    */
bool XNODE::CheckModel(char* Buffer, char* node) {
    char ModeloID[10];
    sprintf(ModeloID, "XN%s=", node);                                              //Performs concatenation, returning a string of characters
    if (!(strncmp(Buffer, ModeloID, 6))) {                                         //Performs a logical operation to determine if the size of the array is correct
        return true;                                                               //Returns a bool value indicating that it is correct
    } else {
        return false;                                                              //Returns a bool value indicating that it is incorrect
    }
}

    /*
    Function:
        void XNODE::SerialDebugGet(char* node, char* command, char* Buffer)
    Overview:
        If the constructor method defines a second serial port, this function will be in charge 
        of verifying which one has been defined and will show through it which data is being 
        sent through the transmission serial port to the node. Tts particular use is to monitor 
        information request type commands.
    Precondition:
        A second serial port is defined in the constructor.
    Input:
        char* node: X-NODE's ID; e.g. 016A, 007E.
        char* command: Commands for the X-NODEs; R, G, F, S.
        char* Buffer: XNODE´s Answer from the serial port.
    Output:
        none
    */
void XNODE::SerialDebugGet(char* node, char* command, char* Buffer) {
    if (RECEIVING_SERIAL != NULL){                                                 //Checks if a second serial port has been defined
        RECEIVING_SERIAL->write(">");  
        RECEIVING_SERIAL->write("XN");
        RECEIVING_SERIAL->write(node);                                             //Node's ID
        RECEIVING_SERIAL->write("+");
        RECEIVING_SERIAL->write(command);                                          //Command that defines the action to be performed
        RECEIVING_SERIAL->write("\r\n");                                           //An "Enter" is written on the serial port to send the previously defined command
        RECEIVING_SERIAL->write("<");
        RECEIVING_SERIAL->write(Buffer);                                           //The numeric value that was received from the node
        RECEIVING_SERIAL->write("\r\n");                                           //An "Enter" is written on the serial port to send the previously defined command
    }
}

    /*
    Function:
        void XNODE::SerialDebugSet(char* node, char* command, char* value)
    Overview:
        If the constructor method defines a second serial port, this function will be in charge 
        of verifying which one has been defined and will show through it which data is being 
        sent through the transmission serial port to the node. Its particular use is to monitor 
        action request type commands.
    Precondition:
        A second serial port is defined in the constructor.
    Input:
        char* node: X-NODE's ID; e.g. 016A, 007E.
        char* command: Commands for the X-NODEs; R, G, F, S.
        char* value: Command's value; e.g. 101010, 00001111.
    Output:
        none
    */
void XNODE::SerialDebugSet(char* node, char* command, char* value) {
    if (RECEIVING_SERIAL != NULL){                                                 //Comprueba si un segundo puerto serial ha sido definido
        RECEIVING_SERIAL->write(">");  
        RECEIVING_SERIAL->write("XN");
        RECEIVING_SERIAL->write(node);                                             //Node's ID
        RECEIVING_SERIAL->write("+");
        RECEIVING_SERIAL->write(command);                                          //Command that defines the action to be performed
        RECEIVING_SERIAL->write("=");
        RECEIVING_SERIAL->write(value);                                            //Value of the action that the node will perform
        RECEIVING_SERIAL->write("\r\n");                                           //An "Enter" is written on the serial port to send the previously defined command 
    }
}

    /*
    Function:
        float XNODE::StringtoFloat(char* Buffer)
    Overview:
        Performs the conversion of a numeric value in string type format 
        to a float type format.
    Precondition:
        None.
    Input:
        char* Buffer: XNODE´s Answer from the serial port.
    Output:
        float: The X-NODE's response as a float.
    */
float XNODE::StringtoFloat(char* Buffer) {
    String Buffer1 = Buffer;                                                       //The "Buffer" array is stored in the variable of type String Buffer1
    String NumberInString = Buffer1.substring(6);                                  //Returns the construction of the number array
    return (NumberInString.toFloat());                                             //The conversion of the numeric value is returned, from String to float format
}

    /*
    Function:
        int XNODE::StringtoInt(char* Buffer)
    Overview:
        Performs the conversion of a numeric value in string type format 
        to an Integer type format.
    Precondition:
        None.
    Input:
        char* Buffer: XNODE´s Answer from the serial port.
    Output:
        int: The X-NODE's response as an Integer.
    */
int XNODE::StringtoInt(char* Buffer) {
    String Buffer1 = Buffer;                                                       //The "Buffer" array is stored in the variable of type String Buffer1
    String NumberInString = Buffer1.substring(6);                                  //Returns the construction of the number array
    return (NumberInString.toInt());                                               //The conversion of the numeric value is returned, from String to Integer format
}

    /*
    Function:
        void errorMessage()
    Overview:
        Sends a message of possible solutions to a specific error.
    Precondition:
        none
    Input:
        none
    Output:
        none
    */
void XNODE::errorMessage() {
    if(RECEIVING_SERIAL != NULL) {
        RECEIVING_SERIAL->write("Conexión No Exitosa");
        RECEIVING_SERIAL->write("\r\n");
        RECEIVING_SERIAL->write("Verifique: ");
        RECEIVING_SERIAL->write("\r\n");
        RECEIVING_SERIAL->write("> Que el ID del node sea correcto ");
        RECEIVING_SERIAL->write("\r\n");
        RECEIVING_SERIAL->write("> Que el comando tenga la sintaxis correcta");
        RECEIVING_SERIAL->write("\r\n");
        RECEIVING_SERIAL->write("> Que su dispositivo XNODE este conectado correctamente");
        RECEIVING_SERIAL->write("\r\n");
    }
}