import type { FnN2, NumericArray } from "@thi.ng/api";
/**
 * Produces a new function which computes derivative of the given single-arg
 * function.
 *
 * @remarks
 * The extra optional arg `eps` is used to define the step width for
 * computing derived values:
 *
 * `f'(x) = (f(x + eps) - f(x)) / eps`
 *
 * The original function is assumed to be fully differentiable in the interval
 * the returned function is going to be used. No validity checks of any form are
 * done.
 *
 * https://en.wikipedia.org/wiki/Derivative#Continuity_and_differentiability
 *
 * @param fn -
 * @param eps -
 */
export declare const derivative: (f: (x: number) => number, eps?: number) => (x: number) => number;
/**
 * Computes solution for linear equation: `ax + b = 0`. Returns 0 iff `a == 0`
 *
 * @param a - slope
 * @param b - constant offset
 */
export declare const solveLinear: FnN2;
/**
 * Computes solutions for quadratic equation: `ax^2 + bx + c = 0`. Returns array
 * of real solutions.
 *
 * @remarks
 * `a` MUST NOT be zero. If the quadratic term is missing, use
 * {@link solveLinear} instead.
 *
 * - https://en.wikipedia.org/wiki/Quadratic_function
 * - https://en.wikipedia.org/wiki/Quadratic_equation
 *
 * @param a - quadratic coefficient
 * @param b - linear coefficient
 * @param c - constant offset
 * @param eps - tolerance to determine multiple roots
 */
export declare const solveQuadratic: (a: number, b: number, c: number, eps?: number) => number[];
/**
 * Computes solutions for quadratic equation: `ax^3 + bx^2 + c*x + d = 0`.
 * Returns array of solutions, both real & imaginary. Note: `a` MUST NOT be
 * zero. If the cubic term is missing (i.e. zero), use {@link solveQuadratic} or
 * {@link solveLinear} instead.
 *
 * https://en.wikipedia.org/wiki/Cubic_function
 *
 * @param a - cubic coefficient
 * @param b - quadratic coefficient
 * @param c - linear coefficient
 * @param d - constant offset
 * @param eps - tolerance to determine multiple roots
 */
export declare const solveCubic: (a: number, b: number, c: number, d: number, eps?: number) => number[];
/**
 * Solves a system of linear equations for N unknowns: `a[i]*x[i−1] + b[i]*x[i]
 * + c[i]*x[i+1] = d[i]`, where a[0]=0 and c[N-1]=0. Writes solutions `x[i]` back into
 * array of input coefficients `d` and returns it. The other arrays are not modified.
 *
 * @remarks
 * Reference:
 * https://en.wikipedia.org/wiki/Tridiagonal_matrix_algorithm
 *
 * @param a - subdiagonal [1,N-1], a[0] is lower left corner
 * @param b - main diagonal [0,N-1]
 * @param c - superdiagonal [0,N-2], c[N-1] is upper right corner
 * @param d - input coefficients & output solutions [0,N-1]
 */
export declare const solveTridiagonal: (a: NumericArray, b: NumericArray, c: NumericArray, d: NumericArray) => NumericArray;
//# sourceMappingURL=solve.d.ts.map