import type { Comparator, Fn, FnN, FnN2 } from "@thi.ng/api";
/**
 * Returns the supposed index of `x` in pre-sorted array-like collection `buf`.
 *
 * @remarks
 * If `x` can't be found, returns `-index-1`, representing the negative of the
 * index, were `x` to be inserted into `buf`. E.g if the return value is -3, `x`
 * would appear/insert at index 2.
 *
 * The optional `key` function is used to obtain the actual sort value of `x`
 * and each array item (default: identity).
 *
 * The optional `cmp` comparator (default:
 * [`compare()`](https://docs.thi.ng/umbrella/compare/functions/compare.html))
 * is then used to identify the index of `x`. The sort order of `buf` MUST be
 * compatible with that of `cmp`.
 *
 * @example
 * ```ts tangle:../export/binary-search1.ts
 * import { binarySearch } from "@thi.ng/arrays";
 *
 * console.log(
 *   binarySearch([2, 4, 6], 5)
 * );
 * // -3
 * ```
 *
 * @param buf - array
 * @param x - search value
 * @param key - key function
 * @param cmp - comparator
 * @param low - min index
 * @param high - max index
 */
export declare const binarySearch: <A, B>(buf: ArrayLike<A>, x: A, key?: Fn<A, B>, cmp?: Comparator<B>, low?: number, high?: number) => number;
/**
 * Similar to {@link binarySearch}, but optimized for numeric arrays and
 * supporting custom comparators (default:
 * [`compareNumAsc()`](https://docs.thi.ng/umbrella/compare/functions/compareNumAsc.html)).
 *
 * @param buf - array
 * @param x - search value
 * @param cmp - comparator
 * @param low - min index
 * @param high - max index
 */
export declare const binarySearchNumeric: (buf: ArrayLike<number>, x: number, cmp?: Comparator<number>, low?: number, high?: number) => number;
export declare const binarySearch2: (buf: ArrayLike<number>, x: number) => number;
/**
 * Non-recursive, optimized binary search for fixed size numeric arrays of 4
 * values. Returns index of `x` or `-index-1` if not found.
 *
 * @param buf -
 * @param x -
 */
export declare const binarySearch4: (buf: ArrayLike<number>, x: number) => number;
/**
 * Non-recursive, optimized binary search for fixed size numeric arrays of 8
 * values. Returns index of `x` or `-index-1` if not found.
 *
 * @param buf -
 * @param x -
 */
export declare const binarySearch8: (buf: ArrayLike<number>, x: number) => number;
/**
 * Non-recursive, optimized binary search for fixed size numeric arrays of 16
 * values. Returns index of `x` or `-index-1` if not found.
 *
 * @param buf -
 * @param x -
 */
export declare const binarySearch16: (buf: ArrayLike<number>, x: number) => number;
/**
 * Non-recursive, optimized binary search for fixed size numeric arrays of 32
 * values. Returns index of `x` or `-index-1` if not found.
 *
 * @param buf -
 * @param x -
 */
export declare const binarySearch32: (buf: ArrayLike<number>, x: number) => number;
/**
 * {@link binarySearch} result index classifier for predecessor queries.
 * Returns index of last item less than search value or -1 if no such
 * values exist.
 *
 * @example
 * ```ts tangle:../export/binary-search2.ts
 * import { binarySearch, bsLT } from "@thi.ng/arrays";
 *
 * console.log(
 *   bsLT(binarySearch([10, 20, 30, 40], 25))
 * )
 * // 1
 * ```
 *
 * @param i - binarySearch result index
 */
export declare const bsLT: FnN;
/**
 * Similar to {@link bsLT}, but for less-than-equals queries.
 *
 * @param i - binarySearch result index
 */
export declare const bsLE: FnN;
/**
 * {@link binarySearch} result index classifier for successor queries.
 * Returns index of first item greater than search value or -1 if no
 * such values exist.
 *
 * @example
 * ```ts tangle:../export/binary-search3.ts
 * import { binarySearch, bsGT } from "@thi.ng/arrays";
 *
 * const src = [10, 20, 30, 40];
 *
 * console.log(
 *   bsGT(binarySearch(src, 25), src.length)
 * )
 * // 2
 *
 * console.log(
 *   bsGT(binarySearch(src, 40), src.length)
 * )
 * // -1
 * ```
 *
 * @param i - binarySearch result index
 * @param n - array length
 */
export declare const bsGT: FnN2;
/**
 * Similar to {@link bsGT}, but for greater-than-equals queries.
 *
 * @param i - binarySearch result index
 * @param n - array length
 */
export declare const bsGE: FnN2;
/**
 * {@link binarySearch} result index classifier for equals queries.
 * Merely syntax sugar, casting any non-found result indices to -1.
 *
 * @param i - binarySearch result index
 */
export declare const bsEQ: FnN;
//# sourceMappingURL=binary-search.d.ts.map