import type { FnN, FnN2, FnN3, FnN4, FnN5, FnN6 } from "@thi.ng/api";
/**
 * Linear interpolation without clamping. Computes `a + (b - a) * t`
 *
 * @param a - start value
 * @param b - end value
 * @param t - interpolation factor [0..1]
 */
export declare const mix: FnN3;
/**
 * Bilinear interpolation of given values (`a`,`b`,`c`,`d`).
 *
 * @example
 * ```text
 * c    d
 * +----+
 * |    |
 * +----+
 * a    b
 * ```
 *
 * @param a - BL value
 * @param b - BR value
 * @param c - TL value
 * @param d - TR value
 * @param u - 1st interpolation factor
 * @param v - 2nd interpolation factor
 */
export declare const mixBilinear: FnN6;
/**
 * Computes quadratic bezier interpolation for normalized value `t`.
 *
 * @param a
 * @param b
 * @param c
 * @param t
 */
export declare const mixQuadratic: FnN4;
/**
 * Computes cubic bezier interpolation for normalized value `t`.
 *
 * @param a
 * @param b
 * @param c
 * @param d
 * @param t
 */
export declare const mixCubic: FnN5;
/**
 * Returns hermite interpolation of `a, b, c, d` at normalized position `t`,
 * where `a` and `d` are used as predecessor/successor of `b` / `c` and only
 * inform the tangent of the interpolation curve. The interpolated result is
 * that of `b` and `c`.
 *
 * Assumes all inputs are uniformly spaced. If that's not the case, use
 * {@link mixCubicHermite} with one of the tangent generators supporting
 * non-uniform spacing of points.
 *
 * See: https://www.desmos.com/calculator/j4gf8g9vkr
 *
 * Source:
 * https://www.musicdsp.org/en/latest/Other/93-hermite-interpollation.html
 *
 * - {@link mixCubicHermite}
 * - {@link tangentCardinal}
 * - {@link tangentDiff3}
 *
 * @param a -
 * @param b -
 * @param c -
 * @param d -
 * @param t -
 */
export declare const mixHermite: FnN5;
/**
 * Computes cubic-hermite interpolation between `a` / `b` at normalized
 * time `t` and using respective tangents `ta` / `tb`.
 *
 * https://en.wikipedia.org/wiki/Cubic_Hermite_spline
 *
 * - {@link mixHermite}
 * - {@link tangentCardinal}
 * - {@link tangentDiff3}
 *
 * @param a -
 * @param ta -
 * @param b -
 * @param tb -
 * @param t -
 */
export declare const mixCubicHermite: FnN5;
/**
 * Similar to {@link mixCubicHermite}, but takes 4 control values (uniformly
 * spaced) and computes tangents automatically. Returns `b` iff `t=0` and `c`
 * iff `t=1.0`.
 *
 * @param a -
 * @param b -
 * @param c -
 * @param d -
 * @param t -
 */
export declare const mixCubicHermiteFromPoints: FnN5;
/**
 * Bicubic interpolation of given 4x4 sample values (in row major order, i.e.
 * `s00..s03` = 1st row).
 *
 * @remarks
 * Result will not be clamped and might fall outside the total range of the
 * input samples.
 *
 * @param s00 -
 * @param s01 -
 * @param s02 -
 * @param s03 -
 * @param s10 -
 * @param s11 -
 * @param s12 -
 * @param s13 -
 * @param s20 -
 * @param s21 -
 * @param s22 -
 * @param s23 -
 * @param s30 -
 * @param s31 -
 * @param s32 -
 * @param s33 -
 * @param u -
 * @param v -
 */
export declare const mixBicubic: (s00: number, s01: number, s02: number, s03: number, s10: number, s11: number, s12: number, s13: number, s20: number, s21: number, s22: number, s23: number, s30: number, s31: number, s32: number, s33: number, u: number, v: number) => number;
/**
 * Helper function for {@link mixCubicHermite}. Computes cardinal tangents based
 * on point neighbors of a point B (not given), i.e. `a` (predecessor) and `c`
 * (successor) and their times (defaults to uniformly spaced). The optional
 * `tension` parameter can be used to scale the tangent where 0.0 produces a
 * Cardinal spline tangent and 1.0 a Catmull-Rom (opposite to the Wikipedia
 * ref).
 *
 * https://en.wikipedia.org/wiki/Cubic_Hermite_spline#Cardinal_spline
 *
 * @param prev -
 * @param next -
 * @param scale -
 * @param ta -
 * @param tc -
 */
export declare const tangentCardinal: (prev: number, next: number, scale?: number, ta?: number, tc?: number) => number;
/**
 * Helper function for {@link mixCubicHermite}. Computes tangent for `curr`,
 * based on 3-point finite difference, where `prev` & `next` are `curr`'s
 * neighbors and the `tX` the three points' respective time values. The latter
 * are equally spaced by default (each 1.0 apart).
 *
 * Using this function with equal spacing of 1.0 and together with
 * {@link mixCubicHermite} will produce same results as the somewhat optimized
 * variant {@link mixHermite}.
 *
 * https://en.wikipedia.org/wiki/Cubic_Hermite_spline#Finite_difference
 *
 * @param prev -
 * @param curr -
 * @param next -
 * @param ta -
 * @param tb -
 * @param tc -
 */
export declare const tangentDiff3: (prev: number, curr: number, next: number, ta?: number, tb?: number, tc?: number) => number;
/**
 * HOF interpolator. Takes a timing function `f` and interval `[from,to]`.
 * Returns function which takes normalized time (in [0,1] range) as single arg
 * and returns interpolated value.
 *
 * @example
 * ```ts tangle:../export/tween.ts
 * import { easeInOut2, tween } from "@thi.ng/math";
 *
 * // create tweening function
 * const anim = tween(easeInOut2, 100, 200);
 *
 * for(let i=0; i<=10; i++) console.log(anim(i / 10));
 * // 100
 * // 102
 * // 108
 * // 118
 * // 132
 * // 150
 * // 168
 * // 182
 * // 192
 * // 198
 * // 200
 * ```
 *
 * @param f -
 * @param from -
 * @param to -
 */
export declare const tween: (f: (t: number) => number, from: number, to: number) => (t: number) => number;
/**
 * Circular interpolation (ease out): `sqrt(1 - (1 - t)^2)`
 *
 * @remarks
 * Reference: https://www.desmos.com/calculator/tisoiazdrw
 *
 * @param t - interpolation factor [0..1]
 */
export declare const circular: FnN;
/**
 * Inverse/flipped version of {@link circular} (ease in).
 *
 * @remarks
 * Reference: https://www.desmos.com/calculator/tisoiazdrw
 *
 * @param t - interpolation factor [0..1]
 */
export declare const invCircular: FnN;
/**
 * Zoomlens interpolation with customizable lens position, behavior and
 * strength.
 *
 * @remarks
 * Lens position must be given in (0..1) interval. Lens strength must be in
 * [-1,1] range. If negative, the lens will be bundling values near `pos`, if
 * positive the lens has dilating characteristics and will spread values near
 * `pos` towards the edges.
 *
 * Also see {@link schlick} for an alternative approach and {@link tween} for
 * reference to below example.
 *
 * @example
 * ```ts tangle:../export/lens.ts
 * import { mix, lens, tween } from "@thi.ng/math";
 * import { partial } from "@thi.ng/compose";
 *
 * // interpolated position in [100..400] interval for given `t`
 * // const y = mix(100, 400, lens(0.5, 1, t));
 *
 * // or compose tween function via `tween()` & `partial()`
 * const f = tween(partial(lens, 0.75, 1), 100, 400);
 *
 * for(let i=0; i<=10; i++) console.log(f(i / 10));
 * // 100.0
 * // 102.0
 * // 108.1
 * // 118.7
 * // 134.6
 * // 157.2
 * // 190.0
 * // 244.2
 * // 370.0
 * // 393.7
 * // 400.0
 * ```
 *
 * @param pos - lens pos
 * @param strength - lens strength
 * @param t - interpolation factor [0..1]
 */
export declare const lens: FnN3;
export declare const cosine: FnN;
export declare const decimated: FnN2;
/**
 * Spring oscillator with damping.
 *
 * @remarks
 * Interactive graph:
 * https://www.desmos.com/calculator/tywbpw8pck
 *
 * @param k
 * @param amp
 * @param t
 */
export declare const bounce: FnN3;
/**
 * Exponential easing.
 *
 * - `ease = 1` -> linear
 * - `ease > 1` -> ease in
 * - `ease < 1` -> ease out
 *
 * @param ease - easing behavior [0.0 .. ∞]
 * @param t -
 */
export declare const ease: FnN2;
/**
 * Impulse generator. Peaks at `t = 1/k`
 *
 * @param k - impulse width (higher values => shorter impulse)
 */
export declare const impulse: FnN2;
export declare const gain: FnN2;
export declare const parabola: FnN2;
export declare const cubicPulse: FnN3;
/**
 * Unnormalized Sinc function: sin(x)/x. Returns 1 for t=0.
 *
 * @remarks
 * https://en.wikipedia.org/wiki/Sinc_function
 *
 * @param k -
 * @param t -
 */
export declare const sinc: FnN;
/**
 * Normalized Sinc function, returns sinc(π*k*t).
 *
 * @remarks
 * https://en.wikipedia.org/wiki/Sinc_function
 *
 * @see {@link sinc}
 *
 * @param k -
 * @param t -
 */
export declare const sincNormalized: FnN2;
/**
 * Lanczos filter. Returns `sinc(πt)sinc(πt/a)` iff `t` in (-a,a) interval, else
 * returns 0.
 *
 * @remarks
 * Interactive graph: https://www.desmos.com/calculator/pmypqgefle
 *
 * @param a -
 * @param t -
 */
export declare const lanczos: FnN2;
/**
 * Sigmoid function for inputs arounds center bias.
 *
 * @remarks
 * Updated in v3.0.0 to add bias value to satisfy more use cases. Use
 * {@link sigmoid01} for old behavior.
 *
 * @param bias - center value (for which result = 0.5)
 * @param k - steepness
 * @param t - input value
 */
export declare const sigmoid: FnN3;
/**
 * Sigmoid function for inputs in [0..1] interval. Center bias = 0.5.
 *
 * @param k - steepness
 * @param t - input value
 */
export declare const sigmoid01: FnN2;
/**
 * Sigmoid function for inputs in [-1..+1] interval. Center bias = 0
 *
 * @param k -
 * @param t -
 */
export declare const sigmoid11: FnN2;
/**
 * Generalized Schlick bias gain curve, based on:
 * https://arxiv.org/abs/2010.09714
 *
 * @remarks
 * Interactive graph:
 * https://www.desmos.com/calculator/u6bkm5rb7t
 *
 * @param a - curve strength. recommended (0..64]
 * @param b - pivot position [0..1]
 * @param t - input val [0..1]
 */
export declare const schlick: FnN3;
/**
 * Computes exponential factor to interpolate from `a` to `b` over
 * `num` steps. I.e. multiplying `a` with the returned factor will yield
 * `b` after `num` steps. All args must be > 0.
 *
 * @param a -
 * @param b -
 * @param num -
 */
export declare const expFactor: FnN3;
/**
 * Computes gaussian bell curve for given center `bias` and `sigma` (spread).
 *
 * @remarks
 * Interactive graph: https://www.desmos.com/calculator/aq6hdzxprv
 *
 * @param bias -
 * @param sigma -
 * @param t -
 */
export declare const gaussian: FnN3;
//# sourceMappingURL=mix.d.ts.map