/**
 * Returns function accepting a single index arg used to
 * lookup value in given array. No bounds checks are done.
 *
 * @example
 * ```ts tangle:../export/lookup.ts
 * import { lookup1d, map } from "@thi.ng/transducers";
 *
 * console.log(
 *   [...map(lookup1d([10, 20, 30]), [2,0,1])]
 * );
 * // [ 30, 10, 20 ]
 * ```
 *
 * @param src - source data
 */
export declare const lookup1d: <T>(src: T[]) => (i: number) => T;
/**
 * Returns function accepting a single `[x, y]` index tuple, used to
 * lookup value in given array. Useful for transducers processing 2D
 * data.
 *
 * @remarks
 * The source data MUST be in row major linearized format, i.e. 1D
 * representation of 2D data (pixel buffer). No bounds checks are done.
 *
 * @example
 * ```ts tangle:../export/lookup2d.ts
 * import { lookup2d, map, range, range2d } from "@thi.ng/transducers";
 *
 * console.log(
 *   [...map(lookup2d([...range(9)], 3), range2d(2, -1, 0, 3))]
 * );
 * // [ 2, 1, 0, 5, 4, 3, 8, 7, 6 ]
 * ```
 *
 * @param src - source data
 * @param width - number of items along X (columns)
 */
export declare const lookup2d: <T>(src: T[], width: number) => (i: number[]) => T;
/**
 * Same as {@link lookup2d}, but for 3D data. The index ordering of the
 * source data MUST be in Z, Y, X order (i.e. a stack of row major 2D slices).
 * No bounds checks are done.
 *
 * @param src - source data
 * @param width - number of items along X (columns)
 * @param height - number of items along Y (rows)
 */
export declare const lookup3d: <T>(src: T[], width: number, height: number) => (i: number[]) => T;
//# sourceMappingURL=lookup.d.ts.map