import type { Fn2, FnN } from "@thi.ng/api";
export interface TweenOpts<A, B, C> {
    /**
     * Total number (n+1) of tweened values to produce
     */
    num: number;
    /**
     * Min time boundary. Only values in the closed `[min..max]`
     * time interval will be computed.
     */
    min: number;
    /**
     * Max time boundary. Only values in the closed `[min..max]`
     * time interval will be computed.
     */
    max: number;
    /**
     * Interval producer (from 2 keyframe values, i.e. `stops`)
     */
    init: Fn2<A, A, B>;
    /**
     * Interval interpolator
     */
    mix: Fn2<B, number, C>;
    /**
     * Optional easing function to transform the interval relative `time` param
     * for `mix`.
     */
    easing?: FnN;
    /**
     * Keyframe definitions, i.e. `[time, value]` tuples
     */
    stops: [number, A][];
}
/**
 * Keyframe based interpolator. Yields a sequence of `num+1` equally spaced,
 * tweened values from given keyframe tuples (`stops`). Keyframes are defined as
 * `[time, value]` tuples. Only values in the closed `[min..max]` time interval
 * will be computed.
 *
 * @remarks
 * Interpolation happens in two stages: First the given `init` function is
 * called to transform/prepare pairs of consecutive keyframes into a single
 * interval (user defined). Then, to produce each tweened value calls `mix` with
 * the currently active interval and interpolation time value `t` (re-normalized
 * and relative to current interval). The iterator yields results of these
 * `mix()` function calls.
 *
 * Depending on the overall `num`ber of samples requested and the distance
 * between keyframes, some keyframes MIGHT be skipped. E.g. if requesting 10
 * samples within [0,1], the interval between two successive keyframes at 0.12
 * and 0.19 would be skipped entirely, since samples will only be taken at
 * multiples of `1/num` (i.e. 0.0, 0.1, 0.2... in this example).
 *
 * The given keyframe times can lie outside the `min`/`max` range and also don't
 * need to cover the range fully. In the latter case, tweened values before the
 * first or after the last keyframe will yield the value of the first/last
 * keyframe. If only a single keyframe is given in total, all `num` yielded
 * samples will be that keyframe's transformed value.
 *
 * @example
 * ```ts tangle:../export/tween.ts
 * import { tween } from "@thi.ng/transducers";
 *
 * console.log(
 *   [...tween({
 *     num: 10,
 *     min: 0,
 *     max: 100,
 *     init: (a, b) => [a, b],
 *     mix: ([a, b], t) => Math.floor(a + (b - a) * t),
 *     stops: [[20, 100], [50, 200], [80, 0]]
 *   })]
 * );
 * // [ 100, 100, 100, 133, 166, 200, 133, 66, 0, 0, 0 ]
 * ```
 *
 * Using easing functions (e.g. via [`thi.ng/math`](https://thi.ng/math)),
 * non-linear interpolation within each keyframe interval can be achieved:
 *
 * @example
 * ```ts tangle:../export/tween-2.ts
 * import { tween } from "@thi.ng/transducers";
 * import { mix, smoothStep } from "@thi.ng/math"
 *
 * console.log(
 *   [...tween({
 *     num: 10,
 *     min: 0,
 *     max: 100,
 *     init: (a, b) => [a, b],
 *     mix: ([a, b], t) => Math.floor(mix(a, b, smoothStep(0.1, 0.9, t))),
 *     stops: [[20, 100], [50, 200], [80, 0]]
 *   })]
 * );
 * // [ 100, 100, 100, 120, 179, 200, 158, 41, 0, 0, 0 ]
 * ```
 *
 * - {@link TweenOpts}
 * - {@link interpolate}
 * - {@link interpolateHermite}
 * - {@link interpolateLinear}
 *
 * @param opts -
 */
export declare function tween<A, B, C>(opts: TweenOpts<A, B, C>): IterableIterator<C>;
//# sourceMappingURL=tween.d.ts.map