<?php
/**
 * @noinspection SpellCheckingInspection
 */

declare(strict_types=1);

namespace Siel\Acumulus\ApiClient;

use DateTimeInterface;
use Siel\Acumulus\Api;
use Siel\Acumulus\Config\Environment;
use Siel\Acumulus\Data\EmailInvoiceAsPdf;
use Siel\Acumulus\Data\EmailPackingSlipAsPdf;
use Siel\Acumulus\Data\Invoice;
use Siel\Acumulus\Data\StockTransaction;
use Siel\Acumulus\Fld;
use Siel\Acumulus\Helpers\Container;
use Siel\Acumulus\Helpers\Log;
use Siel\Acumulus\Helpers\Severity;

/**
 * Acumulus provides an easy interface towards the different API calls of the
 * Acumulus web API.
 *
 * This class simplifies the communication so that the different webshop-specific
 * interfaces can be more rapidly developed.
 *
 * More info:
 * - {@link https://www.siel.nl/acumulus/API/}
 * - {@link https://www.siel.nl/acumulus/koppelingen/}
 *
 * The ApiClient API call wrappers return their information as a keyed array,
 * which is a simplified version of the call-specific part of the response
 * structure.
 */
class Acumulus
{
    protected Environment $environment;
    protected Log $log;
    protected Container $container;

    public function __construct(Container $container, Environment $environment, Log $log)
    {
        $this->environment = $environment;
        $this->container = $container;
        $this->log = $log;
    }

    /**
     * Retrieves the "about information".
     *
     * See {@link https://www.siel.nl/acumulus/API/Misc/About/}.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "about" array, being a keyed array with keys:
     *   - about: General name for the API.
     *   - tree: stable, current, deprecated or closed.
     *   - role: Name of userrole, current known roles: Beheerder, Gebruiker,
     *       Invoerder, API-beheerder, API-gebruiker, API-invoerder, API-open
     *       (not a real role, just to indicate the calls that are available
     *       without authentication).
     *   - roleid: Numeric identifier of userrole.
     *   Possible errors:
     *   - 553 XUPR7NEC8: Warning: You are using a deprecated user role to
     *     connect to the Acumulus API. Please add another user with an
     *     API-compliant role or change the role for the current user.
     *   - 403 A8N403GCN: Forbidden - Insufficient credential level for
     *     general/general_about.php. Not authorized to perform request.
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getAbout(): AcumulusResult
    {
        return $this->callApiFunction('general/general_about', [])->setMainAcumulusResponseKey('general');
    }

    /**
     * Retrieves the "My Acumulus" information.
     *
     * See {@link https://www.siel.nl/acumulus/API/Misc/My_Acumulus/}
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "mydata" array, being a keyed array with keys:
     *   - 'mycontractcode'
     *   - 'mycompanyname'
     *   - 'mycontactperson'
     *   - 'myaddress'
     *   - 'mypostalcode'
     *   - 'mycity'
     *   - 'mytelephone'
     *   - 'myemail'
     *   - 'myiban'
     *   - 'mysepamandatenr'
     *   - 'mycontractenddate'
     *   - 'mysalutation'
     *   - 'myemailstatusid'
     *   - 'myemailstatusreferenceid'
     *   - 'myvatnumber'
     *   - 'mystatusid'
     *   - 'myentries'
     *   - 'mymaxentries'
     *   - 'myentriesleft'
     *   - 'mydebt'
     *   - 'mysupport': optinal array with 1 key:
     *     - 'item': 1 "item" or a numerical array with "items", an "item"
     *       being a keyed array with keys:
     *       - 'description': type of support bought.
     *       - 'location': The server for which the support is bought.
     *       - 'token': Acumulus token: 30 hex characters.
     *       - 'startdate' yyyy-mm-dd
     *       - 'enddate': yyyy-mm-dd
     *   Possible errors: todo.
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getMyAcumulus(): AcumulusResult
    {
        return $this->callApiFunction('general/my_acumulus', [])->setMainAcumulusResponseKey('mydata');
    }

    /**
     * Retrieves a list of accounts.
     *
     * @param bool $enabled
     *   Whether to retrieve enabled (true, default) or disabled (false)
     *   accounts.
     *
     * See {@link https://www.siel.nl/acumulus/API/Accounts/List_Accounts/}
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   a non-keyed array of "account" arrays, each "account" array being a
     *   keyed array with keys:
     * - 'accountid'
     * - 'accountnumber'
     * - 'accountdescription'
     * - 'accountorderid'
     * - 'accountstatus'
     * - 'accounttypeid'
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getPicklistAccounts(bool $enabled = true): AcumulusResult
    {
        $filters = [
            'accountstatus' => $enabled ? 1 : 0,
        ];
        return $this->getPicklist('accounts', $filters);
    }

    /**
     * Retrieves a list of invoice templates.
     *
     * See {@link https://www.siel.nl/acumulus/API/Picklists/Company_Types/}.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   a non-keyed array of "companytype" arrays, each "companytype"
     *   array being a keyed array with keys:
     * - 'companytypeid'
     * - 'companytypename'
     * - 'companytypenamenl'
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getPicklistCompanyTypes(): AcumulusResult
    {
        return $this->getPicklist('companytypes', [], false);
    }

    /**
     * Retrieves a list of contact types.
     *
     * See {@link https://www.siel.nl/acumulus/API/Picklists/Contact_Types/}.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   a non-keyed array of "contacttype" arrays, each "contacttype" array
     *   being a keyed array with keys:
     * - 'contacttypeid'
     * - 'contacttypename'
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getPicklistContactTypes(): AcumulusResult
    {
        return $this->getPicklist('contacttypes');
    }

    /**
     * Retrieves a list of cost centers.
     *
     * See {@link https://www.siel.nl/acumulus/API/Picklists/Cost_Centers/}
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   a non-keyed array of "costcenter" arrays, each "costcenter" array being
     *   a keyed array with keys:
     * - 'costcenterid'
     * - 'costcentername'
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getPicklistCostCenters(): AcumulusResult
    {
        return $this->getPicklist('costcenters');
    }

    /**
     * Retrieves a list of invoice templates.
     *
     * See {@link https://www.siel.nl/acumulus/API/Invoicing/Invoice_Templates/}
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   a non-keyed array of "invoicetemplate" arrays, each "invoicetemplate"
     *   array being a keyed array with keys:
     * - 'invoicetemplateid'
     * - 'invoicetemplatename'
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getPicklistInvoiceTemplates(): AcumulusResult
    {
        return $this->getPicklist('invoicetemplates');
    }

    /**
     * Retrieves a list of products.
     *
     * See {@link https://www.siel.nl/acumulus/API/Products/List_Products/}
     *
     * @param string|null $filter
     *   Free search param, checks against the productid, description, type,
     *   SKU, EAN and price of the product.
     * @param int|null $productTagId
     *   To filter products by status:
     *   - null: All (DEFAULT)
     *   - -1: Vervallen (discontinued)
     *   - 0: Actief (active/available)
     *   - 1000: Favoriet (Favorite)
     * @param int|null $offset
     * @param int|null $rowcount
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   a non-keyed array of "product" arrays, each "product"
     *   array being a keyed array with keys:
     *  - 'productid'
     *  - 'productnature'
     *  - 'productdescription'
     *  - 'producttagid'
     *  - 'productcontactid'
     *  - 'productprice'
     *  - 'productvatrate'
     *  - 'productsku'
     *  - 'productstockamount'
     *  - 'productean'
     *  - 'producthash'
     *  - 'productnotes'
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getPicklistProducts(?string $filter = null, ?int $offset = null, ?int $rowcount = null, ?int $productTagId = null): AcumulusResult
    {
        $filters = [];
        if ($filter !== null) {
            $filters['filter'] = $filter;
        }
        if ($productTagId !== null) {
            $filters['producttagid'] = $productTagId;
        }
        if ($offset !== null) {
            $filters['offset'] = $offset;
        }
        if ($rowcount !== null) {
            $filters['rowcount'] = $rowcount;
        }
        return $this->getPicklist('products', $filters);
    }

    /**
     * A helper method to retrieve a given picklist.
     *
     * The Acumulus API for picklists is so well standardized that it is
     * possible to use 1 general picklist retrieval function that can process
     * all picklist types.
     *
     * @param string $picklist
     *   The picklist to retrieve, specify in plural form: accounts,
     *   contacttypes, costcenters, etc.
     * @param array $filters
     *   A set of filters to filter the picklist. Currently, only the Products
     *   picklist supports filters.
     * @param bool $needContract
     *   Whether the contract part needs to be sent with the request.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   a non-keyed array of "picklist" arrays, each 'picklist' array being a
     *   keyed array with keys that depend on the requested picklist.
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    protected function getPicklist(string $picklist, array $filters = [], bool $needContract = true): AcumulusResult
    {
        // For picklists, the main result is found under the name of the
        // picklist but in singular form, i.e. without the s at the end.
        return $this->callApiFunction("picklists/picklist_$picklist", $filters, $needContract)->setMainAcumulusResponseKey($picklist, true);
    }

    /**
     * Retrieves a list of VAT rates for the given country at the given date.
     *
     * See {@link https://www.siel.nl/acumulus/API/Picklists/VAT_Info/}
     *
     * @param string $countryCode
     *   Country code of the country to retrieve the VAT info for.
     * @param \DateTimeInterface|string|null $date
     *   DateTimeInterface object or ISO date string (yyyy-mm-dd) for the date to retrieve
     *   the VAT info for.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   a non-keyed array of "vatinfo" arrays, each "vatinfo" array being a
     *   keyed array with keys:
     *   - vattype: (string) name for the vat rate, most of the time something
     *     like 'normal' or 'reduced'.
     *   - vatrate: (float (as a string)) the vat rate (number between 0 and
     *     100)
     *   - countryregion: (int) one of the Api::Region_... constants.
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getVatInfo(string $countryCode, DateTimeInterface|string|null $date = null): AcumulusResult
    {
        if ($date instanceof DateTimeInterface) {
            $date = $date->format(Api::DateFormat_Iso);
        } elseif (empty($date)) {
            $date = date(Api::DateFormat_Iso);
        }
        $message = [
            'vatcountry' => $countryCode,
            'vatdate' => $date,
        ];
        return $this->callApiFunction('lookups/lookup_vatinfo', $message)->setMainAcumulusResponseKey('vatinfo', true);
    }

    /**
     * Retrieves a report on the threshold for EU commerce.
     *
     * See {@link https://www.siel.nl/acumulus/API/Reports/EU_eCommerce_Threshold/}
     *
     * @param int|null $year
     *   The year to get a report for or null for the current year.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   an array with keys:
     *   - year: int, Year to which information in response is applicable.
     *   - threshold: float, threshold value at which the EU e-Commerce
     *     directive applies.
     *   - nltaxed, float, Amount of turnover taxed using Dutch VAT against target
     *     customer set. Should ideally not gain when the threshold has been reached.
     *   - reached: int, 0 when the threshold has not yet been reached. 1 when so.
     *   Possible errors:
     *   - AAC37EAA: Ongeldig year. EU regelgeving van toepassing vanaf 2021.
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function reportThresholdEuCommerce(?int $year = null): AcumulusResult
    {
        $message = [];
        if ($year !== null) {
            $message['year'] = $year;
        }
        return $this->callApiFunction('reports/report_threshold_eu_ecommerce', $message)->setMainAcumulusResponseKey('');
    }

    /**
     * Sends an invoice to Acumulus.
     *
     * See {@link https://www.siel.nl/acumulus/API/Invoicing/Add_Invoice/}
     *
     * @param \Siel\Acumulus\Data\Invoice $invoice
     *   The invoice to send.
     *
     * @return AcumulusResult
     * The Result of the webservice call. A successful call will contain a
     * response array with keys:
     * - invoice: an array of information about the created invoice, being an
     *   array with keys:
     * - 'invoicenumber'
     * - 'token'
     * - 'entryid'
     * - 'conceptid'
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function invoiceAdd(Invoice $invoice): AcumulusResult
    {
        return $this->callApiFunction('invoices/invoice_add', $invoice->toArray())->setMainAcumulusResponseKey('invoice');
    }

    /**
     * Retrieves information about a concept.
     *
     * See {@link https://www.siel.nl/acumulus/API/Invoicing/Concept_Info/}
     *
     * @param int $conceptId
     *   The id of the concept.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "concept" array, being a keyed array with keys:
     *   - conceptid: int
     *   - entryid: int|int[]
     *   Possible errors:
     *   - FGYBSN040: Requested invoice for concept $conceptId not found: No
     *     definitive invoice has yet been created for this concept.
     *   - FGYBSN048: Information not available for $conceptId older than 127466.
     *   - todo: others?
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getConceptInfo(int $conceptId): AcumulusResult
    {
        $message = [
            'conceptid' => $conceptId,
        ];
        return $this->callApiFunction('invoices/invoice_concept_info', $message)->setMainAcumulusResponseKey('concept');
    }

    /**
     * Retrieves Entry (Boeking) Details.
     *
     * See {@link https://siel.nl/acumulus/API/Entry/Get_Entry_Details/}
     *
     * @param int $entryId
     *   The id of the entry.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "entry" array, being a keyed array with keys:
     * - 'entryid'
     * - 'entrydate'
     * - 'entrytype'
     * - 'entrydescription'
     * - 'entrynote'
     * - 'fiscaltype'
     * - 'vatreversecharge'
     * - 'foreigneu'
     * - 'foreignnoneu'
     * - 'marginscheme'
     * - 'foreignvat'
     * - 'contactid'
     * - 'accountnumber'
     * - 'costcenterid'
     * - 'costtypeid'
     * - 'invoicenumber'
     * - 'invoicenote'
     * - 'descriptiontext'
     * - 'invoicelayoutid'
     * - 'totalvalueexclvat'
     * - 'totalvalue'
     * - 'paymenttermdays'
     * - 'paymentdate'
     * - 'paymentstatus'
     * - 'deleted'
     *   Possible errors:
     *   - XGYBSN000: Requested invoice for entry $entryId not found: $entryId
     *     does not exist.
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function getEntry(int $entryId): AcumulusResult
    {
        $message = [
            'entryid' => $entryId,
        ];
        return $this->callApiFunction('entry/entry_info', $message)->setMainAcumulusResponseKey('entry');
    }

    /**
     * Moves the entry into or out of the recycle bin.
     *
     * See {@link https://siel.nl/acumulus/API/Entry/Set_Delete_Status/}
     *
     * @param int $entryId
     *   The id of the entry.
     * @param int $deleteStatus
     *   The delete action to perform: one of the Api::Entry_Delete or
     *   Api::Entry_UnDelete constants. Api::Entry_UnDelete does not work for
     *   now.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "entry" array, being a keyed array with keys:
     * - 'entryid'
     *   - entryproc: (description new status): 'removed', 'recovered' or 'no
     *     changes made'.
     *   Possible errors:
     *   - XCM7ELO12: Invalid entrydeletestatus value supplied: $deleteStatus
     *     is not one of the indicated constants.
     *   - XCM7ELO14: Invalid entrydeletestatus value supplied: $deleteStatus
     *     is not one of the indicated constants.
     *   - P2XFELO12: Requested for entryid: $entryId not found or forbidden:
     *     $entryId does not exist or already has requested status.
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function setDeleteStatus(int $entryId, int $deleteStatus): AcumulusResult
    {
        $message = [
            'entryid' => $entryId,
            'entrydeletestatus' => $deleteStatus,
        ];
        return $this->callApiFunction('entry/entry_deletestatus_set', $message)->setMainAcumulusResponseKey('entry');
    }

    /**
     * Retrieves the payment status for an invoice.
     *
     * See {@link https://www.siel.nl/acumulus/API/Invoicing/Payment_Get_Status/}
     *
     * @param string $token
     *   The token for the invoice.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "invoice" array, being a keyed array with keys:
     * - 'entryid'
     * - 'token'
     * - 'paymentstatus'
     * - 'paymentdate'
     *   Possible errors:
     *   - XGYTTNF04: Requested invoice for $token not found: $token does not
     *     exist.
     *
     * @throws AcumulusException|AcumulusResponseException
     *
     * @noinspection PhpUnused
     */
    public function getPaymentStatus(string $token): AcumulusResult
    {
        $message = [
            'token' => $token,
        ];
        return $this->callApiFunction('invoices/invoice_paymentstatus_get', $message)->setMainAcumulusResponseKey('invoice');
    }

    /**
     * Sets the payment status for an invoice.
     *
     * See {@link https://www.siel.nl/acumulus/API/Invoicing/Payment_Set_Status/}
     *
     * @param string $token
     *   The token for the invoice.
     * @param int $paymentStatus
     *   The new payment status, 1 of the Api::PaymentStatus_Paid or
     *   Api::PaymentStatus_Due constants.
     * @param string $paymentDate
     *   ISO date string (yyyy-mm-dd) for the date to set as payment date, may
     *   be empty for today, or if the payment status is Api::PaymentStatus_Due.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "invoice" array, being a keyed array with keys:
     * - 'entryid'
     * - 'token'
     * - 'paymentstatus'
     * - 'paymentdate'
     *   Possible errors:
     *   - DATE590ZW: Missing mandatory paymentdate field. Unable to proceed."
     *   - DATE590ZW: Incorrect date range (2000-01-01 to 2099-12-31) or invalid
     *     date format (YYYY-MM-DD) used in paymentdate field. We received:
     *     $paymentDate. Unable to proceed."
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function setPaymentStatus(string $token, int $paymentStatus, string $paymentDate = ''): AcumulusResult
    {
        if (empty($paymentDate)) {
            $paymentDate = date(Api::DateFormat_Iso);
        }
        $message = [
            'token' => $token,
            'paymentstatus' => $paymentStatus,
            'paymentdate' => $paymentDate,
        ];
        return $this->callApiFunction('invoices/invoice_paymentstatus_set', $message)->setMainAcumulusResponseKey('invoice');
    }

    /**
     * Signs up for a 30-day trial and receive credentials.
     *
     * See {@link https://www.siel.nl/acumulus/API/Sign_Up/Sign_Up/}
     *
     * @param array $signUp
     *   An array with the fields:
     *   - companyname (mandatory) Name of company to sign up.
     *   - fullname (mandatory) Full name of person associated with company.
     *   - loginname (mandatory) Preferred login name to be used as credentials
     *     when logging in.
     *   - gender (mandatory) Indication of gender. Used to predefine some
     *     strings within Acumulus.
     *     - F Female
     *     - M Male
     *     - X Neutral
     *   - address (mandatory) Address including house number.
     *   - postalcode (mandatory)
     *   - city (mandatory)
     * - 'telephone'
     *   - bankaccount Preference is to use a valid IBAN-code so Acumulus can
     *     improve preparation of the (trial) sign up.
     *   - email (mandatory)
     *   - createapiuser Include the creation of an additional user specifically
     *     suited for API-usage.
     *     - 0 Do not create additional user (default)
     *     - 1 Generate additional user specifically suited for API-usage
     *   - Notes or remarks which you would like to be part of the sign-up
     *     request. If filled, a ticket will be opened with the notes as
     *     content, so can be used as a request for comment by customer support.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "signup" array, being a keyed array with keys:
     *   - 'contractcode'
     *   - 'contractloginname'
     *   - 'contractpassword'
     *   - 'contractapiuserloginname'
     *   - 'contractapiuserpassword'
     *
     *   Possible errors/warnings:
     *   - AA7E10AA: Verplichte companyname ontbreekt
     *   - AAC8C3AA: Verplichte fullname ontbreekt
     *   - AAFA1AAA: Verplichte loginname ontbreekt
     *   - AAE9CDAA: Verplichte address ontbreekt
     *   - AAC34DAA: Verplichte postalcode ontbreekt
     *   - AA6894AA: Onjuiste postalcode
     *   - AABC1FAA: Verplichte city ontbreekt
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function signUp(array $signUp): AcumulusResult
    {
        $message = [
            'signup' => $signUp,
        ];
        return $this->callApiFunction('signup/signup', $message, false)->setMainAcumulusResponseKey('signup');
    }

    /**
     * Registers a support token for the current webshop.
     *
     * The webshop will be identified by the servername, note that 1 shop may
     * run on multiple domain names and that the actual server name (the one
     * used by this admin user to connect) will be used.
     *
     * @param string $token
     *   The token as received after buying pro-support in the Siel shop.
     *
     * @param string $location
     *   A string that serves to identify the website, typically the domain
     *   name.
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 'support' array, being a keyed array with keys:
     *   - 'token': the token as sent with the request
     *   - 'startdate': the date (yyyy-mm-dd) the support starts, which will be
     *     today.
     *   - 'enddate': the date (yyyy-mm-dd) the support ends, 1 year ahead.
     *   Example:
     *   {
     *     "support": {
     *       "token": "wdo1m8RVt2tcCC3gECNGlzznbm8dsGQN",
     *       "startdate":"2022-06-07",
     *       "enddate":"2023-06-08"
     *     }
     *     ... "basic response fields" ...
     *   }
     */
    public function registerSupport(string $token, string $location): AcumulusResult
    {
        $message = compact('token', 'location');
        return $this->callApiFunction('support/register', $message, true)->setMainAcumulusResponseKey('support');
    }

    /**
     * Updates the stock for a product.
     *
     * See {@link https://www.siel.nl/acumulus/API/Stock/Add_Stock_Transaction/}
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "stock" array, being a keyed array with keys:
     *   - 'productid'
     *   - 'stockamount' (the new stock level for this product)
     *
     *   Possible errors:
     *   - 'AA5B85AA' Product not found
     *   - Not passing a stock amount is not an error, as isn't passing 0.0. The current
     *     stock level will be returned.
     *
     * @throws AcumulusException|AcumulusResponseException
     */
    public function stockTransaction(StockTransaction $stockTransaction): AcumulusResult
    {
        return $this->callApiFunction('stock/stock_add', $stockTransaction->toArray())->setMainAcumulusResponseKey('stock');
    }

    /**
     * Returns the uri to download the invoice PDF.
     *
     * See {@link https://siel.nl/acumulus/API/Invoicing/Get_PDF_Invoice/}
     *
     * @param string $token
     *   The token for the invoice.
     * @param ?bool $reminder
     *   False, null or absent to retrieve the normal invoice, true to retrieve
     *   a reminder invoice.
     * @param ?bool $applyGraphics
     *   False to prevent any embedded graphics from being applied to the
     *   document; true, null, or absent otherwise.
     *
     * @return string
     *   The uri to download the invoice PDF.
     *   Errors (in download from the retunned uri, not in this method's return value):
     *   ```
     *   <response><token>$token</token>
     *   <errors>
     *       <error><code>400 Bad Request</code><codetag>AA69CBAA</codetag><message>Invalid token.</message></error>
     *       <count_errors>1</count_errors>
     *   </errors>
     *   ```
     */
    public function getInvoicePdfUri(string $token, ?bool $reminder = null, ?bool $applyGraphics = null): string
    {
        $uri = $this->constructUri('invoices/invoice_get_pdf');
        $uri .= "?token=$token";
        if ($reminder !== null) {
            $uri .= '&invoicetype=' . ($reminder ? '1' : '0');
        }
        if ($applyGraphics !== null) {
            $uri .= '&gfx=' . ($applyGraphics ? '1' : '0');
        }
        return $uri;
    }

    /**
     * Sends out an invoice or reminder as PDF.
     *
     * See {@link https://siel.nl/acumulus/API/Invoicing/Email/}
     *
     * @param string $token
     *   The token for the invoice.
     * @param EmailInvoiceAsPdf $emailAsPdf
     *   An array with the fields:
     * - 'emailto'
     * - 'emailbcc'
     * - 'emailfrom'
     * - 'subject'
     * - 'message'
     * - 'confirmreading'
     * - 'ubl'
     * @param int|null $invoiceType
     *   One of the constants {@see Api::Email_Normal} (default) or
     *   {@see Api::Email_Reminder}.
     * @param string $invoiceNotes
     *   Multiline field for additional remarks. Use \n for newlines and \t for
     *   tabs. Contents are placed in the notes/comments section of the invoice.
     *   Content will not appear on the actual invoice or associated emails.
     * @param ?bool $applyGraphics
     *   False to prevent any embedded graphics from being applied to the
     *   document; true, null, or absent otherwise.
     *   @todo: not used for now, will become part of emailAsPdf structure?
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "invoice" array, being a keyed array with keys:
     * - 'token'
     * - 'invoicetype'
     *   Possible errors/warnings:
     *   - GK6FKHU52: Incorrect invoicetype value used (9) in invoicetype tag as
     *     part of the invoice section in the XML. Using default value of 0 normal."
     *   - TNFE4035G: Requested token not found or invalid token supplied.
     *     Unable to proceed."
     *
     * @throws AcumulusException|AcumulusResponseException
     *
     * @noinspection PhpUnusedParameterInspection
     */
    public function emailInvoiceAsPdf(
        string $token,
        EmailInvoiceAsPdf $emailAsPdf,
        ?int $invoiceType = null,
        string $invoiceNotes = '',
        ?bool $applyGraphics = null
    ): AcumulusResult
    {
        $message = [
            Fld::Token => $token,
            Fld::EmailAsPdf => $emailAsPdf->toArray(),
        ];
        if ($invoiceType !== null) {
            $message['invoicetype'] = $invoiceType;
        }
        if (!empty($invoiceNotes)) {
            $message['invoicenotes'] = $invoiceNotes;
        }
        return $this->callApiFunction('invoices/invoice_mail', $message)->setMainAcumulusResponseKey('invoice');
    }

    /**
     * Returns the uri to download the packing slip PDF.
     *
     * See {@link https://siel.nl/acumulus/API/Delivery/Get_PDF_Packing_Slip/}
     *
     * @param string $token
     *   The token for the invoice to get the packing slip for.
     * @param ?bool $applyGraphics
     *   False to prevent any embedded graphics from being applied to the
     *   document; true, null, or absent otherwise.
     *
     * @return string
     *   The uri to download the packing slip PDF.
     *   Possible errors (in download, not in return value):
     *   - ZKFATNF04: Requested packing slip for $token not found or no longer
     *     available.
     */
    public function getPackingSlipPdfUri(string $token, ?bool $applyGraphics = null): string
    {
        $uri = $this->constructUri('delivery/packing_slip_get_pdf');
        $uri .= "?token=$token";
        if ($applyGraphics !== null) {
            $uri .= '&gfx=' . ($applyGraphics ? '1' : '0');
        }
        return $uri;
    }

    /**
     * Sends out the packing slip as PDF.
     *
     * See {@link https://siel.nl/acumulus/API/Delivery/Email/}
     *
     * @param string $token
     *   The token for the invoice.
     * @param EmailPackingSlipAsPdf $emailAsPdf
     *   An array with the fields:
     *   - 'emailto'
     *   - 'emailbcc'
     *   - 'emailfrom'
     *   - 'subject'
     *   - 'message'
     * @param string $deliveryNotes
     *   Multiline field for additional remarks. Use \n for newlines and \t for
     *   tabs. Contents are placed in the notes/comments section of the invoice.
     *   Content will not appear on the actual packing slip or associated emails.
     * @param ?bool $applyGraphics
     *   False to prevent any embedded graphics from being applied to the
     *   document; true, null, or absent otherwise.
     *   @todo: not used for now, will become part of emailAsPdf structure?
     *
     * @return AcumulusResult
     *   The result of the webservice call. The structured response will contain
     *   1 "packingslip" array, being a keyed array with keys:
     *   - 'token'
     *
     * @throws AcumulusException|AcumulusResponseException
     *
     * @noinspection PhpUnusedParameterInspection
     */
    public function emailPackingSlipAsPdf(
        string $token,
        EmailPackingSlipAsPdf $emailAsPdf,
        string $deliveryNotes = '',
        ?bool $applyGraphics = null
    ): AcumulusResult {
        $message = [
            Fld::Token => $token,
            Fld::EmailAsPdf => $emailAsPdf->toArray(),
        ];
        if (!empty($deliveryNotes)) {
            $message['deliverynotes'] = $deliveryNotes;
        }
        return $this->callApiFunction('delivery/packing_slip_mail_pdf', $message)->setMainAcumulusResponseKey('packingslip');
    }

    /**
     * Constructs and returns the uri for the requested API service.
     *
     * @param string $apiFunction
     *   The api service to get the uri for.
     *
     * @return string
     *   The uri to the requested API service.
     */
    protected function constructUri(string $apiFunction): string
    {
        $environment = $this->environment->toArray();
        return "{$environment['baseUri']}/{$environment['apiVersion']}/$apiFunction.php";
    }

    /**
     * Wrapper around
     * {@see \Siel\Acumulus\ApiClient\AcumulusRequest::execute()}.
     *
     * For error handling see: {@see AcumulusResult}.
     *
     * @param string $apiFunction
     *   The API function to invoke.
     * @param array $message
     *   The values to submit.
     * @param bool $needContract
     *   Indicates whether this api function needs the contract details. Most
     *   API functions do, do the default is true, but for some general listing
     *   functions, like vat info, it is optional, and for signUp, it is even
     *   not allowed.
     *
     * @return AcumulusResult
     *   An AcumulusResult object containing the results.
     *
     * @throws AcumulusException|AcumulusResponseException
     *   A communication level error occurred:
     *   - {@see AcumulusRequest} will be set;
     *   - {@see HttpRequest} will probably also be set;
     *   - {@see HttpResponse} might be set or not;
     *   - {@see AcumulusResult} will not be set.
     */
    protected function callApiFunction(string $apiFunction, array $message, bool $needContract = true): AcumulusResult
    {
        $acumulusRequest = $this->createAcumulusRequest();
        $uri = $this->constructUri($apiFunction);
        try {
            $acumulusResult = $acumulusRequest->execute($uri, $message, $needContract);
            $logLevel = $acumulusResult->getStatus();
            return $acumulusResult;
        } catch (AcumulusException $e) {
            // Situation 1 or 2. We will log the situation and rethrow.
            $exception = $e;
            throw $e;
        } finally {
            $logLevel = $logLevel ?? Severity::Exception;
            $this->log->log($logLevel, $acumulusRequest->getMaskedRequest());
            if (isset($acumulusResult)) {
                $this->log->log($logLevel, $acumulusResult->getMaskedResponse());
            }
            if (isset($exception)) {
                $this->log->exception($exception);
            }
        }
    }

    /**
     * Wrapper around the factory method that creates an AcumulusRequest.
     *
     * @return \Siel\Acumulus\ApiClient\AcumulusRequest
     */
    protected function createAcumulusRequest(): AcumulusRequest
    {
        return $this->container->createAcumulusRequest();
    }
}
