# ===================================================================================
# Get-BPServerUpdateStatus
# Author: Aleksandar Draskovic
# Date: 2015-05-15
# ===================================================================================

<#
	.SYNOPSIS
		Checks the available Windows Updates and SharePoint patch level


	.DESCRIPTION
		Checks all SharePoint servers in the farm for the following:
			- Number of available Windows Updates for the installation
			- SharePoint patch level
			- SharePoint language packs patch level

		Requires WinRM configured and allowed on all SharePoint servers in the farm.
	
	.PARAMETER WindowsUpdateCheck
		Specifies the method of determining the status of Windows Updates installation. Valid values:
		
		Disable
		Do not check the status of Windows Updates
		
		Quick
		Check the date and time of last Windows Update check and installation
		
		Detailed
		Check all missing updates on all SharePoint servers. This option will provide the most reliable results but it will also take the most of time to complete.
		
	.EXAMPLE
		Get-BPServerUpdateStatus -WindowsUpdateCheck Detailed
        
        Iterates through all SharePoint servers in the farm and checks update status. Performs a detailed Windows Update check.
	
	.LINK
	SharePoint 2013 build numbers
	http://www.toddklindt.com/blog/Lists/Posts/Post.aspx?ID=346
	
	SharePoint 2010 build numbers
	http://www.toddklindt.com/blog/Lists/Posts/Post.aspx?ID=224
#>

param(
	[ValidateSet("Disable","Quick","Detailed")]
	[string]$WindowsUpdateCheck="Quick")

function Get-SPServerList
{
	$spServerList=@()
	$serverList = Get-SPServer
	foreach ($server in $serverList)
	{
		if ($server.Role -ne [Microsoft.SharePoint.Administration.SPServerRole]::Invalid)
		{
			$spServerList += $server.Address
		}
	}
	return $spServerList
}

function Get-WinUpdateLastInstalledHotfixDate([string]$serverName)
{
	$result = Get-HotFix -ComputerName $serverName -ErrorAction SilentlyContinue | Measure-Object InstalledOn -Maximum
	if (![string]::IsNullOrEmpty($result)) { $result = $result.Maximum.ToString("yyyy-MM-dd hh:mm:ss") }
	return $result
}

function Get-WinUpdateLastCheckDate([string]$serverName)
{
	$result = Invoke-Command -ComputerName $serverName -ScriptBlock {Get-ItemProperty -Path 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WindowsUpdate\Auto Update\Results\Detect' -Name LastSuccessTime -ErrorAction SilentlyContinue | select -ExpandProperty LastSuccessTime}
	return $result	
}

function Get-WinUpdateLastInstallDate([string]$serverName)
{
	$result = Invoke-Command -ComputerName $serverName -ScriptBlock {Get-ItemProperty -Path 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WindowsUpdate\Auto Update\Results\Install' -Name LastSuccessTime -ErrorAction SilentlyContinue | select -ExpandProperty LastSuccessTime}
	return $result
}

function Get-AvailableWindowsUpdatesCount([string]$serverName)
{
	$numUpdates = Invoke-Command -ComputerName $serverName -ScriptBlock {$Searcher = New-Object -ComObject Microsoft.Update.Searcher; $results = $searcher.search("Type='software' AND IsInstalled = 0 AND IsHidden = 0"); $results.Updates.Count} -ErrorAction SilentlyContinue
	if ([string]::IsNullOrEmpty($numUpdates))
	{
		$numUpdates = -1
	}
	return $numUpdates
}

function Get-SPFarmVersion
{
	return (Get-SPFarm).BuildVersion.ToString()
}

# based on the script by Stefan Goßner
# for more information, please visit http://blogs.technet.com/b/stefan_gossner/archive/2015/04/20/powershell-script-to-display-version-info-for-sharepoint-product-and-language-packs.aspx
function Get-SPProductsBuild([string]$serverName)
{
	$farmVersion = (Get-SPFarm).BuildVersion.ToString()
	# Get SharePoint Products and language packs
	
	$programs = Invoke-Command -ComputerName $serverName -ScriptBlock {$regLoc = Get-ChildItem "HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall" ; $RegLoc | where-object { $_.PsPath -like "*\Office*" } | foreach {Get-ItemProperty $_.PsPath}}

	$consoleForegroundColor = [console]::ForegroundColor
	$Programs | foreach {  
		$_ | Select  DisplayName, DisplayVersion | foreach { if ( $_.DisplayVersion.Trim() -ne $farmVersion ) { [console]::ForegroundColor="Red"} else {[console]::ForegroundColor=$consoleForegroundColor} $_ }; 
	}
	
	[console]::ForegroundColor = $consoleForegroundColor
}

function Check-WindowsUpdatesDetailed([string]$serverName)
{
	Write-Host "`t Available Windows Updates: " -NoNewLine
	$numWU = Get-AvailableWindowsUpdatesCount $serverName
	if ($numWU -gt 0)
	{
		$fc = "Yellow"
	}
	else
	{
		$fc = "Green"
	}
	if ($numWU -eq -1)
	{
		$fc = "Red"
		$numWU = "Failed to connect"
	}
	Write-Host $numWU -ForegroundColor $fc
}

function Check-WindowsUpdatesQuick([string]$serverName)
{
	$wuLastCheck = Get-WinUpdateLastCheckDate $serverName
	$wuLastInstall = Get-WinUpdateLastInstallDate $serverName
	$fcC = $fcI = [console]::ForegroundColor
	
	if ([string]::IsNullOrEmpty($wuLastCheck))
	{
		$wuLastCheck = "Unknown"
		$fcC = "Red"
	}
	
	if ([string]::IsNullOrEmpty($wuLastInstall))
	{ 
		$wuLastInstall = Get-WinUpdateLastInstalledHotfixDate $serverName
		if ([string]::IsNullOrEmpty($wuLastInstall))
		{
			$fcI = "Red"
			$wuLastInstall = "Unknown"
		}
	}
	
	Write-Host "`t Last Windows Update check: " -NoNewLine
	Write-Host "$wuLastCheck" -ForegroundColor $fcC
	Write-Host "`t Last Windows Update installation: " -NoNewLine
	Write-Host "$wuLastInstall" -ForegroundColor $fcI
	
}

function Get-SPServerNeedingUpgrade
{
	$serverList = Get-SPServer | Where{ ($_.NeedsUpgrade -eq $TRUE) -and ($_.Role -ne [Microsoft.SharePoint.Administration.SPServerRole]::Invalid)}
	if ([string]::IsNullOrEmpty($serverList))
	{
		Write-Host "None." -ForegroundColor Green
	}
	else
	{
		$serverList | ft -a
	}
}

### Main

$spServerList = Get-SPServerList

Write-Host "SharePoint farm build $((Get-SPFarm).BuildVersion.ToString())" -ForegroundColor Yellow
Write-Host ""

switch ($WindowsUpdateCheck)
{
	"Disable"		{ Write-Host "Skipping Windows Update check" -ForegroundColor Yellow }
	"Quick"			{ Write-Host "Last Windows Update check and installation time will be retrieved." -ForegroundColor Yellow }
	"Detailed"		{ Write-Host "Detailed Windows Update check will be performed. This may take a long time." -ForegroundColor Yellow }
}

foreach ($server in $spServerList)
{
	Write-Host "=================================================" -ForegroundColor Green
	Write-Host "Server: $server" -ForegroundColor Green
	Write-Host "=================================================" -ForegroundColor Green
	
	switch ($WindowsUpdateCheck)
	{
		"Disable"		{  }
		"Quick"			{ Check-WindowsUpdatesQuick $server }
		"Detailed"		{ Check-WindowsUpdatesQuick $server; Check-WindowsUpdatesDetailed $server }
	}
	
	Write-Host ""
	Write-Host "Installed binaries:"
	Get-SPProductsBuild $server
	
	Write-Host ""
}

Write-Host "Following SharePoint servers must be upgraded via PSCONFIG: " -ForegroundColor Yellow
Get-SPServerNeedingUpgrade
