/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
package com.urbancode.air.plugin.scm

import com.urbancode.air.*
import com.urbancode.air.plugin.scm.changelog.*

import java.util.Date;
import java.util.TimeZone;
import java.text.SimpleDateFormat


public class SCMQuietPeriod extends SCMChangelog {

    //**************************************************************************
    // CLASS
    //**************************************************************************

    //**************************************************************************
    // INSTANCE
    //**************************************************************************

    /**
     *
     * @return date of the last change
     */
    public def execute() {
        def latestChangeDate
        boolean error = false
        // Login if not authenticated
        this.login();
        
        try {
            // Get Change Log Output
            def logOutput = runLogCommand(streamName)
    
            // Parse Change Log for most recent changeset
            latestChangeDate = determineLatestChangeDate(logOutput)
        }
        catch (ExitCodeException e) {
            println e.getMessage()
            error = true
        }
        finally {
            this.logout()
            
            if (error) {
                System.exit(1)
            }
        }
        
        return latestChangeDate
    }
        
    private def determineLatestChangeDate(logOutput) {
        // the result change set date
        Date latestChangeDate = null;

        groovy.util.slurpersupport.GPathResult log = new XmlSlurper().parseText(logOutput);

        log.transaction.each{ trans ->
            ChangeSet changeSet = new ChangeSet();
            changeSet.id = trans.@id.text();
            changeSet.user = trans.@user.text();
            changeSet.date = new Date((trans.@time.text() as Long) * 1000);
            changeSet.message = trans.comment.text();

            def changeSetType = trans.@type.text()

            trans.version.each{ ver ->
                def path = ver.@path.text() ;

                // Accurev seems to prefix all of their file paths with
                // /./ so I am removing it here for consistency with the rest
                // of our file paths.
                path = path.replace('^/\\./', '');

                ChangeSetFileEntry entry = new ChangeSetFileEntry();
                entry.revision = changeSet.id;
                entry.type = changeSetType;
                entry.path = path;
                changeSet.fileSet << entry;
            }

            boolean allowAuthor = changeSet.hasAllowedAuthor(changeSetFilter);
            boolean allowAnyPaths = changeSet.hasAllowedPath(changeSetFilter);
            if (allowAuthor && allowAnyPaths) {
                latestChangeDate = changeSet.date;
            }
            else {
                def message = new StringBuilder("Changeset ${changeSet.id} skipped because ")
                if (!allowAuthor) {
                    message << "it has excluded author ${changeSet.user}"
                }
                if (!allowAuthor && !allowAnyPaths) {
                    message << " and "
                }
                if (!allowAnyPaths) {
                    message << "it contains only excluded file paths (${changeSet.fileSet.collect{it.path}})"
                }
                println message
            }
        }

        return latestChangeDate;
    }
}
