#!/usr/bin/env groovy

final String lineSep = System.getProperty('line.separator')
final String osName = System.getProperty('os.name').toLowerCase(Locale.US)
final String pathSep = System.getProperty('path.separator')
final boolean windows = (osName =~ /windows/)
final boolean vms = (osName =~ /vms/)
final boolean os9 = (osName =~ /mac/ && !osName.endsWith('x'))
final boolean unix = (pathSep == ':' && !vms && !os9)

final def workDir = new File('.').canonicalFile
final def props = new Properties();
final def inputPropsFile = new File(args[0]);
final def inputPropsStream = null;
try {
    inputPropsStream = new FileInputStream(inputPropsFile);
    props.load(inputPropsStream);
}
catch (IOException e) {
    throw new RuntimeException(e);
}
finally {
    inputPropsStream.close();
}

final def directoryOffset = props['directoryOffset']
final def antFileName = props['antFileName']
final def targetNames = props['targetNames']
final def antProperties = props['antProperties']
final def jvmProperties = props['jvmProperties']
final def properties = props['properties']
final def scriptContent = props['scriptContent']
final def ANT_HOME = props['antHome']?.trim()
final def JAVA_HOME = props['javaHome']

final def verbose = false

//
// Validation
//

if (directoryOffset) {
    workDir = new File(workDir, directoryOffset).canonicalFile
}

if (workDir.isFile()) {
    throw new IllegalArgumentException("Working directory ${workDir} is a file!")
}

if (antFileName == null) {
    throw new IllegalStateException("Ant Script File not specified.");
}

if (ANT_HOME == null) {
    throw new IllegalStateException("ANT_HOME not specified");
}

if (JAVA_HOME == null) {
    throw new IllegalStateException("JAVA_HOME not specified");
}

//
// Create workDir and antFile
//

// ensure work-dir exists
workDir.mkdirs()


final def antFile = new File(workDir, antFileName)

// check if script content was specified, if so, we need to write out the file
boolean deleteOnExit = false
if (scriptContent) {
    antFile.text = scriptContent
    deleteOnExit = true
}

int exitCode = -1;
try {
    def antExe = new File(ANT_HOME, "bin/ant" + (windows ? ".bat" : ""))
    
    //
    // Build Command Line
    //
    def commandLine = []
    commandLine.add(antExe.absolutePath)
    
    if (antProperties) {
        antProperties.readLines().each() { antProperty ->
            if (antProperty) {
                commandLine.add(antProperty)
            }
        }
    }

    commandLine.add("-f")
    commandLine.add(antFile.absolutePath)
    
    if (targetNames) {
        targetNames.split("\\s").each() { targetName ->
            if (targetName) {
                commandLine.add(targetName)
            }
        }
    }
    
    if (properties) {
        properties.split("\\s").each() { property ->
            if (property) {
                if (!property.startsWith("-D")) {
                    property = "-D" + property
                }
                commandLine.add(property)
            }
        }
    }

    
    // print out command info
    println("")
    println("command line: ${commandLine.join(' ')}")
    println("working directory: ${workDir.path}")
    println('===============================')
    println("command output: ")

    //
    // Launch Process
    //
    final def processBuilder = new ProcessBuilder(commandLine as String[]).directory(workDir)
    processBuilder.environment().put("ANT_HOME", ANT_HOME)
    processBuilder.environment().put("JAVA_HOME", JAVA_HOME)
    processBuilder.environment().put("JAVACMD", "") // ant.bat uses this over the JAVA_HOME
    final def process = processBuilder.start()
    process.out.close() // close stdin
    process.waitForProcessOutput(System.out, System.err) // forward stdout and stderr
    process.waitFor()
    
    // print results
    println('===============================')
    println("command exit code: ${process.exitValue()}")
    println("")
    
    exitCode = process.exitValue();
}
finally {
    if (deleteOnExit) {
        antFile.delete()
    }
}
System.exit(exitCode);
