/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Deploy
 * (c) Copyright IBM Corporation 2014. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */

package com.urbancode.air.plugin.artifactory

import java.io.File

import groovy.json.JsonSlurper

import java.security.NoSuchAlgorithmException

import org.apache.commons.lang.ObjectUtils
import org.apache.commons.lang.StringUtils;

import org.apache.http.HttpResponse
import org.apache.http.HttpStatus
import org.apache.http.client.methods.HttpGet
import org.apache.http.client.methods.HttpPut
import org.apache.http.entity.StringEntity
import org.apache.http.entity.FileEntity

import org.apache.http.impl.client.DefaultHttpClient
import org.apache.http.util.EntityUtils

import com.urbancode.commons.fileutils.FileUtils
import com.urbancode.commons.fileutils.digest.DigestUtil
import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder
import com.urbancode.commons.util.StringUtil

import org.jfrog.artifactory.client.ArtifactoryClient
import org.jfrog.artifactory.client.Artifactory

import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

public class ArtifactoryHelper {
    private static final HttpClientBuilder builder = new HttpClientBuilder();
    private static DefaultHttpClient client;
    private static final ARTIFACT_FILE_HASH_ALGORITHM = "sha1";
    //This is Artifactory default built-in user and password
    private static def userName = "admin";
    private static def passWord = "password";

    static {
        LogManager.resetConfiguration();
        Logger rootLogger = Logger.getRootLogger();
        rootLogger.setLevel(Level.WARN);

        Logger wireLogger = Logger.getLogger("org.apache.http.wire");
        Logger headerLogger = Logger.getLogger("org.apache.http.headers");
        wireLogger.setLevel(Level.WARN);
        headerLogger.setLevel(Level.WARN);
    }

    public static HttpClientBuilder getBuilder() {
        return builder;
    }

    public static void createClient() {
        client = builder.buildClient();
    }

    public static void createClient(username, password) {
        builder.setPreemptiveAuthentication(true);
        builder.setUsername(username);
        builder.setPassword(password);
        client = builder.buildClient();

        userName = username;
        passWord = password;
    }

    public static DefaultHttpClient getClient() {
        return client;
    }

    public static void verifyHash (File fileToVerify, storedDigest) {
        if (storedDigest != null) {
            String computedDigest;
            try {
                computedDigest = DigestUtil.getHexDigest(fileToVerify, ARTIFACT_FILE_HASH_ALGORITHM);
                if (!ObjectUtils.equals(storedDigest, computedDigest)) {
                    throw new Exception("Artifact file verification of " + fileToVerify.getName() +
                    " failed. Expected digest of " + storedDigest + " but the downloaded file was " + computedDigest);
                }
            }
            catch (NoSuchAlgorithmException e) {
                throw new Exception("Algorithm to verify Maven remote artifacts not supported: " +
                ARTIFACT_FILE_HASH_ALGORITHM);
            }
            catch (IOException e) {
                throw new Exception("Error verifying downloaded Maven remote artifacts: " +
                e.getMessage(), e);
            }
        }
    }

    public static File downloadFileFromRepo(String url, String checkHash) {
        HttpGet get = new HttpGet(url);
        HttpResponse response = client.execute(get);
        int status = response.getStatusLine().getStatusCode();
        if (status == HttpStatus.SC_OK) {
            def jsonString = EntityUtils.toString(response.getEntity());
            def slurper = new JsonSlurper();
            def infoJSON = slurper.parseText(jsonString);
            def checksumMap = infoJSON.checksums;
            def downloadUrl = infoJSON.downloadUri;
            System.out.println("Downloading: " + downloadUrl);
            get = new HttpGet(downloadUrl);
            response = client.execute(get);
            status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_OK) {
                String tempFileSuffix = ".maven2";
                int extIndex = url.lastIndexOf(".");
                if (extIndex >= 0) {
                    tempFileSuffix = url.substring(extIndex);
                }
                File artifactFile = File.createTempFile("maven2-", tempFileSuffix);
                FileUtils.writeInputToFile(response.getEntity().getContent(), artifactFile);
                if (checkHash && Boolean.valueOf(checkHash)) {
                    //verify checksum
                    verifyHash(artifactFile, checksumMap.sha1);
                    System.out.println("Verification for file : " + artifactFile + " : succeeded!");
                }
                return artifactFile;
            } else {
                throw new Exception("Exception downloading file : " + downloadUrl + "\nErrorCode : " + status.toString());
            }
        }
    }

    public static void uploadFileToRepo(String baseUrl, String repoName, String targetPath, File baseDir, String fileExtName) {
        ArrayList fileList = findFiles(baseDir, fileExtName);
        for (File file : fileList) {
            String uploadURL = baseUrl + "/" + repoName + "/" + targetPath + "/"+ file.getName()
            println "Upload file:" + file.getName() + " to path:" + targetPath + " of repository:" + repoName + " by URL:${uploadURL}"

            Artifactory artifactory = ArtifactoryClient.create(baseUrl, userName, passWord);
            org.jfrog.artifactory.client.model.File result = artifactory.repository(repoName)
                    .upload(targetPath + "/"+ file.getName(), file).doUpload();
            String printMsg = "Upload Successfully. DownloadUrl is " + result.getDownloadUri();
            println printMsg;
        }
    }

    private static ArrayList findFiles(File baseDir, String fileExtName) {
        ArrayList<File> fileList = new ArrayList<File>();

        if ((!baseDir.exists()) || baseDir.isFile()) {
            throw new Exception("The baseDir directory does not exist or it is a file");
        }
        else {
            File[] sourceFiles = baseDir.listFiles();
            for(File file : sourceFiles){
                if (file.isFile() && file.getName().endsWith(fileExtName)) {
                    fileList.add(file);
                }
            }
        }

        return fileList;
    }
}
