/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Build
 * (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */


import org.apache.http.client.HttpClient
import org.apache.http.client.methods.HttpPost
import org.apache.http.entity.StringEntity
import org.apache.http.impl.client.DefaultHttpClient
import java.lang.Double

import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder
import com.urbancode.commons.util.IO

public class CloverCoverageXMLHelper {

    //**************************************************************************
    // CLASS
    //**************************************************************************

    //**************************************************************************
    // INSTANCE
    //**************************************************************************
    private def testCount = 0
    private def suiteCount = 0

    public CloverCoverageXMLHelper() {
    }

    def uploadResult (reportName, buildLifeId, coverageXmlFile) {
        def testsuiteName
        def xml
        def result
        def name
        def time
        testCount = 0
        suiteCount = 0

        XmlParser parser = new XmlParser()
        parser.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
        parser.setFeature("http://xml.org/sax/features/namespaces", false)
        def rootNode = parser.parse(coverageXmlFile)

        def reportXml = new java.io.StringWriter()
        new groovy.xml.MarkupBuilder(reportXml).
            "coverage-report"(
                "name": reportName,
                "type": 'Clover',
                "build-life-id": buildLifeId,
                "line-percentage": calculatePercentage(rootNode.project.metrics.@"coveredelements",
                    rootNode.project.metrics.@"elements"),
                "method-percentage": calculatePercentage(rootNode.project.metrics.@"coveredmethods",
                    rootNode.project.metrics.@"methods"),
                "branch-percentage": calculatePercentage(rootNode.project.metrics.@"coveredconditionals",
                    rootNode.project.metrics.@"conditionals"),
                "complexity": "0"
            ) {
            rootNode.project."file".each { fileNode ->
                "coverage-group"(
                    "name": fileNode.@"name",
                    "line-percentage": calculatePercentage(fileNode.metrics.@"coveredelements",
                        fileNode.metrics.@"elements"),
                    "method-percentage": calculatePercentage(fileNode.metrics.@"coveredmethods",
                        fileNode.metrics.@"methods"),
                    "branch-percentage": calculatePercentage(fileNode.metrics.@"coveredconditionals",
                        fileNode.metrics.@"conditionals"),
                    "complexity": "0"
                )
            }
        }

        if (reportXml) {
            sendPostRequest(reportName, reportXml.toString())
        }
        else {
            println 'No report was able to be generated'
        }
    }

    private String calculatePercentage(def covered, def total) {
        double percentage = 0
        def numCovered = Double.parseDouble(covered.get(0))
        def numTotal = Double.parseDouble(total.get(0))
        if(numTotal > 0) {
            percentage = numCovered / numTotal
        }
        println("covered: " + numCovered)
        println("total: " + numTotal)
        println("percentage: " + percentage)
        return percentage.toString()
    }

    private void sendPostRequest(String name, String xml) {
        def authToken = System.getenv("AUTH_TOKEN")
        String buildLifeId = System.getenv("BUILD_LIFE_ID")
        name = encode(name)

        // construct the URL with property replacements
        String baseUrl = System.getenv("WEB_URL")

        baseUrl += baseUrl.endsWith("/") ? "" : "/"
        String url = baseUrl + "rest/buildlife/${buildLifeId}/testcoverage?reportName=${name}"

        println "Sending request to $url"

        HttpClientBuilder clientBuilder = new HttpClientBuilder();
        clientBuilder.setTrustAllCerts(true);
        DefaultHttpClient client = clientBuilder.buildClient();

        HttpPost postMethod = new HttpPost(url);
        if (authToken) {
            postMethod.setHeader("Authorization-Token", authToken)
            postMethod.setHeader("Content-Type", "application/xml")
        }

        postMethod.setEntity(new StringEntity(xml));

        def httpResponse = client.execute(postMethod)
        def responseCode = httpResponse.getStatusLine().getStatusCode()
        if (isGoodResponseCode(responseCode)) {
            IO.copy(postMethod.getEntity().getContent(), System.out)
            println ""
        }
        else {
            IO.copy(postMethod.getEntity().getContent(), System.err)
            throw new RuntimeException("Failed to upload Clover coverage report. StatusCode: ${responseCode}")
        }
    }

    private boolean isGoodResponseCode(int responseCode) {
        return responseCode >= 200 && responseCode < 300;
    }

    private def encode = {
        return !it ? it : new java.net.URI(null, null, it, null).toASCIIString()
    }
}