#!/usr/bin/env groovy
/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

import java.io.IOException;

import com.urbancode.air.*
import com.urbancode.ubuild.codestation.client.*

import org.apache.log4j.ConsoleAppender
import org.apache.log4j.Level
import org.apache.log4j.Logger
import org.apache.log4j.LogManager
import org.apache.log4j.PatternLayout


System.setProperty("log4j.defaultInitOverride", "true")
LogManager.resetConfiguration()
Logger.getRootLogger().setLevel(Level.ERROR)
Logger.getRootLogger().addAppender(new ConsoleAppender(new PatternLayout("%m%n")))
Logger.getLogger("com.urbancode.ubuild.codestation").setLevel(Level.DEBUG)
Logger.getLogger("com.urbancode.vfs.client").setLevel(Level.INFO)

final def workDir = new File('.').canonicalFile
final def apTool = new AirPluginTool(this.args[0], this.args[1]);

final def props = apTool.getStepProperties();

final def projectName       = props['projectName']
final def workflowName      = props['workflowName']
final def buildLifeId       = props['buildLifeId']
//final def fileFilter        = props['fileFilter']
//final def artifactFilter    = props['artifactFilter']
final def permissions       = props['permissions']
final def useCache          = Boolean.valueOf(props['useCache'])

try {
    def url = System.getenv("WEB_URL")
    def authToken = System.getenv("AUTH_TOKEN")
    def userHomeDir  = new File(System.getProperty("user.home"))
    def cacheDir = new File(userHomeDir, ".codestation/cache")
    def credentials = AuthenticationCredentials.createAuthTokenCredentials(authToken)
    def codestationBaseUrl = url.endsWith("/") ? url + "rest2" : url + "/rest2"
    def codestationClient = new CodestationClient(codestationBaseUrl, credentials)
    
    if (useCache) {
        println "Using CodeStation cache at ${cacheDir}"
        codestationClient.setCacheDir(cacheDir)
    }
    else {
        println "NOT using CodeStation cache"
    }

    if ("SET_ALL".equals(permissions)) {
        codestationClient.setSetPermissions(true)
    }
    else if ("SET_EXECUTE_ONLY".equals(permissions)) {
        codestationClient.setExecuteBitsOnly(true)
    }

    if (buildLifeId) {
        println "Downloading dependencies for build life ${buildLifeId}"
        def buildLife = codestationClient.buildLifeLookup.findBuildLife(buildLifeId)
        codestationClient.downloadDependencies(buildLife, workDir)
    }
    else {
        println "Downloading latest dependencies for ${projectName} - ${workflowName}"
        def buildLife = codestationClient.buildLifeLookup.findLatestBuildLife(projectName, workflowName, null, null, null)
        codestationClient.downloadDependencies(buildLife, workDir)
    }
}
catch (IOException e) {
    String message = "An error occurred accessing the CodeStation Repository at " + System.getenv("WEB_URL") + ". It is " +
        "defined in the System - Server Settings page as the Agent External URL value. If this URL is incorrect, " +
        "please change it. Detailed error message: " + e.toString();
    throw new Exception(message, e);
}
catch (Exception e) {
    String message = "Error retrieving dependencies from CodeStation: " + e.toString();
    e.printStackTrace();
    throw new Exception(message, e);
}