package com.urbancode.air.plugin.bugs;

import groovy.util.slurpersupport.GPathResult
import java.util.zip.ZipEntry
import java.util.zip.ZipFile
import com.urbancode.commons.xml.XPathUtils
import org.w3c.dom.Document
import org.w3c.dom.Element
import org.w3c.tidy.Tidy

public class FindBugsHandler extends org.xml.sax.helpers.DefaultHandler {

    def tagNameStack = new Stack()
    def findingId
    def findingFile
    def findingLine
    def findingName
    def findingSeverity
    def findingDesc
    def addFinding
    
    static String BUG_DESCRIPTIONS_FILE_PATH = "/doc/allBugDescriptions.html"
    Element bugDescriptionsElement
    
    Map<String, String> namesToDescriptions = new HashMap<String, String>()

    FindBugsHandler(def addFinding, File findbugsZip) {
        this.addFinding = addFinding
        
        if (findbugsZip) {
            // The FindBugs zip files always contain a root folder that matches the version, so we need extract that information
            String zipPath = findbugsZip.absolutePath
            //Strip out the path to the zip file
            String findbugsVersion = zipPath.substring(zipPath.lastIndexOf(File.separator) + 1)
            //Strip out the file extension
            findbugsVersion = findbugsVersion.substring(0, findbugsVersion.lastIndexOf("."))
            BUG_DESCRIPTIONS_FILE_PATH = findbugsVersion + BUG_DESCRIPTIONS_FILE_PATH
            
            ZipFile findbugsZipFile = new ZipFile(findbugsZip)
            ZipEntry bugDescriptionsEntry = findbugsZipFile.getEntry(BUG_DESCRIPTIONS_FILE_PATH)
            InputStream bugDescriptionsStream = findbugsZipFile.getInputStream(bugDescriptionsEntry)
            
            // Clean up any HTML and convert to DOM.
            Tidy tidy = new Tidy()
            //  We don't care about any logging while parsing the HTML document
            tidy.setShowWarnings(false)
            tidy.setShowErrors(0);
            tidy.setQuiet(true)
            Document document = tidy.parseDOM(bugDescriptionsStream, null)
            bugDescriptionsElement = document.getDocumentElement()
        }
    }

    void startElement(String ns, String localName, String qName, org.xml.sax.Attributes attribs) {
        switch (qName) {

            case 'BugInstance':
                findingId = attribs.getValue('instanceHash')
                findingName = attribs.getValue('type')
                findingSeverity = attribs.getValue('category')
                findingLine = null
                break

            case 'ShortMessage':
                findingDesc = new StringBuilder()
                break

            case 'SourceLine':
                if (tagNameStack.peek() == 'Class') {
                    findingFile = attribs.getValue('sourcepath')
                }
                else if (tagNameStack.peek() == 'BugInstance' && findingLine == null) {
                    findingLine = attribs.getValue('start')
                }
                break
        }
        tagNameStack.push(qName)
    }

    void characters(char[] ch, int start, int length) {
        if (tagNameStack.peek() == 'ShortMessage') {
            String desc = getDescriptionForName(findingName)
            if (desc) {
                findingDesc.append(desc);
            }
            else {
                findingDesc.append(ch, start, length)
            }
            
        }
    }

    void endElement(String ns, String localName, String qName) {
        tagNameStack.pop()
        switch (qName) {

            case 'BugInstance':
                addFinding(findingId, findingFile, findingLine, findingName, findingSeverity, findingDesc, '')
                break
        }
    }
    
    private String getDescriptionForName(String name) {
        String result 
        try {
            result = namesToDescriptions.get(name)
            if (!result && bugDescriptionsElement) {
                XPathUtils xpath = new XPathUtils()
                String xPathStr = "//h3/a[@name=${xpath.quotedXPathString(name)}]/../following-sibling::p"
                result = xpath.getXPathText(bugDescriptionsElement, xPathStr)
                // Remove <p> and </p> from the resulting text
                result.replace("</?p>", "");
                if (result) {
                    namesToDescriptions.put(name, result)
                }
            }
        }
        catch (Exception e) {
            // Swallow to allow the plugin to continue with shorter descriptions
        }
        return result;
    }
}
