package com.urbancode.air.plugin.scm

import com.urbancode.air.*
import com.urbancode.air.plugin.scm.changelog.*;

import org.apache.commons.httpclient.HttpClient
import org.apache.commons.httpclient.UsernamePasswordCredentials
import org.apache.commons.httpclient.auth.AuthScope
import org.apache.commons.httpclient.methods.StringRequestEntity
import org.apache.commons.httpclient.methods.PostMethod
import org.apache.commons.httpclient.params.HttpClientParams
import org.apache.commons.httpclient.protocol.Protocol
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory

import com.urbancode.commons.util.IO
import com.urbancode.commons.util.https.OpenSSLProtocolSocketFactory

import java.util.Date;
import java.util.TimeZone;
import java.util.regex.Pattern;
import java.text.SimpleDateFormat


public class SCMChangelog extends SCMStep {

    //**************************************************************************
    // CLASS
    //**************************************************************************

    //**************************************************************************
    // INSTANCE
    //**************************************************************************

    // changelog boundaries
    Date startDate;
    Date endDate;
    String startRevision;
    String endRevision;
    String changesUrl;

    // filters for changeset
    final ChangeSetFilter changeSetFilter = new ChangeSetFilter();
    List<ChangeSet> changeSets;

    /**
     *
     * @return xml document of changes
     */
    public def execute() {
        initWorkspace()

        // Get Change Log
        def logOutput = runLogCommand()

        // the result change set data
        changeSets = parseLogIntoChangeSets(logOutput);

        // Construct XML message and upload
        if (changeSets) {
            ChangeSetXMLHelper xmlHelper = new ChangeSetXMLHelper()
            xmlHelper.repoType = REPO_TYPE;
            String xml = xmlHelper.toXML(changeSets);
            sendPostRequest(xml);
        }
        else {
            println "No changes detected"
            return null
        }
    }

    /**
     * Parse an input of changelog command into a list of Air ChangeSets
     * @param logOutput output from a {@link #runLogCommand()} method
     * @return List of ahp changesets
     */
    public List<ChangeSet> parseLogIntoChangeSets(String logOutput) {

        List<ChangeSet> changeSets = []

        // parse output
        def entrySepPattern = '\0|'+UNIT_SEPARATOR;
        for (String entry : logOutput.split("(?:$entrySepPattern)\0")) {
            if (!entry) {
                // skip empty string - usually start or end of log output
                continue;
            }
            String[] chunks = entry.split(entrySepPattern);

            ChangeSet changeSet = new ChangeSet();
            changeSet.id = chunks[0];
            changeSet.user = chunks[1];
            changeSet.date = new Date(chunks[2].toLong() * 1000);
            changeSet.message = chunks[3]?.trim();

            // read file paths
            for (int i = 4; i < chunks.length;) {
                def changeSetType = chunks[i++].trim(); // first type entry has a newline char
                def filePath = chunks[i++];

                ChangeSetFileEntry fileEntry = new ChangeSetFileEntry();
                fileEntry.revision = changeSet.id;
                fileEntry.type = changeSetType;
                fileEntry.path = filePath;
                changeSet.fileSet << fileEntry;
            }
            changeSets << changeSet;
        }

        // filter results
        changeSets.retainAll{ ChangeSet changeSet ->
            boolean allowAuthor = changeSet.hasAllowedAuthor(changeSetFilter);
            boolean allowAnyPaths = changeSet.hasAllowedPath(changeSetFilter);
            if (allowAuthor && allowAnyPaths) {
                // retain changeset
                return true;
            }
            else {
                def message = new StringBuilder("Changeset ${changeSet.id} skipped because ")
                if (!allowAuthor) {
                    message << "it has excluded author ${changeSet.user}"
                }
                if (!allowAuthor && !allowAnyPaths) {
                    message << " and "
                }
                if (!allowAnyPaths) {
                    message << "it contains only excluded file paths (${changeSet.fileSet.collect{it.path}})"
                }
                println message
                // exclude changeset
                return false;
            }
        }

        return changeSets;
    }

    protected String runLogCommand() {
        // we want  a trailing separator from the change sets
        // use unit-separator %x1F in format string, null (%x00) breaks output on msysgit
        // %B is preferred as it doesn't reformat the commit messages, but %s%n%b will have to suffice
        def format = ['%H', '%an', '%at', '%s%n%b', ''].join(UNIT_SEPARATOR_FMT)

        def logCommand = [scmCommand, '--no-pager', 'log', '-z', '--no-renames', '--name-status', "--format=format:$format"]
        String startPoint = startRevision ?: lookupRevision(startDate);
        String endPoint = endRevision ?: lookupRevision(endDate);

        // QP should use Dates, CL uses revisions
        if (startPoint || endPoint) {
            def range = "${startPoint ?: ''}..${endPoint ?: 'HEAD'}"
            logCommand << range
        }

        logCommand << "$branch";
        logCommand << '--'

        def logOutput = getCommandText(logCommand);
        return logOutput.replaceFirst('\0$', ''); // trim the dangling null
    }

    protected String formatChangelogDate(Date date) {
        def df = this.getDateArgumentFormatter()
        return date ? df.format(date) : null;
    }
    
    private void sendPostRequest(String xml) {
        // construct the URL with property replacements
        String url = changesUrl
        def authToken = System.getenv("AUTH_TOKEN")
        
        // Debug/testing stub
       if (url.startsWith("file://")) {
           File xmlOut = new File(directory, "xmlOut.xml")
           xmlOut << xml
       }
       else {
           ProtocolSocketFactory socketFactory = new OpenSSLProtocolSocketFactory()
           Protocol https = new Protocol("https", socketFactory, 443)
           Protocol.registerProtocol("https", https)
    
           PostMethod postMethod = new PostMethod(url)
           if (authToken) {
               postMethod.setRequestHeader("Authorization-Token", authToken)
               postMethod.setRequestHeader("Content-Type", "application/xml")
           }
           
           println "Sending ${changeSets.size()} changes"
           postMethod.setRequestEntity(new StringRequestEntity(xml));
    
           HttpClient client = new HttpClient()
    
           def responseCode = client.executeMethod(postMethod)
           InputStream responseStream = postMethod.getResponseBodyAsStream()
           if (isGoodResponseCode(responseCode)) {
               IO.copy(responseStream, System.out)
               println ""
           }
           else {
               IO.copy(responseStream, System.err)
               throw new RuntimeException("Failed to upload source changes. StatusCode: ${responseCode}")
           }
       }
    }
    
    private boolean isGoodResponseCode(int responseCode) {
        return responseCode >= 200 && responseCode < 300;
    }
}
