/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
package com.urbancode.air.plugin.quality.jacoco

import org.apache.http.client.HttpClient
import org.apache.http.client.methods.HttpPost
import org.apache.http.entity.StringEntity
import org.apache.http.impl.client.DefaultHttpClient

import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder
import com.urbancode.commons.util.IO

public class JacocoCoverageXMLHelper {
    
    //**************************************************************************
    // CLASS
    //**************************************************************************

    //**************************************************************************
    // INSTANCE
    //**************************************************************************
    private def testCount = 0
    private def suiteCount = 0
    
    public JacocoCoverageXMLHelper() {
    }
    
    def uploadResult (reportName, buildLifeId, coverageXmlFile) {
        def testsuiteName
        def xml
        def result
        def name
        def time
        testCount = 0
        suiteCount = 0

        XmlParser parser = new XmlParser()
        parser.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
        parser.setFeature("http://xml.org/sax/features/namespaces", false)
        def rootNode = parser.parse(coverageXmlFile)
        
        def reportXml = new java.io.StringWriter()
        new groovy.xml.MarkupBuilder(reportXml).
            "coverage-report"(
                "name": reportName,
                "type": 'JaCoCo',
                "build-life-id": buildLifeId,
                "line-percentage": getValue(rootNode."counter".find{it.@"type" == "LINE"}),
                "method-percentage": getValue(rootNode."counter".find{it.@"type" == "METHOD"}),
                "branch-percentage": getValue(rootNode."counter".find{it.@"type" == "BRANCH"}),
                "complexity": getValue(rootNode."counter".find{it.@"type" == "COMPLEXITY"})
            ) {
            
            def nodes
            if (rootNode."group") {
                nodes = rootNode."group" 
            }
            else {
                nodes = rootNode."package"
            }
            
            nodes.each { node ->
                "coverage-group"(
                    "name": node.@name,
                    "line-percentage": getValue(node."counter".find{it.@"type" == "LINE"}),
                    "method-percentage": getValue(node."counter".find{it.@"type" == "METHOD"}),
                    "branch-percentage": getValue(node."counter".find{it.@"type" == "BRANCH"}),
                    "complexity": getValue(node."counter".find{it.@"type" == "COMPLEXITY"})
                )
            }
        }
            
        if (reportXml) {
            //println "Sending report:\n" + reportXml.toString()
            sendPostRequest(reportName, reportXml.toString())
        }
        else {
            println 'No report was able to be generated'
        }
    }
    
    private def getValue(def counter) {
        int missed = 0;
        int covered = 0;
    
        if (counter) {
            if (counter.@"missed") {
                missed = Integer.parseInt(counter.@"missed") 
            }
            
            if (counter.@"covered") {
                covered = Integer.parseInt(counter.@"covered")
            }
        }
        
        int total = missed + covered
        return (double) covered / (total ?: 1)
    }
    
    private void sendPostRequest(String name, String xml) {
        def authToken = System.getenv("AUTH_TOKEN")
        String buildLifeId = System.getenv("BUILD_LIFE_ID")
        def reportName = URLEncoder.encode(name)
        
        // construct the URL with property replacements
        String baseUrl = System.getenv("WEB_URL")
        
        baseUrl += baseUrl.endsWith("/") ? "" : "/"
        String url = baseUrl + "rest/buildlife/${buildLifeId}/testcoverage?reportName=${reportName}"
        println "Sending request to $url"

        HttpClientBuilder clientBuilder = new HttpClientBuilder();
        clientBuilder.setTrustAllCerts(true);
        DefaultHttpClient client = clientBuilder.buildClient();
        
        HttpPost postMethod = new HttpPost(url);
        if (authToken) {
            postMethod.setHeader("Authorization-Token", authToken)
            postMethod.setHeader("Content-Type", "application/xml")
        }
        postMethod.setEntity(new StringEntity(xml));
    
        println "Sending coverage report"
        def httpResponse = client.execute(postMethod)
        def responseCode = httpResponse.getStatusLine().getStatusCode()
        if (isGoodResponseCode(responseCode)) {
            IO.copy(postMethod.getEntity().getContent(), System.out)
            println ""
        }
        else {
            IO.copy(postMethod.getEntity().getContent(), System.err)
            throw new RuntimeException("Failed to upload JaCoCo coverage report. StatusCode: ${responseCode}")
        }
    }
    
    private boolean isGoodResponseCode(int responseCode) {
        return responseCode >= 200 && responseCode < 300;
    }
}