/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
package com.urbancode.air.plugin.quality.jacoco

import org.apache.http.client.methods.HttpPost
import org.apache.http.entity.StringEntity
import org.apache.http.impl.client.DefaultHttpClient

import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder
import com.urbancode.commons.util.IO

public class JacocoCoverageXMLHelper {
    
    //**************************************************************************
    // CLASS
    //**************************************************************************

    //**************************************************************************
    // INSTANCE
    //**************************************************************************


    public JacocoCoverageXMLHelper() {
    }
    
    def uploadResult (reportFiles, reportName, buildLifeId) {
        XmlParser parser = new XmlParser()
        parser.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
        parser.setFeature("http://xml.org/sax/features/namespaces", false)

        CoverageReport report = new CoverageReport(reportName, buildLifeId)
        reportFiles.each { reportFile ->
            def rootNode = parser.parse(reportFile)
            addCounts(rootNode, "LINE", report.totals.lineCounts)
            addCounts(rootNode, "METHOD", report.totals.methodCounts)
            addCounts(rootNode, "BRANCH", report.totals.branchCounts)
            addCounts(rootNode, "COMPLEXITY", report.totals.complexityCounts)

            def nodes
            if (rootNode."group") {
                nodes = rootNode."group"
            }
            else {
                nodes = rootNode."package"
            }

            nodes.each { node ->
                CoverageGroup group
                String name = node.@name
                if (report.coverageGroups.containsKey(name)) {
                    group = report.coverageGroups.get(name)
                }
                else {
                    group = new CoverageGroup()
                    report.coverageGroups.put(name, group)
                }

                addCounts(node, "LINE", group.lineCounts)
                addCounts(node, "METHOD", group.methodCounts)
                addCounts(node, "BRANCH", group.branchCounts)
                addCounts(node, "COMPLEXITY", group.complexityCounts)
            }
        }

        def reportXml = report.generateReportXml()
        if (reportXml) {
            sendPostRequest(reportName, reportXml.toString())
        }
        else {
            println 'No report was able to be generated'
        }
    }

    private void addCounts(def node, def type, def counts) {
        def countsNode = node."counter".find{it.@"type" == type}
        if (countsNode) {
            def missedCountAttr = countsNode.@"missed"
            if (missedCountAttr) {
                counts.missed += Integer.parseInt(missedCountAttr)
            }

            def coveredCountAttr = countsNode.@"covered"
            if (coveredCountAttr) {
                counts.covered += Integer.parseInt(coveredCountAttr)
            }
        }
    }

    private void sendPostRequest(String name, String xml) {
        def authToken = System.getenv("AUTH_TOKEN")
        String buildLifeId = System.getenv("BUILD_LIFE_ID")
        def reportName = URLEncoder.encode(name)
        
        // construct the URL with property replacements
        String baseUrl = System.getenv("WEB_URL")
        
        baseUrl += baseUrl.endsWith("/") ? "" : "/"
        String url = baseUrl + "rest/buildlife/${buildLifeId}/testcoverage?reportName=${reportName}"
        println "Sending request to $url"

        HttpClientBuilder clientBuilder = new HttpClientBuilder();
        clientBuilder.setTrustAllCerts(true);
        DefaultHttpClient client = clientBuilder.buildClient();
        
        HttpPost postMethod = new HttpPost(url);
        if (authToken) {
            postMethod.setHeader("Authorization-Token", authToken)
            postMethod.setHeader("Content-Type", "application/xml")
        }
        postMethod.setEntity(new StringEntity(xml));
    
        println "Sending coverage report"
        def httpResponse = client.execute(postMethod)
        def responseCode = httpResponse.getStatusLine().getStatusCode()
        if (isGoodResponseCode(responseCode)) {
            IO.copy(postMethod.getEntity().getContent(), System.out)
            println ""
        }
        else {
            IO.copy(postMethod.getEntity().getContent(), System.err)
            throw new RuntimeException("Failed to upload JaCoCo coverage report. StatusCode: ${responseCode}")
        }
    }
    
    private boolean isGoodResponseCode(int responseCode) {
        return responseCode >= 200 && responseCode < 300;
    }
}
