/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import com.urbancode.air.CommandHelper
import com.urbancode.air.AirPluginTool

final def workDir = new File('.').canonicalFile
def ch = new CommandHelper(workDir)
def apTool = new AirPluginTool(this.args[0], this.args[1])
def props = apTool.getStepProperties()

final def cmdPath = props['cmdPath']
final def buildFile = props['buildFile']
final def targets = props['targets']
final def verbosity = props['verbosity']
final def buildProps = props['buildProps']
final def buildParams = props['buildParams']
final def envVars = props['envVars']
final def scriptContent = props['scriptContent']

def commandLineArgs = []
if (!buildFile) {
    throw new IllegalArgumentException("No build file was specified.")
}

if (!verbosity) {
    verbosity = "normal"
}

// Get the default path if one is not supplied
cmdPath = cmdPath?: "msbuild.exe"

final def msbuildFile = new File(workDir, buildFile)

// Check if script content was specified, if so, we need to write out the file
boolean deleteOnExit = false
if (scriptContent) {
    if (msbuildFile.exists()) {
        throw new IllegalArgumentException("A file by the name of " + buildFile + " already exists")
    }
    
    msbuildFile.text = scriptContent
    deleteOnExit = true
}

try {
    // Build command line
    commandLineArgs << cmdPath << buildFile << "/nologo" << "/v:" + verbosity
    
    if (targets) {
        def targetCommand = "";
        def addSemicolon = false;
        targets.split(',').each() { target ->
            if (target) {
                if (!addSemicolon) {
                    addSemicolon = true;
                }
                else {
                    targetCommand += ";"
                }
                targetCommand += target
            }
        }
        
        commandLineArgs << "/t:" + targetCommand
    }
    
    if (buildProps) {
        def propCommand = "";
        def addSemicolon = false;
        buildProps.split('\n').each() { prop ->
            if (prop) {
                if (!addSemicolon) {
                    addSemicolon = true
                }
                else {
                    propCommand += ";"
                }
                propCommand += prop
            }
        }
        
        commandLineArgs << "/p:" + propCommand
    }
    
    if (buildParams) {
        buildParams.split('\n').each() { param ->
            if (param) {
                commandLineArgs << param
            }
        }
    }
    
    // Set any environment variables on the environment itself
    if (envVars) {
        def pb = ch.getProcessBuilder()
        def envProps = pb.environment()
        
        envVars.split('\n').each() { prop ->
            if (prop) {
                def propMap = prop.split("=", 2)
                envProps[propMap[0]] = propMap[1]
            }
        }
    }
    
    ch.runCommand("Running MSBuild command", commandLineArgs)
}
finally {
    if (deleteOnExit) {
        msbuildFile.delete()
    }
}