import com.urbancode.air.*

final def apTool = new AirPluginTool(this.args[0], this.args[1]);

//------------------------------------------------------------------------------
// PARSE PROPERTIES
//------------------------------------------------------------------------------

final def stepProps = apTool.getStepProperties()
final def workDir = new File('.').canonicalFile
final String osName = System.getProperty('os.name').toLowerCase(Locale.US)
final boolean windows = (osName =~ /windows/)

def pomFileName                 = stepProps['pomFileName']
def goals                       = stepProps['goals']
def flags                       = stepProps['flags'] // WARN values start with ?
def mavenHome                   = stepProps['mavenHome']
def mavenVersion                = stepProps['mavenVersion']
def mavenProps                  = stepProps['mavenProps'] // WARN values start with ?
def localRepoMetaPath           = stepProps['localRepoMetaDir']
def javaHome                    = stepProps['javaHome']
def jvmProperties               = stepProps['jvmProperties'] // WARN values start with ?
def scriptContent               = stepProps['scriptContent']

//
// Validation
//
assert !workDir.isFile() : "Working directory ${workDir} is a file!"

// ensure work-dir exists
workDir.mkdirs()

String mvnExe;
if (mavenHome) {
    mvnExe = new File(mavenHome, 'bin/' + mavenVersion).absolutePath
}
else {
    mvnExe = mavenVersion
}

if (windows && !mvnExe.toLowerCase().endsWith(".exe") && !mvnExe.toLowerCase().endsWith(".bat")) {
    mvnExe += ".bat"
}

final def pomFile = new File(workDir, pomFileName)
// check if script content was specified, if so, we need to write out the file
boolean deleteOnExit = false
if (scriptContent) {
    pomFile.text = scriptContent
    deleteOnExit = true
}

try {
    // jvm will release if exceptions occur
    def lock = null;
    def repoPath = null;
    if (localRepoMetaPath) {
        def localRepoMetaDir = new File(localRepoMetaPath);
        localRepoMetaDir.mkdirs();
        assert localRepoMetaDir.isDirectory();
    
        for (int i = 0; i < MAX_REPOS && lock == null; i++) {
            def testRepo = new File(localRepoMetaDir, 'repo' + i);
            testRepo.mkdirs();
            def testRepoLck = new File(localRepoMetaDir, 'repo' + i + '.sync');
            testRepoLck.createNewFile();
    
            def randFile = new RandomAccessFile(testRepoLck, 'rw')
            def channel = randFile.channel;
    
            lock = channel.tryLock(); // assums no other locking happens within this jvm
            repoPath=testRepo.absolutePath;
            println "Locked maven local repository $repoPath"
        }
    
        assert repoPath != null : "could not acquire any repoX within $localRepoMetaPath"
    }
    
    //
    // Build Command Line
    //
    def commandLine = []
    commandLine << mvnExe << '--batch-mode'
    
    mavenProps?.eachLine{ prop ->
    if (prop && !prop.startsWith('-D')) {
            prop = '-D' + prop
        }
        commandLine << prop;
    }
    
    if (repoPath) {
        commandLine << '-Dmaven.repo.local=' + repoPath
    }
    
    if (pomFileName) {
        commandLine << '--file' << pomFile.absolutePath
    }
    
    if (flags) {
        flags.split('\\s+').each() { flag ->
            if (flag) {
                commandLine << flag
            }
        }
    }
    
    if (goals) {
        goals.split('\\s+').each() { goal ->
            if (goal) {
                commandLine << goal
            }
        }
    }
    
    def commandHelper = new CommandHelper(workDir);
    if (javaHome) {
        commandHelper.addEnvironmentVariable('JAVA_HOME', javaHome)
    }
    if (jvmProperties) {
        commandHelper.addEnvironmentVariable('MAVEN_OPTS', jvmProperties.replace((char)'\n', (char)' '))
    }
    
    commandHelper.runCommand('Executing Maven', commandLine);
    
    
    if (lock) {
        // cleanly release the lock
        lock.release();
    }
}
finally {
    if (deleteOnExit) {
        pomFile.delete()
    }
}
