package com.urbancode.air.plugin.ncover

import com.urbancode.air.CommandHelper
import com.urbancode.commons.util.IO
import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder

import org.apache.http.client.HttpClient
import org.apache.http.client.methods.HttpPost
import org.apache.http.entity.StringEntity
import org.apache.http.impl.client.DefaultHttpClient

import groovy.xml.XmlUtil

public class NcoverHelper {
    private def props

    public NcoverHelper(def props) {
        this.props = props
    }

    // generate & format xml coverage report then upload result to uBuild
    public def parseNcover = {
        final def workDir = new File('.').canonicalFile
        final def buildLifeId = System.getenv()['BUILD_LIFE_ID']
        final def filename = props['covFile']
        final def convertToTrends = Boolean.valueOf(props['convertToTrends'])
        final def reportExeLoc = props['reportExeLoc'] ?: 'NCover.Reporting'
        final def reportName = props['reportName'] ?: 'NCover Report'
        final def isNCoverComplete = Boolean.valueOf(props['isNcoverComplete'])
        final def htmlExport = Boolean.valueOf(props['htmlExport'])
        final def additionalReportFlags = props['additionalReportFlags']?.split(/\r?\n/) ?: []
        final def additionalConversionFlags = props['additionalConversionFlags']?.split(/\r?\n/) ?: []

        CommandHelper ch = new CommandHelper(workDir);

        if (htmlExport){
            println ("Creating full Html coverage file for: " + filename);
            def reportArgs = [];
            reportArgs = [reportExeLoc, filename, '//or', 'FullCoverageReport:Html:FullCoverageReport\\index.html'];
            additionalReportFlags.each { reportArgs << it }
            def createFullHtmlExitCode = ch.runCommand(reportArgs.join(' '), reportArgs)
            if (createFullHtmlExitCode != 0) {
                throw new Exception("Failed to create full HTML coverage file, exit code: $createFullHtmlExitCode")
            }
            println ("Finished Creating full Html coverage file for: " + filename);
        }

        if (convertToTrends) {
            println ("Converting coverage file: " + filename + " to Trends Report");
            def reportArgs = [];
            reportArgs = [reportExeLoc, filename, '//or', 'Trends:Xml', '//rdf', 'Assembly'];
            additionalConversionFlags.each { reportArgs << it }
            def convertToTrendsExitCode = ch.runCommand(reportArgs.join(' '), reportArgs);
            if (convertToTrendsExitCode != 0) {
                throw new Exception("Failed to convert to Trends Report, exit code: $convertToTrendsExitCode")
            }
            println ("Converted coverage file: " + filename + " to Trends Report: Trends.xml");
            filename = 'Trends.xml'
        }

        println ("Processing coverage file: " + filename);
        File file = new File(filename);
        def coverage = new XmlParser().parse(file);
        // Calculate full / average coverage
        def totalMethodCoveragePercent = 0;
        def totalBranchCoveragePercent = 0;
        def totalLineCoveragePercent = 0;
        def totalComplexity = 0;
        def myStats = coverage.stats[0];
        def double totVLine = myStats.@vl.toDouble();
        def double totULine = myStats.@ul.toDouble();

        if ((totVLine + totULine) > 0) {
            totalLineCoveragePercent = totVLine / (totVLine + totULine)
        }
        def double totVMethod = myStats.@vm.toDouble();
        def double totUMethod = myStats.@um.toDouble();

        if ((totVMethod + totUMethod) > 0) {
            totalMethodCoveragePercent = totVMethod / (totVMethod + totUMethod)
        }
        if (isNCoverComplete) {
            def double totVBranch = myStats.@vbp.toDouble();
            def double totUBranch = myStats.@ubp.toDouble();
            if ((totVMethod + totUMethod) > 0) {
                totalBranchCoveragePercent = totVBranch / (totVBranch + totUBranch)
            }
        }
        // Calculate group coverage
        List<Map<String, String>> coverGroupData = new ArrayList<Map<String, String>>();
        coverage.mod.ns.each{
            Map<String, String> covInfoMap = new HashMap<String, String>();
            boolean excluded = it.stats[0].attribute("ex").equals("1");
            if (!excluded) {
                def String name = it.en.text();
                def double vLine = it.stats[0].@vl.toDouble()
                def double uvLine = it.stats[0].@ul.toDouble()
                def lCov = 0
                if ((vLine + uvLine) != 0) {
                    lCov = vLine / (vLine + uvLine);
                }
                def bCov = 0;
                def double comp = 0;
                // Only if they have NCover complete
                if (isNCoverComplete) {
                    def double vBranch = it.stats[0].@vbp.toDouble();
                    def double uBranch = it.stats[0].@ubp.toDouble();
                    if ((vBranch + uBranch) != 0) {
                    bCov = vBranch / (vBranch + uBranch);
                }
                comp = it.stats[0].@ccavg.toDouble();
            }
            def double vMethod = it.stats[0].@vm.toDouble()
            def double uMethod = it.stats[0].@um.toDouble()
            def double mCov = 0
            if ((vMethod + uMethod) != 0) {
                mCov = vMethod / (vMethod + uMethod);
            }
            covInfoMap.put('group', name);
            covInfoMap.put('metCov', mCov);
            if (isNCoverComplete) {
                covInfoMap.put('branchCov', bCov);
                covInfoMap.put('complexity', comp)
            }
            covInfoMap.put('lineCov', lCov)
            coverGroupData.add(covInfoMap)
            }
        }
        // produce output xml
        def xml = new groovy.xml.StreamingMarkupBuilder().bind {
            List<String> covGroupsIncluded = new ArrayList<String>()
            mkp.xmlDeclaration()
            'coverage-report' (
                 name:reportName,
                 type:'NCover',
                 'build-life-id':buildLifeId,
                 'method-percentage':totalMethodCoveragePercent,
                 'branch-percentage':totalBranchCoveragePercent,
                 'line-percentage':totalLineCoveragePercent,
                 'complexity': totalComplexity) {

                 coverGroupData.each { findingMap ->
                     System.out.println("Adding group to coverage report : " + findingMap['group'])
                     if (covGroupsIncluded.contains(findingMap['group'])) {
                         println("We already have: " + findingMap['group'])
                     }
                     else {
                         covGroupsIncluded.add(findingMap['group'])
                         if (isNCoverComplete) {
                             'coverage-group'(
                                 name:findingMap['group'],
                                 'method-percentage':Double.valueOf(findingMap['metCov']),
                                 'line-percentage':Double.valueOf(findingMap['lineCov']),
                                 'branch-percentage':Double.valueOf(findingMap['branchCov']),
                                 'complexity':Double.valueOf(findingMap['complexity'])
                             )
                         }
                         else {
                             'coverage-group'(
                                 name:findingMap['group'],
                                 'method-percentage':Double.valueOf(findingMap['metCov'])
                             )
                         }
                     }
                 }
            }
        }

        uploadReport(reportName, xml); //upload report to uBuild
    }

    public def runNcover = {
        final def workDir = new File('.').canonicalFile
        final def buildLifeId = System.getenv()['BUILD_LIFE_ID']
        final def consoleExeLoc = props['consoleExeLoc'] ? props['consoleExeLoc'] : 'NCover.Console.exe'
        final def coverageFile = props['coverageFile']
        final def unitTestApp = props['unitTestApp']
        final def excludeMethods = props['excludeMethods']
        final def excludeAssemblies = props['excludeAssemblies']
        final def excludeTypes = props['excludeTypes']
        final def excludeSource = props['excludeSource']
        final def testDll = props['testDll']
        final def reportExeLoc = props['reportExeLoc'] ? props['reportExeLoc'] : 'NCover.Reporting'
        final def reportType = props['reportType']
        final def htmlExport = Boolean.valueOf(props['htmlExport'])
        final def additionalFlags = props['additionalFlags']?.split(/\r?\n/) ?: []
        def ch = new CommandHelper(new File('.'));
        def consoleArgs = [];
        consoleArgs = [consoleExeLoc, '//x', coverageFile, unitTestApp];
        if (testDll) {
            consoleArgs << testDll;
        }
        if (excludeMethods) {
            consoleArgs << '//em' << excludeMethods;
        }
        if (excludeAssemblies) {
            consoleArgs << '//eas' << excludeAssemblies;
        }
        if (excludeTypes) {
            consoleArgs << '//et' << excludeTypes;
        }
        if (excludeSource) {
            consoleArgs << '//ef' << excludeSource;
        }
        additionalFlags.each { consoleArgs << it }

        ch.runCommand(consoleArgs.join(' '), consoleArgs);
        if (htmlExport){
            def reportArgs = [];
            reportArgs = [reportExeLoc, coverageFile, '//or', 'FullCoverageReport:Html:FullCoverageReport\\index.html'];
            ch.runCommand(reportArgs.join(' '), reportArgs);
        }
        def reportArgs = [];
        reportArgs = [reportExeLoc, coverageFile, '//or', reportType, '//rdf', 'Assembly'];
        return ch.runCommand(reportArgs.join(' '), reportArgs);
    }

    // upload coverage report to uBuild server
    def uploadReport(def reportName, def xml) {
        def authToken = System.getenv("AUTH_TOKEN")
        def buildLifeId = System.getenv("BUILD_LIFE_ID")
        def name = encode(reportName)

        String baseUrl = System.getenv("WEB_URL")
        baseUrl += baseUrl.endsWith("/") ? "" : "/"
        String url = baseUrl + "rest/buildlife/${buildLifeId}/testcoverage?reportName=${name}"

        println("Sending request to ${url}")

        HttpClientBuilder clientBuilder = new HttpClientBuilder()
        clientBuilder.setTrustAllCerts(true)
        DefaultHttpClient client = clientBuilder.buildClient()

        HttpPost postMethod = new HttpPost(url)

        if (authToken) {
            postMethod.setHeader("Authorization-Token", authToken)
            postMethod.setHeader("Content-Type", "application/xml")
        }

        postMethod.setEntity(new StringEntity(XmlUtil.serialize(xml)))

        def httpResponse = client.execute(postMethod)
        def responseCode = httpResponse.getStatusLine().getStatusCode()

        if (isGoodResponseCode(responseCode)) {
            IO.copy(postMethod.getEntity().getContent(), System.out)
            println()
        }
        else {
            IO.copy(postMethod.getEntity().getContent(), System.err)
            throw new RuntimeException("Failed to upload NCover coverage report. StatusCode: ${responseCode}")
        }
    }

    private boolean isGoodResponseCode(int responseCode) {
        return responseCode >= 200 && responseCode < 300
    }

    private def encode = {
        return !it ? it : new java.net.URI(null, null, it, null).toASCIIString()
    }
}