package com.urbancode.air.plugin.scm

import org.apache.http.HttpResponse
import org.apache.http.client.HttpClient
import org.apache.http.client.methods.HttpPost
import org.apache.http.entity.StringEntity

import com.urbancode.air.*
import com.urbancode.air.plugin.scm.changelog.*
import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder
import com.urbancode.commons.util.IO

public class SCMChangelog extends SCMStep {
    
    //**************************************************************************
    // CLASS
    //**************************************************************************
    
    static final long HOUR_IN_MILLIS = 1000L * 60L * 60L

    //**************************************************************************
    // INSTANCE
    //**************************************************************************
      
    // Changeset Variables
    final java.text.SimpleDateFormat PERFORCE_CHANGELOG_OUT_DATE = new java.text.SimpleDateFormat("yyyy/MM/dd HH:mm:ss")
    final java.util.regex.Pattern CHANGE_SUMMARY_PATTERN = ~'(?m)^Change \\d+ by (\\S+)@\\S+ on (\\d+/\\d+/\\d+ \\d+:\\d+:\\d+)'
    
    final ChangeSetFilter changeSetFilter = new ChangeSetFilter()
    List<ChangeSet> changeSets
    
    Long startDate
    Long endDate
    String changesUrl
    
    public def execute() {
        println "Getting changes from " + new Date(startDate) + " to " + new Date(endDate)
        
        this.login()
        
        getClientspecInfo()
        
        def logOutput = runLogCommand()
        changeSets = parseChangesets(logOutput)
        
        if (changeSets) {
            ChangeSetXMLHelper xmlHelper = new ChangeSetXMLHelper()
            xmlHelper.repoType = REPO_TYPE;
            def xml = xmlHelper.toXML(changeSets);
            sendPostRequest(xml)
        }
        else {
            println "No changes detected"
            return null
        }
    }
    
    public List<ChangeSet> parseChangesets(String logOutput) {
        List<ChangeSet> changeSets = []
        
        def typeMap = ['add':'A', 'edit':'M', 'delete':'D']
        
        // Change X on yyyy/MM/dd by user@client 'comment'
        logOutput.eachMatch(/(?m)^Change (\d+)/){ changematch, changeId ->
            def ChangeSet changeSet = new ChangeSet()
            changeSet.id = changeId
        
            // Get details
            def summary = ''
            def descCommand = createP4BaseCommand()
            descCommand << 'describe' << '-s' << changeId
            ch.runCommand(null, descCommand) { Process proc ->
                proc.out.close()                    // close   the stdIn  of the process
                proc.consumeProcessErrorStream(System.out) // forward the stdErr of the process
                summary = proc.text                 // capture the stdout of the process
                println summary                     // echo captured stdOut
            }
        
            // Summary Format:
            //   Change X by user@client on yyyy/MM/dd HH:mm:ss
            //   <some number of lines w/ or w/o content>
            //   Affected files ...
            //   ... //depot/file#3 edit
            //   ... //depot/file#3 edit
            //   ... //depot/file#3 edit
            //   ... //depot/file#3 edit
            //
        
            // first line is "Change X by user@client on yyyy/MM/dd HH:mm:ss"
            summary.find(CHANGE_SUMMARY_PATTERN) { summarymatch, user, dateString ->
                changeSet.user = user
                
                Date p4Date = PERFORCE_CHANGELOG_OUT_DATE.parse(dateString) // TODO deal with 'unknown' date
                Date localizedDate = new Date(adjustDateForTimeDifference(p4Date.time, -timezoneDifference))
                changeSet.date = localizedDate
                
                // Pick out the changelog comment (anything between the first line and "Affected files")
                int commentStart = summary.indexOf(summarymatch)+summarymatch.length()
                int commentEnd = summary.indexOf("Affected files ...", commentStart)
                if (commentStart > -1 && commentEnd > -1 && commentEnd > commentStart) {
                    changeSet.message = summary.substring(commentStart, commentEnd).trim()
                }

                // Eventually there is a line "Affected files ..."
                // followed by lines of "... //depot/file#3 edit"
                summary.eachMatch(/(?m)^\.\.\. (.+)#\d+ (\S+)/){affectedmatch, file, operation ->
                    ChangeSetFileEntry entry = new ChangeSetFileEntry()
                    entry.type = typeMap[operation] ?: 'M' // convert operation to A, M, D types
                    entry.path = file
                    changeSet.fileSet << entry
                }
        
                boolean hasAllowedAuthor = changeSet.hasAllowedAuthor(changeSetFilter);
                boolean hasAllowedFile = changeSet.hasAllowedPath(changeSetFilter);
                if (hasAllowedAuthor && hasAllowedFile) {
                    changeSets << changeSet
                }
                else {
                    def message = new StringBuilder("Changeset ${changeSet.id} skipped because ")
                    if (!hasAllowedAuthor) {
                        message << "it has excluded author ${changeSet.user}"
                    }
                    if (!hasAllowedAuthor && !hasAllowedFile) {
                        message << " and "
                    }
                    if (!hasAllowedFile) {
                        message << "it contains only excluded file paths (${changeSet.fileSet.collect{it.path}})"
                    }
                    println message
                }
            }
        }
        
        return changeSets
    }
    
    public String runLogCommand() {
        def serverStartDate = adjustDateForTimeDifference(startDate, timezoneDifference)
        def serverEndDate = adjustDateForTimeDifference(endDate, timezoneDifference)
        final def startDateString = formatChangelogDateParameter(serverStartDate) ?: null
        final def endDateString   = formatChangelogDateParameter(serverEndDate) ?: 'now'
        
        def changes;
        
        def command = createP4BaseCommand()
        command << 'changes' << '-s' << 'submitted' << '-t' << '-i'
        if (startDateString) {
            command << "//${client}/...@${startDateString},${endDateString}"
        }
        else if (endDateString) { // isn't this always true?
            command << "//${client}/...@${endDateString}"
        }
        else {
            command << "//${client}/..."
        }
        
        ch.runCommand('Getting Changes', command) { Process proc ->
            proc.out.close()                    // close   the stdIn  of the process
            proc.consumeProcessErrorStream(System.out) // forward the stdErr of the process
            changes = proc.text                 // capture the stdout of the process
            println changes                     // echo captured stdout
        }
        
        return changes
    }
    
    public String formatChangelogDateParameter(Long commandDate){
        if (!commandDate) {
            return null
        }
        return new java.text.SimpleDateFormat("yyyy/MM/dd:HH:mm:ss").format(new Date(commandDate))
    }
    
    public Long adjustDateForTimeDifference(Long date, Double hourDifference) {
        if (!date) {
            return null
        }
        long adjustedDate = date + (hourDifference * HOUR_IN_MILLIS)
        return adjustedDate;
    }
    
    private void sendPostRequest(String xml) {
        // construct the URL with property replacements
        String url = changesUrl
        def authToken = System.getenv("AUTH_TOKEN")
        
        println "Sending request to $url"
 
        // Debug/testing stub
        if (url.startsWith("file://")) {
            File xmlOut = new File(workDir, "xmlOut.xml")
            xmlOut << xml
        }
        else {
            HttpPost postMethod = new HttpPost(url)
            if (authToken) {
                postMethod.addHeader("Authorization-Token", authToken)
                postMethod.addHeader("Content-Type", "application/xml")
            }
            
            println "Sending ${changeSets.size()} changes:"
            println xml
            postMethod.setEntity(new StringEntity(xml));
     
            HttpClientBuilder builder = new HttpClientBuilder()
            builder.setTrustAllCerts(true)
            HttpClient client = builder.buildClient()
     
            HttpResponse response = client.execute(postMethod)
            def responseCode = response.statusLine.statusCode
            InputStream responseStream = response.entity.content
            if (isGoodResponseCode(responseCode)) {
                IO.copy(responseStream, System.out)
                println ""
            }
            else {
                IO.copy(responseStream, System.err)
                throw new RuntimeException("Failed to upload source changes. StatusCode: ${responseCode}")
            }
        }
    }
    
    private boolean isGoodResponseCode(int responseCode) {
        return responseCode >= 200 && responseCode < 300;
    }
}