package com.urbancode.air.plugin.automation;

import com.urbancode.air.*
import org.apache.commons.httpclient.*
import org.apache.commons.httpclient.methods.*;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.protocol.*

import com.urbancode.commons.util.IO
import com.urbancode.commons.util.https.OpenSSLProtocolSocketFactory

import java.util.regex.Pattern

public class PublishDefectReport extends AutomationBase {
    
    //**************************************************************************
    // CLASS
    //**************************************************************************
    
    //**************************************************************************
    // INSTANCE
    //**************************************************************************
    
    String defectPattern
    String taskPattern
    String usPattern
    
    def workspaceRef
    def defectIdSet = [] as Set
    def defectIdToChangeSetListMap = [:]
    def taskIdSet = [] as Set
    def taskIdToChangeSetListMap = [:]
    def usIdSet = [] as Set
    def usIdToChangeSetListMap = [:]
    
    def defectArtifactPattern
    def taskArtifactPattern
    def usArtifactPattern
    
    String issueUrlBase
    
    public void execute() {
        setupHttpClient()
        
        if (defectPattern) {
            defectArtifactPattern = java.util.regex.Pattern.compile(defectPattern)
        }
        
        if (taskPattern) {
            taskArtifactPattern = java.util.regex.Pattern.compile(taskPattern)
        }
        
        if (usPattern) {
            usArtifactPattern = java.util.regex.Pattern.compile(usPattern)
        }
        
        println "Getting Workspace"
        workspaceRef = configureWorkspace()
        println "Workspace Ref = ${workspaceRef}"
        
        try {
            String changesXml = getChangesets()
            
            parseChangesets(changesXml)
            
            String issuesXml = createXml()
            
            if (issuesXml) {
                sendPost(issuesXml)
            }
        }
        catch (Exception e) {
            e.printStackTrace()
            System.exit(1)
        }
    }
    
    private String getChangesets() {
        def authToken = System.getenv("AUTH_TOKEN")
        int buildLifeId = Integer.parseInt(System.getenv("BUILD_LIFE_ID"))
        
        String webUrl = System.getenv("WEB_URL")
        webUrl += webUrl.endsWith("/") ? "" : "/"
        String changesRequestUrl = webUrl + "rest/buildlife/${buildLifeId}/sourcechanges"
        
        println "\nGetting source changes from server"
        ProtocolSocketFactory socketFactory = new OpenSSLProtocolSocketFactory()
        Protocol https = new Protocol("https", socketFactory, 443)
        Protocol.registerProtocol("https", https)
        
        GetMethod getMethod = new GetMethod(changesRequestUrl)
        if (authToken) {
            getMethod.setRequestHeader("Authorization-Token", authToken)
        }
        
        String changesXml
        
        try {
            def responseCode = client.executeMethod(getMethod)
            InputStream responseStream = getMethod.getResponseBodyAsStream()
            
            changesXml = IO.readText(responseStream)
            if (!isGoodResponseCode(responseCode)) {
                throw new Exception("Failed to get build life source changes from the server: $changesXml")
            }
        }
        finally {
            getMethod.releaseConnection()
        }
        
        return changesXml
    }
    
    private void parseChangesets(String changesXml) {
        if (!changesXml.contains("change-set")) {
            println "No changes detected. Nothing to publish."
            System.exit(0)
        }
        
        println "Parsing Changelog"
        
        new XmlSlurper().parseText(changesXml)."change-set".each { changeSetElem ->
            def defectMatcher
            def taskMatcher
            def usMatcher
            
            def changeComment = changeSetElem.'comment'.text()
            if (defectArtifactPattern) {
                defectMatcher = defectArtifactPattern.matcher(changeComment);
            }
            
            if (taskArtifactPattern) {
                taskMatcher = taskArtifactPattern.matcher(changeComment);
            }
            
            if (usArtifactPattern) {
                usMatcher = usArtifactPattern.matcher(changeComment);
            }
            
            while (defectMatcher?.find()) {
                def defectId
                if (defectMatcher.groupCount() > 0 ) {
                    // they specified a '(...)' group within the pattern, use that as the bug-id
                    defectId = defectMatcher.group(1)
                }
                else {
                    // use the whole matching substring as the bug-id
                    defectId = defectMatcher.group()
                }
                
                println "Found defect identifier '$defectId' in change comment: $changeComment"
                defectIdSet.add(defectId)
                def changeSetList = defectIdToChangeSetListMap[defectId]
                if (!changeSetList) {
                    changeSetList = []
                    defectIdToChangeSetListMap[defectId] = changeSetList
                }
                def changeId = changeSetElem."change-id"?.text() ? changeSetElem."change-id".text() : changeSetElem."id".text()
                changeSetList.add(changeId)
            }
            while (taskMatcher?.find()) {
                def taskId
                if (taskMatcher.groupCount() > 0 ) {
                    // they specified a '(...)' group within the pattern, use that as the bug-id
                    taskId = taskMatcher.group(1)
                }
                else {
                    // use the whole matching substring as the bug-id
                    taskId = taskMatcher.group()
                }
                
                println "Found task identifier '$taskId' in change comment: $changeComment"
                taskIdSet.add(taskId)
                def changeSetList = taskIdToChangeSetListMap[taskId]
                if (!changeSetList) {
                    changeSetList = []
                    taskIdToChangeSetListMap[taskId] = changeSetList
                }
                def changeId = changeSetElem."change-id"?.text() ? changeSetElem."change-id".text() : changeSetElem."id".text()
                changeSetList.add(changeId)
            }
            while (usMatcher?.find()) {
                def usId
                if (usMatcher.groupCount() > 0 ) {
                    // they specified a '(...)' group within the pattern, use that as the bug-id
                    usId = usMatcher.group(1)
                }
                else {
                    // use the whole matching substring as the bug-id
                    usId = usMatcher.group()
                }
                
                println "Found User Story identifier '$usId' in change comment: $changeComment"
                usIdSet.add(usId)
                def changeSetList = usIdToChangeSetListMap[usId]
                if (!changeSetList) {
                    changeSetList = []
                    usIdToChangeSetListMap[usId] = changeSetList
                }
                def changeId = changeSetElem."change-id"?.text() ? changeSetElem."change-id".text() : changeSetElem."id".text()
                changeSetList.add(changeId)
            }
        }
    }
    
    private String createXml() {
        if (usIdSet.size() == 0 && taskIdSet.size() == 0 && defectIdSet.size() == 0) {
            println "No Rally artifact references found in build life changes."
        }
        else {
            println "Creating Issues.xml"
            def issuesXml = new java.io.StringWriter()
            new groovy.xml.MarkupBuilder(issuesXml).issues() {
                for (def defectId in defectIdSet) {
                    println "Adding ${defectId} to Issues.xml"
                    def defectData=null
                    HttpMethod method = null
                    try {
                        method = new GetMethod(url + "defect?workspace=" + URLEncoder.encode(workspaceRef, "UTF-8") + 
                            "&query=" + URLEncoder.encode("(FormattedID = " + defectId+ ")" ,"UTF-8") + "&" + URLEncoder.encode("fetch", "UTF-8") + 
                                "=" + URLEncoder.encode("true", "UTF-8"))
                        def result = client.executeMethod(method)
                        defectData = IO.readText(method.getResponseBodyAsStream())
                    }
                    finally {
                        if(method != null) {
                            method.releaseConnection();
                        }
                    }
                    def defectResult = new XmlParser().parseText(defectData).Results.Object;
                    if (defectResult.size() > 0) {
                        println "Found Rally defect $defectId - ${defectResult.'@refObjectName'}"

                        // create the issue element
                        defectIdToChangeSetListMap.get(defectId).each() { changeId ->
                            'issue'('id': defectId, "issue-tracker": "Rally", "change-id": changeId) {
                                // create the name element
                                'name'(defectResult.'@refObjectName'[0])
                                // create the type element
                                // type would be the tracker name in TeamForge, but their SOAP services do not allow you to get
                                // the tracker from the defect id
                                'type'(defectResult.'@type'[0])
                                // create the description element
                                'description'(defectResult.'Description'.text())
                                // create the status element
                                'status'(defectResult.'State'.text())
                                
                                'url'(genUrl(defectResult, "defect"))
                            }
                        }
                    }
                    else {
                        println "Unable to find Rally defect $defectId"
                    }
                }
                
                for (def taskId in taskIdSet) {
                    println "Adding ${taskId} to Issues.xml"
                    def taskData=null
                    HttpMethod method = null
                    try {
                        method = new GetMethod(url + "task?workspace=" + URLEncoder.encode(workspaceRef, "UTF-8") + 
                            "&query=" + URLEncoder.encode("(FormattedID = " + taskId+ ")" ,"UTF-8") + "&" + URLEncoder.encode("fetch", "UTF-8") + 
                                "=" + URLEncoder.encode("true", "UTF-8"))
                        def result = client.executeMethod(method)
                        taskData = IO.readText(method.getResponseBodyAsStream())
                    }
                    finally {
                        if (method != null) {
                            method.releaseConnection();
                        }
                    }
                    def taskResult = new XmlParser().parseText(taskData).Results.Object;
                    if (taskResult.size() > 0) {
                        println "Found Rally task $taskId - ${taskResult.'@refObjectName'}"
                        
                        // create the issue elements
                        taskIdToChangeSetListMap.get(taskId).each() { changeId ->
                            'issue'(id:taskId, "issue-tracker": "Rally", "change-id": changeId) {
                                // create the name element
                                'name'(taskResult.'@refObjectName'[0])
                                // create the type element
                                // type would be the tracker name in TeamForge, but their SOAP services do not allow you to get
                                // the tracker from the task id
                                'type'(taskResult.'@type'[0])
                                // create the description element
                                'description'(taskResult.'Description'.text())
                                // create the status element
                                'status'(taskResult.'State'.text())
                                'url'(genUrl(taskResult, "task"))
                            }
                        }
                    }
                    else {
                        println "Unable to find Rally task $taskId"
                    }
                }
                
                for (def usId in usIdSet) {
                    def usData = null
                    println "Adding ${usId} to Issues.xml"
                    HttpMethod method = null
                    try {
                        method = new GetMethod(url + "hierarchicalrequirement?workspace=" + URLEncoder.encode(workspaceRef, "UTF-8") + 
                            "&query=" + URLEncoder.encode("(FormattedID = " + usId+ ")" ,"UTF-8") + "&" + URLEncoder.encode("fetch", "UTF-8") + 
                                "=" + URLEncoder.encode("true", "UTF-8"))
                        def result = client.executeMethod(method)
                        usData = IO.readText(method.getResponseBodyAsStream())
                    }
                    finally {
                        if (method != null) {
                            method.releaseConnection();
                        }
                    }
                    def usResult = new XmlParser().parseText(usData).Results.Object;
                    if (usResult.size() > 0) {
                        println "Found Rally User Story $usId - ${usResult.'@refObjectName'}"
                        
                        // create the issue elements
                        usIdToChangeSetListMap.get(usId).each() { changeId ->
                            'issue'(id:usId, "issue-tracker": "Rally", "change-id": changeId) {
                                // create the name element
                                'name'(usResult.'@refObjectName'[0])
                                // create the type element
                                // type would be the tracker name in TeamForge, but their SOAP services do not allow you to get
                                // the tracker from the us id
                                'type'(usResult.'@type'[0])
                                // create the description element
                                'description'(usResult.'Description'.text())
                                'url'(genUrl(usResult, "userstory"))
                            }
                        }
                    }
                    else {
                        println "Unable to find Rally User Story $usId"
                    }
                }
            }
            
            return issuesXml.toString()
        }
    }
    
    private void sendPost(String issuesXml) {
        int buildLifeId = Integer.parseInt(System.getenv("BUILD_LIFE_ID"))
        def authToken = System.getenv("AUTH_TOKEN")
        String webUrl = System.getenv("WEB_URL")
        webUrl += webUrl.endsWith("/") ? "" : "/"
        
        String postUrl = webUrl + "rest/buildlife/${buildLifeId}/issues"
        
        println "Uploading Issues.xml"
        //println issuesXml
        
        ProtocolSocketFactory socketFactory = new OpenSSLProtocolSocketFactory()
        Protocol https = new Protocol("https", socketFactory, 443)
        Protocol.registerProtocol("https", https)
 
        PostMethod postMethod = new PostMethod(postUrl)
        try {
            if (authToken) {
                postMethod.setRequestHeader("Authorization-Token", authToken)
                postMethod.setRequestHeader("Content-Type", "application/xml")
            }
            
            postMethod.setRequestEntity(new StringRequestEntity(issuesXml));
    
            def responseCode = client.executeMethod(postMethod)
            InputStream responseStream = postMethod.getResponseBodyAsStream()
            if (isGoodResponseCode(responseCode)) {
                IO.copy(responseStream, System.out)
                println ""
            }
            else {
                IO.copy(responseStream, System.err)
                throw new RuntimeException("Rally results upload to server failed. StatusCode: ${responseCode}")
            }
        }
        finally {
            postMethod.releaseConnection()
        }
    }
    
    private boolean isGoodResponseCode(int responseCode) {
        return responseCode >= 200 && responseCode < 300;
    }
    
    private String genUrl(def result, String type) {
        issueUrlBase = issueUrlBase.endsWith("/") ? issueUrlBase : (issueUrlBase + "/")
        def workspaceUrl = result.Workspace.'@ref'[0]
        def workspaceId = workspaceUrl.find(~'/workspace/(.*)$') { match, id -> return id }
        
        def rallyIssueUrl = issueUrlBase  + "#/" + workspaceId + "/detail/${type}/" + result.ObjectID.text()
        
        return rallyIssueUrl
    }
}