/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
package com.urbancode.air.plugin.scm

import org.apache.http.HttpResponse
import org.apache.http.client.HttpClient
import org.apache.http.client.methods.HttpGet
import org.apache.http.client.methods.HttpPost
import org.apache.http.entity.StringEntity

import com.urbancode.air.plugin.scm.changelog.*
import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder
import com.urbancode.commons.util.IO

public class SCMChangelog extends SCMStep {
    
    def srcName
    def rtcSnapshot
    def rtcBaseline
    def startSnapshot
    def isStartBaseline
    def endSnapshot
    def isEndBaseline
    String startBuildLifeId
    String changesUrl
    
    final static def CHANGELOG_DATE_FORMAT = 'yyyy-MM-dd HH:mm:ss.S Z'
    
    final ChangeSetFilter changeSetFilter = new ChangeSetFilter()
    List<ChangeSet> changeSets
    
    public def execute() {
        setUpSnapshots()
        
        if (startSnapshot && endSnapshot) {
            def componentSet
            if (!components && (isStartBaseline || isEndBaseline)) {
                componentSet = getWorkspaceInfo(null).componentList.keySet()
            }
            else if (components) {
                componentSet = components.split(',')
            }
            String changelogText = getChangelog(startSnapshot, isStartBaseline ? 'baseline' : 'snapshot',
                    endSnapshot, isEndBaseline ? 'baseline' : 'snapshot', componentSet)
            parseChangelog(changelogText)
            
            // Construct xml message and send to server
            if (changeSets) {
                ChangeSetXMLHelper xmlHelper = new ChangeSetXMLHelper()
                xmlHelper.repoType = REPO_TYPE
                String xml = xmlHelper.toXML(changeSets)
                sendPostRequest(xml)
            }
            else {
                println "No changes detected"
            }
        
        }
        else {
            println "Start Buildlife ID: ${startBuildLifeId}"
            println "Start Snapshot: ${startSnapshot} ${isStartBaseline?'is basline':''}"
            println "End Snapshot: ${endSnapshot} ${isEndBaseline?'is basline':''}"
            println "Cannot get a changelog without start and end snapshots!"
        }
    }
    
    private void setUpSnapshots() {
        def propertiesXml
        if (startBuildLifeId) {
            propertiesXml = getPropertiesXml(startBuildLifeId)
            parseProperties(propertiesXml)
            startSnapshot = rtcSnapshot
            isStartBaseline = rtcBaseline
        }
        
        String endBuildLifeId = System.getenv("BUILD_LIFE_ID")
        propertiesXml = getPropertiesXml(endBuildLifeId)
        parseProperties(propertiesXml)
        endSnapshot = rtcSnapshot
        isEndBaseline = rtcBaseline
    }
    
    private String getPropertiesXml(String buildLifeId) {
        def authToken = System.getenv("AUTH_TOKEN")
        
        // construct the URL with property replacements
        String baseUrl = System.getenv("WEB_URL")
        
        baseUrl += baseUrl.endsWith("/") ? "" : "/"
        String url = baseUrl + "rest/buildlife/${buildLifeId}/properties"
        
        println "Sending request to $url"

        HttpGet getMethod = new HttpGet(url)
        if (authToken) {
            getMethod.addHeader("Authorization-Token", authToken)
        }

        HttpClientBuilder builder = new HttpClientBuilder()
        builder.setTrustAllCerts(true)
        HttpClient client = builder.buildClient()

        HttpResponse response = client.execute(getMethod)
        def responseCode = response.statusLine.statusCode
        InputStream responseStream = response.entity.content
        
        String propertiesXml = IO.readText(responseStream)
        
        if (!isGoodResponseCode(responseCode)) {
            throw new Exception("Failed to get build life properties from the server: $propertiesXml")
        }
        
        getMethod.releaseConnection()
        
        return propertiesXml
    }
    
    private void parseProperties(String propertiesXml) {
        new XmlSlurper().parseText(propertiesXml)."property".each { property ->
            String propName = property.@name
            if (propName == "rtc.snapshot.${srcName}") {
                rtcSnapshot = property.text() ?: ""
            }
            else if (propName == "rtc.baseline.${srcName}") {
                rtcBaseline = true
            }
        }
    }
    
    /**
     * (1032) User:anthill Comment: 7, 15: Test work item - Change readme file.And try a new line Date:2010-03-16 10:13:53 -0500
     *      Work Item 7: Test - work - item
     *      Work Item 15: Test task
     *      /conf/jalopy.xml
     *      /readme.txt
     *
     */
    public def getChangelog(start, startType, end, endType, componentList) {
        StringBuilder changelogText = new StringBuilder()
        if (start && end) {
            if (componentList && componentList.size() > 0) {
                componentList.each { component ->
                    getCmdHelper().runCommand('Get Changelog',
                            [command, '--non-interactive', 'compare',
                                    startType, start,
                                    endType, end,
                                    '-I', 'wsf',
                                    '-f', 'i',
                                    '-C', 'User:{userid} Comment:',
                                    '-p', 'd',
                                    '-S', 'cdi',
                                    '-D', "'Date:'" + CHANGELOG_DATE_FORMAT,
                                    '-c', component,
                                    '-u', username,
                                    '-P', password,
                                    '-r', serverUrl]) { proc ->
                        proc.consumeProcessErrorStream(System.out)
                        proc.outputStream.close()
                        def componentChangelogText = proc.text.trim()
                        changelogText.append(componentChangelogText)
                        changelogText.append("\n")
                        println componentChangelogText
                    }
                }
            }
            else {
                getCmdHelper().runCommand('Get Changelog',
                        [command, '--non-interactive', 'compare',
                                startType, start,
                                endType, end,
                                '-I', 'wsf',
                                '-f', 'i',
                                '-C', 'User:{userid} Comment:',
                                '-p', 'd',
                                '-S', 'cdi',
                                '-D', "'Date:'" + CHANGELOG_DATE_FORMAT,
                                '-u', username,
                                '-P', password,
                                '-r', serverUrl]) { proc ->
                    proc.consumeProcessErrorStream(System.out)
                    proc.outputStream.close()
                    changelogText.append(proc.text.trim())
                    println changelogText.toString()
                }
            }
        }
        
        return changelogText.toString()
    }

    protected void parseChangelog(changelogText) {
        def currentChangeSet
        def currentWorkItemList = []
        
        if (!changeSets) {
            changeSets = []
        }
        
        changelogText.eachLine { it ->
            def line = it.trim()
            if (line ==~ '\\(\\d+\\) User:.*') {
                line.find('\\((\\d+)\\) User:(.*) Comment:(.*) Date:(.*)') {match, id, user, comment, date ->
                    if (currentChangeSet) {
                        if (currentWorkItemList) {
                            currentChangeSet.properties['workItems'] = currentWorkItemList.join(',');
                            currentWorkItemList.clear()
                        }
                        if (addChangeSet(currentChangeSet)) {
                            changeSets << currentChangeSet
                        }
                    }
                    currentChangeSet = new ChangeSet(id, user, comment, date)
                }
            }
            else if (line.startsWith('Work Item')) {
                line.find('Work Item (\\d+): (.+)') {match, workItemId, description ->
                    currentWorkItemList << workItemId
                }
            }
            // 4.0
            else if (line ==~ ('\\(\\d+:_[0-9A-Za-z\\-_]+\\) User:.*')) {
                line.find('\\(\\d+:(_[0-9A-Za-z\\-_]+)\\) User:(.*) Comment:(.*) Date:(.*)') {match, id, user, comment, date ->
                    if (currentChangeSet) {
                        if (currentWorkItemList) {
                            currentChangeSet.properties['workItems'] = currentWorkItemList.join(',');
                            currentWorkItemList.clear()
                        }
                        if (addChangeSet(currentChangeSet)) {
                            changeSets << currentChangeSet
                        }
                    }
                    currentChangeSet = new ChangeSet(id, user, comment, date)
                }
            }
            // 3.0 or 4.0 Work Items
            else if ((line ==~ '\\(\\d+:_[0-9A-Za-z\\-_]+\\) Work Item.*') || (line ==~ '\\(\\d+\\) Work Item.*')) {
                line.find('Work Item (\\d+): (.+)') {match, workItemId, description ->
                    currentWorkItemList << workItemId
                }
            }
            else if (line.size() > 0){
                ChangeSetFileEntry entry = new ChangeSetFileEntry()
                entry.path = line
                currentChangeSet.fileSet << entry
            }
        }
        if (currentChangeSet) {
            if (addChangeSet(currentChangeSet)) {
                changeSets << currentChangeSet
                
                if (currentWorkItemList) {
                    currentChangeSet.properties['workItems'] = currentWorkItemList.join(',');
                }
            }
        }
    }

    protected boolean addChangeSet(ChangeSet changeSet) {
        boolean allowAuthor = changeSet.hasAllowedAuthor(changeSetFilter);
        boolean allowAnyPaths = changeSet.hasAllowedPath(changeSetFilter);
        if (!allowAuthor || !allowAnyPaths) {
            def message = new StringBuilder("Changeset ${changeSet.id} skipped because ")
            if (!allowAuthor) {
                message << "it has excluded author ${changeSet.user}"
            }
            if (!allowAuthor && !allowAnyPaths) {
                message << " and "
            }
            if (!allowAnyPaths) {
                message << "it contains only excluded file paths (${changeSet.fileSet.collect{it.path}})"
            }
            println message
        }
        
        return allowAuthor && allowAnyPaths
    }
    
    private void sendPostRequest(String xml) {
        // construct the URL with property replacements
        String url = changesUrl
        def authToken = System.getenv("AUTH_TOKEN")
        
        println "Sending request to $url"

        // Debug/testing stub
       if (url.startsWith("file://")) {
           File xmlOut = new File(directory, "xmlOut.xml")
           xmlOut << xml
       }
       else {
           HttpPost postMethod = new HttpPost(url)
           if (authToken) {
               postMethod.addHeader("Authorization-Token", authToken)
               postMethod.addHeader("Content-Type", "application/xml")
           }
           
           println "Sending ${changeSets.size()} changes"
           postMethod.setEntity(new StringEntity(xml));
           
           HttpClientBuilder builder = new HttpClientBuilder()
           builder.setTrustAllCerts(true)
           HttpClient client = builder.buildClient()
    
           HttpResponse response = client.execute(postMethod)
           def responseCode = response.statusLine.statusCode
           InputStream responseStream = response.entity.content
           if (isGoodResponseCode(responseCode)) {
               IO.copy(responseStream, System.out)
               println ""
           }
           else {
               IO.copy(responseStream, System.err)
               throw new RuntimeException("Failed to upload source changes. StatusCode: ${responseCode}")
           }
       }
    }
    
    private boolean isGoodResponseCode(int responseCode) {
        return responseCode >= 200 && responseCode < 300;
    }
}