#!/usr/bin/env groovy
import groovy.xml.MarkupBuilder
import com.urbancode.air.*
import com.urbancode.air.plugin.sonar.BuildLifeLinkHelper

//------------------------------------------------------------------------------
// Utility methods and classes
//------------------------------------------------------------------------------

final def apTool = new AirPluginTool(this.args[0], this.args[1]);

final def isWindows = (System.getProperty('os.name') =~ /(?i)windows/).find()
final def out = System.out
final def mvntool = isWindows ? 'mvn.bat' : 'mvn'

final def addBuildLifeLink = {linkName, linkUrl ->
    def linkHelper = new BuildLifeLinkHelper();
    linkHelper.uploadBuildLifeLink(linkName, linkUrl);
}

//-------------------------------------------------------------------------
// Scripts variables
//=========================================================================

final def props = apTool.getStepProperties()

//final File resourceHome = new File(System.getenv()['PLUGIN_HOME'])
final def buildLifeId = System.getenv()['BUILD_LIFE_ID']
final def workDir = new File('.').canonicalFile

final def sonarUrl        = props['automation/sonarUrl']
final def sonarDbUrl      = props['automation/sonarDbUrl']
final def sonarDbUser     = props['automation/sonarDbUser']
final def sonarDbPass     = props['automation/sonarDbPass']
final def sonarDbDriver   = props['automation/sonarDbDriver']
final def sonarVersion    = props['automation/sonarVer']

final def mvnLocation     = props['mvnLocation']
final def qualifiedMvnExe = mvnLocation ? mvnLocation + File.separator + 'bin' + File.separator + mvntool : mvntool
final def mavenopts       = props['mavenOpts']    ?: ""
final def mavenArgs       = props['mavenArgs']    ?: ''

final def createPom       = Boolean.valueOf(props['createPom'])
final def modelVer        = props['modelVersion'] ?: '4.0.0'
final def groupID         = props['groupId']      ?: 'Urbancode'
final def artId           = props['artifactId']
final def projName        = props['projName']
final def srcdir          = props['srcDir']
final def bindir          = props['binDir']
final def snrExcludes     = props['excludes']
final def versionId       = props['version']      ?: '1.0.0'

def commandLine = [qualifiedMvnExe];
commandLine << '--batch-mode'
if (mavenArgs) {
    commandLine.addAll(mavenArgs.tokenize())
}
if (sonarVersion) {
    commandLine << 'org.codehaus.mojo:sonar-maven-plugin:'+sonarVersion+':sonar'
}
else {
    commandLine << 'sonar:sonar'
}

//------------------------------------------------------------------------------
//Script content
//------------------------------------------------------------------------------

if (!createPom) {
    // Pom exists. Not need to create.
}
else{
    def writer = new StringWriter()
    def builder = new groovy.xml.MarkupBuilder(writer) 
    builder.project() {
      modelVersion(modelVer)
      groupId(groupID)
      artifactId(artId)
      name(projName)
      version(versionId)
      build(){
        sourceDirectory(srcdir)
        outputDirectory(bindir)
        plugins() {
          plugin() {
            groupId("org.apache.maven.plugins")
            artifactId("maven-compiler-plugin")
             configuration() {
              source("1.5")
              target("1.5")
              excludes(){
                exclude("**/*.*")
              }
            }
          }
        }
      }
      properties() {
        'sonar.host.url'(sonarUrl)
        'sonar.dynamicAnalysis'("false")
        if (snrExcludes) {
            'sonar.exclusions'(snrExcludes)
        }
      }
    }
    println writer.toString();

    def file = new File("pom.xml")
    file.text = writer
}
//Configure Sonar settings that over-ride those in the Pom
if (sonarUrl) {
    commandLine << "-Dsonar.host.url=" + sonarUrl
}
if (sonarDbDriver) {
    commandLine << "-Dsonar.jdbc.driver="+sonarDbDriver
}
if (sonarDbUser) {
    commandLine << "-Dsonar.jdbc.username=" + sonarDbUser
}
if (sonarDbPass) {
    commandLine << "-Dsonar.jdbc.password=" + sonarDbPass
}
if (sonarDbUrl) {
    commandLine << "-Dsonar.jdbc.url=" + sonarDbUrl
}

// Create the process
def pb = new ProcessBuilder(commandLine).directory(workDir)

// Set environment Variables
env = pb.environment()
env.put("MAVEN_OPTS",mavenopts)
println("MAVEN_OPTS=" + mavenopts)

// Execute the process
println("command: ${pb.command().join(' ')}")
def proc = pb.start() 
proc.consumeProcessOutput(out, out);
proc.waitFor()
if(proc.exitValue()) {
  println("mvn sonar:sonar failed with exit code: " + proc.exitValue())
  System.exit(proc.exitValue());
}
else {
    // Get groupid and artifactid from pom file
    def pomfile = new File('pom.xml')
    def project = new XmlSlurper().parseText(pomfile.text)
    def pomGroupId = project.groupId
    def pomArtId = project.artifactId
    def pomName = project.name
    
    // Upload sonar key to anthill
    def sonarKey = "${pomGroupId}:${pomArtId}"
    def sonarLink = "${sonarUrl}/project/index/${sonarKey}"
    def sonarLinkName = pomName != null ? pomName : sonarKey
    addBuildLifeLink("SonarQube: ${sonarLinkName}", sonarLink)
    
    println "mvn sonar:sonar successfully completed"
}
System.exit(0);
