#!/usr/bin/env groovy
/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import groovy.xml.MarkupBuilder
import com.urbancode.air.*
import com.urbancode.air.plugin.sonar.BuildLifeLinkHelper

//------------------------------------------------------------------------------
// Utility methods and classes
//------------------------------------------------------------------------------

final def apTool = new AirPluginTool(this.args[0], this.args[1]);

final def isWindows = (System.getProperty('os.name') =~ /(?i)windows/).find()
final def out = System.out
final def mvntool = isWindows ? 'mvn.bat' : 'mvn'

final def addBuildLifeLink = {linkName, linkUrl ->
    def linkHelper = new BuildLifeLinkHelper();
    linkHelper.uploadBuildLifeLink(linkName, linkUrl);
}

//-------------------------------------------------------------------------
// Scripts variables
//=========================================================================

final def props = apTool.getStepProperties()

//final File resourceHome = new File(System.getenv()['PLUGIN_HOME'])
final def buildLifeId = System.getenv()['BUILD_LIFE_ID']
final def workDir = new File('.').canonicalFile

final def sonarUrl        = props['automation/sonarUrl']
final def sonarDbUrl      = props['automation/sonarDbUrl']
final def sonarDbUser     = props['automation/sonarDbUser']
final def sonarDbPass     = props['automation/sonarDbPass']
final def sonarDbDriver   = props['automation/sonarDbDriver']
final def sonarVersion    = props['automation/sonarVer']

final def mvnLocation     = props['mvnLocation']
final def qualifiedMvnExe = mvnLocation ? mvnLocation + File.separator + 'bin' + File.separator + mvntool : mvntool
final def mavenopts       = props['mavenOpts']    ?: ""
final def mavenArgs       = props['mavenArgs']    ?: ''

final def createPom       = Boolean.valueOf(props['createPom'])
final def modelVer        = props['modelVersion'] ?: '4.0.0'
final def groupID         = props['groupId']      ?: 'Urbancode'
final def artId           = props['artifactId']
final def projName        = props['projName']
final def slnName         = props['slnName']
final def debugMode       = props['debugMode'] == 'DEBUG'
final def snrExcludes     = props['excludes']
final def versionId       = props['version']      ?: '1.0.0'

def commandLine = [qualifiedMvnExe];
commandLine << '--batch-mode'
if (mavenArgs) {
    commandLine.addAll(mavenArgs.tokenize())
}
if (sonarVersion) {
    commandLine << 'org.codehaus.mojo:sonar-maven-plugin:'+sonarVersion+':sonar'
}
else {
    commandLine << 'sonar:sonar'
}

//------------------------------------------------------------------------------
//Script content
//------------------------------------------------------------------------------

if(!createPom) {
    // Pom exists. Not need to create.
}
else{
     if (!slnName) {
        workDir.eachFileMatch(~'.*\\.sln') { file ->
            if (!slnName && file.isFile()) {
              slnName = file.name;
            }
        }
    }

    if (slnName) {
        println("Using sln file $slnName");
    }
    else {
        println("No sln file specified and none was found as a child of $workDir");
        System.exit(1);
    }

    def writer = new StringWriter()
    def builder = new groovy.xml.MarkupBuilder(writer) 
    builder.project() {
      'modelVersion'(modelVer)
      'groupId'(groupID)
      'artifactId'(artId)
      'name'(projName)
      'version'(versionId)
      'packaging'("sln")
      'build'() {
        'plugins'() {
          'plugin'() {
              'groupId'('org.codehaus.sonar-plugins.dotnet')
              'artifactId'('maven-dotnet-plugin')
              'extensions'('true')
          }
        }
      }
      'properties'() {
        'sonar.host.url'(sonarUrl)
        'sonar.dynamicAnalysis'("false")
        if (snrExcludes) {
            'sonar.exclusions'(snrExcludes)
        }
        'sonar.language'('cs')
        'visual.studio.solution'(slnName)
        'visual.test.project.pattern'('*Test*')
        'dotnet.tool.version'("3.5")
        'maven.compiler.debug'(debugMode)
      }
    }
    println writer.toString();

    def file = new File("pom.xml")
    file.text = writer
}
//Configure Sonar settings that over-ride those in the Pom
if (sonarUrl) {
    commandLine << "-Dsonar.host.url=" + sonarUrl
}
if (sonarDbDriver) {
    commandLine << "-Dsonar.jdbc.driver="+sonarDbDriver
}
if (sonarDbUser) {
    commandLine << "-Dsonar.jdbc.username=" + sonarDbUser
}
if (sonarDbPass) {
    commandLine << "-Dsonar.jdbc.password=" + sonarDbPass
}
if (sonarDbUrl) {
    commandLine << "-Dsonar.jdbc.url=" + sonarDbUrl
}

// Create the process
def cmdHelper = new CommandHelper(workDir);
cmdHelper.addEnvironmentVariable("MAVEN_OPTS", mavenopts)
println("MAVEN_OPTS=" + mavenopts)
cmdHelper.runCommand("launching mvn sonar:sonar", commandLine) // message is first arg
// throws exception if non-zero exit code

// Get groupid and artifactid from pom file
def pomfile = new File('pom.xml')
def project = new XmlSlurper().parseText(pomfile.text)
def pomGroupId = project.groupId
def pomArtId = project.artifactId
def pomName = project.name

// Upload sonar key to anthill
def sonarKey = "${pomGroupId}:${pomArtId}"
def sonarLink = "${sonarUrl}/project/index/${sonarKey}"
def sonarLinkName = pomName != null ? pomName : sonarKey
addBuildLifeLink("SonarQube: ${sonarLinkName}", sonarLink)

println "mvn sonar:sonar successfully completed"
