#!/usr/bin/env groovy

import groovy.xml.StreamingMarkupBuilder
import com.urbancode.anthill3.AHPTool
import com.urbancode.anthill3.CommandHelper

//------------------------------------------------------------------------------
// GET ALL INPUT PARAMETERS
//------------------------------------------------------------------------------
try {
    final def workDir = new File('.').canonicalFile

    final def ahpTool = new AHPTool();

    final def props = ahpTool.getStepProperties();
    final def serverUrl = props['repo/serverUrl']
	final def serverUserName = props['repo/userName']
	final def serverPassword = props['repo/password']
	final def tfsVersion = props['repo/serverVersion']
	final def workItemProperty = props['workItemProperty']
	final def workItemRegex = props['workItemRegex']

    def workItemMap = [:] as Map

    // get changeset info
    if (workItemProperty) {
        new XmlSlurper().parseText(ahpTool.getChangeSetsXml())."change-set".each { element ->
            if (element.'properties'.size() > 0) {
                element.'properties'[0].'property'.each {
                    if (workItemProperty.equalsIgnoreCase(it.'name'[0].text())) {
                        it.'value'[0].text().split(',').each {
                            workItemMap.put(it,element.'anthill-id'[0].text())
                        }
                    }
                }

            }
        }
    }

    if (workItemRegex) {
        ahpTool.getMatchingChangeSets(workItemRegex).each {
            workItemMap.put(it.key, it.value.'anthill-id'[0].text())
        }
    }

    // now prepare tftool command
    if (workItemMap.size() > 0) {
        def command = ['tftool' + tfsVersion + '.exe', 'report', "/server:$serverUrl"]

        if (serverUserName && serverPassword) {
            command << "/user:$serverUserName"
            command << "/password:$serverPassword"
        }

        command << workItemMap.keySet().join(',')

        def workItemXmlBuilder = new StreamingMarkupBuilder()
        def workItemXml
        def hasIssues = false

        new CommandHelper(workDir).runCommand('Getting Work Item Report', command) { proc ->
            proc.consumeProcessErrorStream(System.out)
            proc.out.close()

            workItemXml = workItemXmlBuilder.bind() {
                proc.in.withReader { reader ->
                    issues() {
                        def line = reader.readLine()
                        while (line != null) {
                            if (line.startsWith('ID:')) {
                                hasIssues = true
                                def workItemId = line.substring(3)
                                println "Processing work item $workItemId"
                                issue(id:workItemId, "issue-tracker":"TFS-plugin", "change-id":workItemMap[workItemId]) {
                                    line = reader.readLine()
                                    while (line != null && !line.startsWith("===")) {
                                        if (line.startsWith('Type:')) {
                                            type(line.substring(5))
                                        }
                                        else if (line.startsWith('Title:')) {
                                            name(line.substring(6))
                                        }
                                        else if (line.startsWith('State:')) {
                                            status(line.substring(6))
                                        }
                                        else if (line.startsWith('Description:')) {
                                            description(line.substring(12))
                                        }
                                        line = reader.readLine()
                                    }
                                }
                            }
                            else {
                                println line
                            }
                            line = reader.readLine()
                        }
                    }
                }
            }
            ahpTool.addIssues(workItemXml)

            proc.waitFor()

            if (proc.exitValue()) {
                println "tftool command returned exit code ${proc.exitValue()}"
                System.exit(1)
            }

            if (!hasIssues) {
                println "TFS did not return any results for work items: ${workItemMap.keySet().join(',')}"
            }
        }
    }
    else {
        println 'Did not detect any work items in changelog!'
    }
}
catch (Throwable t) {
    t.printStackTrace()
    System.exit 1
}