/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2017. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import com.urbancode.air.CommandHelper

import groovy.json.JsonSlurper

import org.codehaus.jettison.json.JSONObject
import com.urbancode.air.ExitCodeException
import com.urbancode.build.plugin.ucd.rest.AppProcessRequest
import org.apache.http.client.methods.CloseableHttpResponse
import org.apache.http.client.methods.HttpGet
import org.apache.http.client.methods.HttpPut
import org.codehaus.jettison.json.JSONArray
import org.codehaus.jettison.json.JSONException
import org.apache.commons.lang3.StringUtils
import org.apache.log4j.Logger



class UCDBase extends HelperBase {
    private Logger logger = Logger.getLogger(UCDBase.class)
    def component
    def version
    def isWindows

    public def getVersionStatuses(def status) {
        StatusBase statusFoundInBuild = null
        StatusBase statusFoundInDeploy = null
        CommandHelper helper = getCommandHelper()
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                       "--weburl", url,
                       "getStatuses",
                       "--type", "version"
        ]
        addCredentials(command)

        def statusOutput
        helper.ignoreExitValue(true)
        int exitCode = helper.runCommand("Getting the list of version statuses", command) { proc ->
            proc.out.close() // close stdin
            Thread t = proc.consumeProcessErrorStream(System.out)
            statusOutput = proc.in.text.trim()
            t.join(10000L)
        }
        helper.ignoreExitValue(false)

        StatusHelper statusHelper = new StatusHelper()
        statusHelper.workDir = workDir
        statusHelper.javaFile = javaFile
        statusHelper.udClientJarFile = udClientJarFile
        statusHelper.url = url
        statusHelper.user = user
        statusHelper.password = password
        statusHelper.ucdAuthToken = ucdAuthToken
        statusHelper.cmdHelper = getCommandHelper()

        if (exitCode == 0) {
            def statusList = new JsonSlurper().parseText(statusOutput).name
            if (!statusList) {
                println "No statuses were found in the IBM UrbanCode Deploy server."
            }
            else if (!statusList.contains(status)) {
                println "The status '${status}' does not exist in the IBM UrbanCode Deploy server. Valid statuses: ${statusList}"
            }
            else {
                statusFoundInDeploy = statusHelper.checkUCDStatus(statusOutput, status);
            }
        }
        else {
            println "Could not get any status information from IBM UrbanCode Deploy server. Trying to add the status to the version anyway."
        }

        statusFoundInBuild = statusHelper.checkUCBStatus(status)
        statusHelper.syncStatusBetweenBuildAndDeploy(statusFoundInBuild, statusFoundInDeploy, status)
    }


    public def addVersionStatus(def status) {
        CommandHelper helper = getCommandHelper()
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                                 "--weburl", url,
                                 "addVersionStatus",
                                 "--component", component,
                                 "--version", version,
                                 "--status", status
        ]
        addCredentials(command)

        try {
            helper.runCommand("Adding status '${status}' to version '${version}' on component '${component}'", command)
        }
        catch (Exception e) {
            println "Failed to assign status ''${status}' to version '${version}' on component '${component}'. " +
                    "This is typically caused by using a status that does not exist."
            System.exit(1)
        }
    }

    public def createVersion(def description) {
        CommandHelper helper = getCommandHelper()
        helper.ignoreExitValue(true)
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                       "--weburl", url,
                       "createVersion",
                       "--component", component,
                       "--name", version]

        if (description) {
            command << '--description'
            command << description
        }

        addCredentials(command)

        def versionId
        def createVersionOutput
        def outBuffer
        int exitCode = helper.runCommand("Creating IBM UrbanCode Deploy Component Version", command) { Process proc ->
            proc.out.close() // close stdin
            Thread t = proc.consumeProcessErrorStream(System.out)
            createVersionOutput = proc.in.text.trim()
            t.join(10000L)
        }
        helper.ignoreExitValue(false)

        // Only try to parse as JSON if there was no error
        if (exitCode == 0) {
            def versionJson = new JSONObject(createVersionOutput)
            versionId = versionJson.getString("id")
        }
        else {
            println "Unable to create version ${version} on component ${component}: " + createVersionOutput
        }

        return versionId
    }

    public void publishFiles(def baseDir, def includes, def excludes) {
        CommandHelper helper = getCommandHelper()
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                   "--weburl", url,
                   "addVersionFiles",
                   "--component", component,
                   "--version", version,
                   "--base", baseDir,
                   "--verbose"]

        if (includes) {
            includes.split("\\n").each { include ->
                if (isWindows) {
                    include = "\"${include}\""
                }
                command << "--include" << include
            }
        }

        if (excludes) {
            excludes.split("\\n").each { exclude ->
                if (isWindows) {
                    exclude = "\"${exclude}\""
                }
                command << "--exclude" << exclude
            }
        }

        addCredentials(command)

        def commandOutput
        println "Publishing files for version ${version} on component ${component}"
        helper.runCommand("Publish IBM UrbanCode Deploy Component Version Files", command) { proc ->
            def out = new StringBuffer()
            def err = new StringBuffer()
            proc.out.close() // close stdin
            proc.waitForProcessOutput(out, err)
            commandOutput = err.toString().trim()
            println commandOutput
        }
        if (commandOutput.contains("Request error")) {
            System.exit(1)
        }
        println "Files published"
    }

    public void addVersionLink(def linkName, def linkUrl) {
        CommandHelper helper = getCommandHelper()
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                   "--weburl", url,
                   "addVersionLink",
                   "--component", component,
                   "--version", version,
                   "--linkName", linkName,
                   "--link", linkUrl]
        addCredentials(command)
        helper.runCommand("Adding link ${linkName} (${linkUrl}) to version ${version} on component ${component}", command)
    }

    /**
     * Request an application process with either a snapshot or component & version.
     * @param request An object containing all of the application process request fields.
     * @return The ID of the application process request.
     */
    public String deploy(AppProcessRequest request) {
        logger.info("Requesting application process with the following configuration '${request}'.")

        Map<String, List<String>> versions
        if (request.snapshot) {
            /* Empty versions map still required even if snapshot is being deployed */
            versions = new HashMap<String, List<String>>()
        }
        else if (request.versions) {
            versions = request.versions
        }
        else {
            throw new ExitCodeException("No snapshot or component version specified.")
        }

        UUID requestId = requestApplicationProcess(request.application, request.applicationProcess, request.description,
                request.environment, request.snapshot, request.onlyChanged, versions, request.properties)

        return requestId.toString()
    }

    public UUID requestApplicationProcess(String appName, String processName, String description, String envName,
            String snapshot, boolean onlyChanged, Map<String, List<String>> componentVersions, Map<String, String> requestProps)
    throws IOException, JSONException {
        UUID result = null
        JSONObject jsonToSend = new JSONObject()

        jsonToSend.put("application", appName)
                .put("applicationProcess", processName)
                .put("environment", envName)
                .put("onlyChanged", onlyChanged)
        if (!"".equals(description) && description != null) {
            jsonToSend.put("description", description)
        }
        if (!"".equals(snapshot) && snapshot != null) {
            jsonToSend.put("snapshot", snapshot)
        }
        JSONArray cvMappings = new JSONArray()
        Set<String> keys = componentVersions.keySet()
        for (String component : keys) {
            List<String> versions = componentVersions.get(component)

            for (String version : versions) {
                JSONObject cvMapping = new JSONObject()
                        .put("component", component)
                        .put("version", version)
                cvMappings.put(cvMapping)
            }
        }
        jsonToSend.put("versions", cvMappings)

        /* get any required properties for the application process */
        /* needed to resolve any component process properties that have been referenced */
        JSONArray unfilledProps = getUnfilledApplicationProcessRequestProperties(appName, processName, snapshot)
        JSONObject properties = new JSONObject()
        if (unfilledProps.length() > 0) {
            for (int i = 0; i < unfilledProps.length(); i++) {
                JSONObject property = unfilledProps.getJSONObject(i)
                String name = property.getString("name")
                String value = property.getString("value")
                properties.put(name, value)
            }
        }

        if (requestProps.size() > 0) {
            Set<String> nameKeys = requestProps.keySet()
            for (String name : nameKeys) {
                String value = requestProps.get(name)
                properties.put(name, value)
            }
        }

        jsonToSend.put("properties", properties)

        String uri = url + "/cli/applicationProcessRequest/request"
        HttpPut method = new HttpPut(uri)
        method.setEntity(getStringEntity(jsonToSend))
        CloseableHttpResponse response = invokeMethod(method)

        try {
            String body = getBody(response)
            JSONObject jsonResult = new JSONObject(body)
            result = UUID.fromString((String) jsonResult.get("requestId"))
        }
        finally {
            response.close()
        }

        return result
    }

    public JSONArray getUnfilledApplicationProcessRequestProperties( String appName, String processName, String snapshotName)
    throws IOException, JSONException {
        JSONArray properties

        String uri = url + "/cli/applicationProcess/unfilledProperties?application=" +
                encodePath(appName) + "&processName=" +
                encodePath(processName)

        if (StringUtils.isNotEmpty(snapshotName)) {
            uri += "&snapshot=" + encodePath(snapshotName)
        }

        HttpGet method = new HttpGet(uri)
        CloseableHttpResponse response = invokeMethod(method)

        try {
            String body = getBody(response)
            properties = new JSONArray(body)
        }
        finally {
            response.close()
        }

        return properties
    }
}
