/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* (c) Copyright IBM Corporation 2012, 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import com.urbancode.air.CommandHelper
import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder
import groovy.json.JsonSlurper
import org.apache.http.client.HttpClient
import org.codehaus.jettison.json.JSONObject

class UCDBase {
    def workDir
    def javaFile
    def udClientJarFile

    def url
    def user
    def password
    def ucdAuthToken

    def component
    def version

    def CommandHelper cmdHelper
    def HttpClient client

    public def getVersionStatuses(def status) {
        CommandHelper helper = getCommandHelper()
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                       "--weburl", url,
                       "getStatuses",
                       "--type", "version"
        ]
        addCredentials(command)

        def statusOutput
        helper.ignoreExitValue(true)
        int exitCode = helper.runCommand("Getting the list of version statuses", command) { proc ->
            proc.out.close() // close stdin
            proc.consumeProcessErrorStream(System.out) // forward stderr
            statusOutput = proc.text.trim()
        }
        helper.ignoreExitValue(false)

        if (exitCode == 0) {
            def statusList = new JsonSlurper().parseText(statusOutput).name
            if (!statusList) {
                println "No statuses were found on the IBM UrbanCode Deploy server."
                System.exit(1)
            }

            if (!statusList.contains(status)) {
                println "The status '${status}' does not exist on the IBM UrbanCode Deploy server. Valid statuses: ${statusList}"
                System.exit(1);
            }
        }
        else {
            println "Could not get any status information from IBM UrbanCode Deploy. Trying to add the status to the version anyway."
        }
    }

    public def addVersionStatus(def status) {
        CommandHelper helper = getCommandHelper()
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                                 "--weburl", url,
                                 "addVersionStatus",
                                 "--component", component,
                                 "--version", version,
                                 "--status", status
        ]
        addCredentials(command)

        try {
            helper.runCommand("Adding status '${status}' to version '${version}' on component '${component}'", command)
        }
        catch (Exception e) {
            println "Failed to assign status ''${status}' to version '${version}' on component '${component}'. " +
                    "This is typically caused by using a status that does not exist."
            System.exit(1)
        }
    }

    public def createVersion(def description) {
        CommandHelper helper = getCommandHelper()
        helper.ignoreExitValue(true)
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                       "--weburl", url,
                       "createVersion",
                       "--component", component,
                       "--name", version]

        if (description) {
            command << '--description'
            command << description
        }

        addCredentials(command)

        def versionId
        def createVersionOutput
        int exitCode = helper.runCommand("Creating IBM UrbanCode Deploy Component Version", command) { Process proc ->
            proc.out.close() // close stdin
            proc.consumeProcessErrorStream(System.out) // forward stderr
            createVersionOutput = proc.text.trim()
        }
        helper.ignoreExitValue(false)

        // Only try to parse as JSON if there was no error
        if (exitCode == 0) {
            def versionJson = new JSONObject(createVersionOutput)
            versionId = versionJson.getString("id")
        }
        else {
            println "Unable to create version ${version} on component ${component}"
        }

        return versionId
    }

    public void publishFiles(def baseDir, def includes, def excludes) {
        CommandHelper helper = getCommandHelper()
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                   "--weburl", url,
                   "addVersionFiles",
                   "--component", component,
                   "--version", version,
                   "--base", baseDir,
                   "--verbose"]

        if (includes) {
            includes.split("\\n").each { include ->
                command << "--include" << include
            }
        }

        if (excludes) {
            excludes.split("\\n").each { exclude ->
                command << "--exclude" << exclude
            }
        }

        addCredentials(command)

        def commandOutput
        println "Publishing files for version ${version} on component ${component}"
        helper.runCommand("Publish IBM UrbanCode Deploy Component Version Files", command) { proc ->
            def out = new StringBuffer()
            def err = new StringBuffer()
            proc.out.close() // close stdin
            proc.waitForProcessOutput(out, err)
            commandOutput = err.toString().trim()
        }
        if (commandOutput.contains("Request error")) {
            println commandOutput
            System.exit(1)
        }
        println "Files published"
    }

    public void addVersionLink(def linkName, def linkUrl) {
        CommandHelper helper = getCommandHelper()
        def command = [javaFile.absolutePath, '-jar', udClientJarFile.absolutePath,
                   "--weburl", url,
                   "addVersionLink",
                   "--component", component,
                   "--version", version,
                   "--linkName", linkName,
                   "--link", linkUrl]
        addCredentials(command)
        helper.runCommand("Adding link ${linkName} (${linkUrl}) to version ${version} on component ${component}", command)
    }

    public void addCredentials(def command) {
        if (ucdAuthToken) {
            command << "--authtoken" << ucdAuthToken
        }
        else {
            command << "--username" << user << "--password" << password
        }
    }

    public CommandHelper getCommandHelper() {
        if (!cmdHelper) {
            cmdHelper = new CommandHelper(workDir)
        }
        return cmdHelper
    }

    public HttpClient getHttpClient() {
        if (!client) {
            def httpUser = user
            def httpPassword = password
            if (ucdAuthToken) {
                httpUser = "PasswordIsAuthToken"

                JSONObject tokenJson = new JSONObject()
                tokenJson.put("token", ucdAuthToken)
                httpPassword = tokenJson.toString()
            }

            HttpClientBuilder clientBuilder = new HttpClientBuilder()
            clientBuilder.setTrustAllCerts(true)
            clientBuilder.setUsername(httpUser)
            clientBuilder.setPassword(httpPassword)
            client = clientBuilder.buildClient()
        }
        return client
    }
}
