/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Deploy
 * (c) Copyright IBM Corporation 2016, 2017. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */

import groovy.json.JsonSlurper

import javax.servlet.http.*

import org.apache.http.HttpResponse
import org.apache.http.HttpStatus
import org.apache.http.client.methods.HttpGet
import org.apache.http.util.EntityUtils

import com.urbancode.air.plugin.artifactory.ArtifactoryHelper
import com.urbancode.commons.httpcomponentsutil.HttpClientBuilder
import com.urbancode.commons.util.IO
import com.urbancode.air.AirPluginTool

final AirPluginTool airPluginTool = new AirPluginTool(this.args[0], this.args[1])
final Properties props = airPluginTool.getStepProperties()

def isEmpty(value) {
    return value == null || value.equals("")
}

final def workDir = new File('.').canonicalFile

final def REPO_PATH_SEPARATOR = "/"

def repoName = props['repositoryName']
def artifacts = props['artifacts'].split('\n')
def username = props['username']
def password = props['password']
def apiKey = props['apiKey'].trim()
def checkHash = props['checkHash']
def repoUrl = props['artifactoryUrl']
while (repoUrl.endsWith(REPO_PATH_SEPARATOR)) {
    repoUrl = repoUrl.substring(0, repoUrl.length() - 1)
}
boolean allowInsecure = (props['allowInsecure']?:"false").toBoolean()

def exitVal = 0
ArtifactoryHelper aHelper = new ArtifactoryHelper(username, password, apiKey, allowInsecure)

def searchArtifacts = { searchUrl ->
    println ("[Action] Searching for artifacts at: ${searchUrl}")
    def response = aHelper.executeHttpRequest(searchUrl, new HttpGet(), null)
    if (!response) {
        println ('[Error] Could not find artifacts')
        exitVal=1
        return []
    }
    int status = response.getStatusLine().getStatusCode()
    if (status == HttpStatus.SC_OK) {
        def artifactUris = []
        def jsonString = EntityUtils.toString(response.getEntity())
        def slurper = new JsonSlurper()
        def resultArr = slurper.parseText(jsonString).results
        for (result in resultArr) {
            artifactUris.add(result.uri)
        }
        if (artifactUris.size() > 0) {
            println ("[Ok] Found ${artifactUris.size()} artifacts")
        }
        else {
            println ("[Ok] Did not find any child artifacts to download")
        }
        return artifactUris
    }
    else {
        throw new Exception("[Error] Exception searching: " + searchUrl + "\nErrorCode : " + status.toString())
    }
}

try {
    artifacts.each { artifact ->
        String filter = ''
        String[] attrs = artifact.split('/')
        if (attrs.length == 4) {
            filter = attrs[3]
        }
        String version = attrs[2]
        String artifactId = attrs[1]
        String groupId = attrs[0]
        String searchUrl = repoUrl + REPO_PATH_SEPARATOR + 'api/search/gavc?' +
                '&g=' + groupId + '&a=' + artifactId + '&v=' + version
        if (!isEmpty(filter)) {
            searchUrl = searchUrl + '&c=' + filter
        }
        if (!isEmpty(repoName)) {
            searchUrl = searchUrl + '&repos=' + repoName
        }

        def artifactUris = searchArtifacts(searchUrl)
        for (artifactUri in artifactUris) {
            File artifactFile = aHelper.downloadFileFromRepo(artifactUri, checkHash.toString())
            if (artifactFile == null) {
                throw new Exception("[Error] Failed to download artifact : " + artifact)
            }

            //copy the temp file to this directory with the file name
            String[] currFile = artifactUri.split(REPO_PATH_SEPARATOR)
            def filename = currFile[currFile.length - 1]
            File finalFile = new File(workDir, filename)
            println "[Action] Moving downloaded artifact to : " + finalFile.getAbsolutePath() + "\n"
            try {
                IO.move(artifactFile, finalFile)
            }
            catch (IOException ex) {
                println ("[Error] Could not move downloaded artifact: ${ex.getMessage()}")
                exitVal=1
            }
            artifactFile.delete()
        }
    }
}
catch (Exception e) {
    e.printStackTrace()
    exitVal=1
}
System.exit(exitVal)
