
// <copyright
//     notice="cicscm-rm-source-program"
//     pids="CA0W"
//     years="2015"
//     crc="2054783065" >
//     Licensed Materials - Property of IBM
//     "Restricted Materials of IBM"
//     CA0W
//     (C) Copyright IBM Corp. 2015
//     US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
// </copyright>

package com.ibm.cics.cm.ucd.steps

import java.text.MessageFormat

import org.apache.commons.lang.RandomStringUtils

import java.nio.charset.Charset

import com.ibm.cics.cm.model.runtime.CMServerException
import com.ibm.cics.cm.model.runtime.ConfigurationManager
import com.ibm.cics.cm.model.runtime.MessageResponse
import com.ibm.cics.cm.model.runtime.CMConnection
import com.ibm.cics.cm.model.MigrationPath
import com.ibm.cics.cm.model.MigrationScheme
import com.ibm.cics.cm.ucd.MigrationSchemeHelper
import com.ibm.cics.cm.ucd.msg.CICSUCDMessage
import com.ibm.cics.cm.ucd.properties.CommonCICSNameProperty
import com.ibm.cics.cm.ucd.properties.MigrationSchemeDescriptionProperty
import com.ibm.cics.cm.ucd.properties.StringProperty
import com.ibm.cics.cm.ucd.steplabel.CICSUCDStepLabel
import com.ibm.cics.core.model.ICPSM
import com.urbancode.air.AirPluginTool

import static com.ibm.cics.cm.ucd.StepLogHelper.printStepLog

class CreateMigrationSchemeStep extends CICSStep {
	//  <copyright
	//      notice="cicscm-rm-object-program"
	//      pids="CA0W"
	//      years="2015"
	//      crc="2054783065" >
	//      Licensed Materials - Property of IBM
	//      "Restricted Materials of IBM"
	//      CA0W
	//      (C) Copyright IBM Corp. 2015
	//      US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
	//  </copyright>
	
	public CreateMigrationSchemeStep(String inputPropertyFile, String outputPropertyFile) {
		super(inputPropertyFile, outputPropertyFile)
	}

	public CreateMigrationSchemeStep(AirPluginTool apTool) {
		super(apTool)
	}

	@Override
	protected void runCICSStep(CMConnection conn) throws CMServerException {

		def props = apTool.getStepProperties()
		def sourceConfig = new CommonCICSNameProperty(CICSUCDStepLabel.SOURCE_CONFIG, true).parse(props['sourceConfig']).getValue()
		def targetConfig = new CommonCICSNameProperty(CICSUCDStepLabel.TARGET_CONFIG, true).parse(props['targetConfig']).getValue()
		def transfRule = new CommonCICSNameProperty(CICSUCDStepLabel.TRANSF_RULE, false).parse(props['transfRule']).getValue()
		def migrationSchemeDesc = new MigrationSchemeDescriptionProperty(CICSUCDStepLabel.MIGRATION_SCHEME_DESC, false).parse(props['migrationSchemeDesc']).getValue()
		
		MigrationSchemeHelper schemeHelper = new MigrationSchemeHelper()
		ConfigurationManager configManager = new ConfigurationManager()
		configManager.setConnection(conn)
		
		// create the migration path - (delete at source set to false)
		MigrationPath migrationPath = new MigrationPath(sourceConfig, targetConfig, transfRule, false)

		// assign random migration scheme name and description
		def migrationScheme = generateRandomMigrationSchemeName(configManager)	
		if (migrationSchemeDesc.isEmpty()) migrationSchemeDesc = "UCD: Auto-generated migration scheme"

		try {
			// create migration scheme
			def serverResponse = buildMigrationScheme(migrationScheme, migrationSchemeDesc, migrationPath, configManager, schemeHelper)
			// creation of migration scheme successful 
			printStepLog(MessageFormat.format(CICSUCDMessage.createMigrationSchemeSuccessful, migrationScheme, serverResponse.getReturnResponse()))
		} catch(CMServerException e) {
			printStepLog(MessageFormat.format(CICSUCDMessage.createMigrationSchemeFailed, e.getMessage()))
			throw e
		}

		// set output properties for 'ready' step
		setOutputProperties(migrationScheme)

	}

	MessageResponse buildMigrationScheme(String migrationScheme, String migrationSchemeDesc, MigrationPath migrationPath, 
		ConfigurationManager configManager, MigrationSchemeHelper schemeHelper) throws CMServerException {
		 
		 List<MigrationPath> migrationPathList = new ArrayList<MigrationPath>()
		 migrationPathList.add(migrationPath)
		 
		 schemeHelper.setName(migrationScheme)
		 schemeHelper.setDescription(migrationSchemeDesc)
		 schemeHelper.setApproval(false)				// approval set to false
		 schemeHelper.setMigrationPaths(migrationPathList)
		
		 return schemeHelper.createMigrationScheme(configManager)

	}

	String generateRandomMigrationSchemeName(ConfigurationManager configManager) {

		def existingMigrationSchemesList = configManager.getMigrationSchemeNameList()
		def randomMigrationSchemeName = createRandomString()

		existingMigrationSchemesList.each {
			if(it.equals(randomMigrationSchemeName)) {
				generateRandomMigrationSchemeName(configManager)
			}
		}

		return randomMigrationSchemeName
	}

	String createRandomString() {

		def stringLength = 8
		String chars = (('A'..'Z') + ('0'..'9')).join()
		String randomString = RandomStringUtils.random(stringLength, chars.toCharArray())

		return randomString
	}


	private void setOutputProperties(String migrationScheme) {

		apTool.setOutputProperty("migrationScheme", migrationScheme)
		apTool.storeOutputProperties()
	}

}
