/*<copyright                                                          */
/* notice="cics-rm-source-program"                                    */
/* pids="CA0U"                                                        */
/* years="2014,2015"                                                  */
/* crc="1921302770" >                                                 */
/* 	Licensed Materials - Property of IBM                              */
/* 	"Restricted Materials of IBM"                                     */
/* 	CA0U                                                              */
/* 	(C) Copyright IBM Corp. 2014, 2015                                */
/* 	US Government Users Restricted Rights - Use, duplication or disclosure restricted by GSA ADP Schedule Contract with IBM Corp. */
/* </copyright>                                                       */
package com.ibm.cics.ucd;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;

import com.ibm.cics.core.model.CICSSystemManagerException;
import com.ibm.cics.core.model.Context;
import com.ibm.cics.core.model.FilteredContext;
import com.ibm.cics.core.model.ICPSM;
import com.ibm.cics.core.model.ScopedContext;
import com.ibm.cics.core.model.context.CPSMDefinitionPrimaryKey;
import com.ibm.cics.core.model.internal.CICSResource;
import com.ibm.cics.core.model.CICSObjectPrimaryKey;
import com.ibm.cics.core.ui.ExceptionMessageHelper;
import com.ibm.cics.model.ICICSAttribute;
import com.ibm.cics.model.ICICSDefinition;
import com.ibm.cics.model.ICICSObject;
import com.ibm.cics.model.ICICSType;
import com.ibm.cics.model.IDefinition;
import com.ibm.cics.sm.comm.AbstractCICSAction;
import com.ibm.cics.sm.comm.ICICSAction;
import com.ibm.cics.sm.comm.ICICSActionNames;
import com.ibm.cics.sm.comm.IContext;
import com.ibm.cics.sm.comm.IFilteredContext;
import com.ibm.cics.sm.comm.IPrimaryKey;
import com.ibm.cics.sm.comm.IResourceErrorCode;
import com.ibm.cics.sm.comm.ISystemManagerResponseValues;
import com.ibm.cics.sm.comm.ISystemManagerReasonValues;
import com.ibm.cics.sm.comm.IScopedContext;
import com.ibm.cics.sm.comm.InstallException;
import com.ibm.cics.sm.comm.SMConnectionRecord;
import com.ibm.cics.sm.comm.SMConnectionResponse;
import com.ibm.cics.sm.comm.SystemManagerActions;
import com.ibm.cics.sm.comm.SystemManagerConnectionException;
import com.ibm.cics.sm.comm.actions.SimpleSystemManagerAction;
import com.ibm.cics.ucd.exceptions.StepFailedException;
import com.ibm.cics.ucd.exceptions.CICSUCDPluginException;
import com.ibm.cics.ucd.msg.CICSUCDMessage;
import com.ibm.cics.ucd.steplabel.CICSUCDStepLabel;
import com.ibm.cics.ucd.ExceptionHandler;

import static com.ibm.cics.ucd.StepLogHelper.printStepLog;
import static com.ibm.cics.model.AttributeValue.av;

import com.ibm.cics.core.model.AbstractCICSDefinitionType;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

class CheckStatusHelper{
	private List<String> failedResourceNameList;
	private List<String> notExistResourceNameList;
	private ExceptionHandler exceptionHandler = new ExceptionHandler();
    private ICICSAction Query = new SimpleSystemManagerAction("Query"); // create our own action for QUERY as CICS explorer doesn't have this action now. This action is used by CICS UCD only.
	
	enum ResourceStatus{
		OK,NOT_IN_EXPECTED_STATUS,NOT_FOUND,
	}
	
	CheckStatusHelper(){
		failedResourceNameList = new ArrayList<String>();
		notExistResourceNameList = new ArrayList<String>();
	}
	
	//query status of resources in a list with timeout
	boolean queryActionWithTimeout(ICPSM cpsm, ICICSType type, IContext context, List<String> resourceNameList, String expectedStatus, int timeout,String checkedField,List errorList){
		int retryTimes = 0;
		//get current time
		Long startDate = System.currentTimeMillis();
		Long currentDate = System.currentTimeMillis();
		List<String> tempResourceNameList = new ArrayList<String>();

		//try the first time.
		queryStatusofResourceList(cpsm,type,context,resourceNameList,expectedStatus,checkedField,errorList);

		//retry if some of the resource is not in expected status and we haven't timeout.
		while(timeout!=0 && currentDate < (startDate + timeout*1000) && failedResourceNameList.size()>0) {
			retryTimes++;
			printStepLog(MessageFormat.format(CICSUCDMessage.StartRetryCheck, retryTimes ));
			
			// store failed resource name list into temprary one as we will clean failedresourceNameList once we enter queryStatusofResourceList
			tempResourceNameList.clear();
			tempResourceNameList.addAll(failedResourceNameList);
			queryStatusofResourceList(cpsm,type,context,tempResourceNameList,expectedStatus,checkedField,errorList);

			if(failedResourceNameList.size()!=0){
				Thread.sleep(1000); // delay for 1s
				currentDate = System.currentTimeMillis(); // update current time
			}
		}

		//print out summary result for step
		if(failedResourceNameList.size()>0|notExistResourceNameList.size()>0){
			printStepLog(MessageFormat.format(CICSUCDMessage.CheckResultSummaryFail,retryTimes+1,expectedStatus));
			return false;
		}
		else{
			printStepLog(MessageFormat.format(CICSUCDMessage.CheckResultSummarySuccess,expectedStatus));
			return true;
		}
	}
	//query status of resources in a list
	void queryStatusofResourceList(ICPSM cpsm, ICICSType type, IContext context,List<String> resourceNameList,String expectedStatus,String checkedField, List errorList){
		failedResourceNameList.clear();
		int successCount = 0;
		int failCount=0;
		//result of check
		ResourceStatus result;
		//Iterate to query each resource
		for(int i=0; i < resourceNameList.size(); i++) {
			//Set a filter for the context
			FilteredContext filteredContext = new FilteredContext(context);
			filteredContext.setAttributeValue(type.NAME, resourceNameList.get(i));
			//Perform action on the resource
			try{
			result = queryAction(cpsm, type, filteredContext, resourceNameList.get(i),expectedStatus,checkedField);
			switch(result){
				case ResourceStatus.OK:
					successCount++;
					break;
				case ResourceStatus.NOT_IN_EXPECTED_STATUS:
					failCount++;
					failedResourceNameList.add(resourceNameList.get(i)); // add the failed resource name into list.
					printStepLog(MessageFormat.format(CICSUCDMessage.CheckStatusFail,resourceNameList.get(i),expectedStatus ));
					break;
				case ResourceStatus.NOT_FOUND:
					failCount++; // if resource doesn't exist, we count it into failcount. continue the process, but we don't retry it next time
					notExistResourceNameList.add(resourceNameList.get(i));
					break;
			}
			}
			catch(CICSSystemManagerException e){
				//The most common reason that an exception happen is incorrect scope. In this situation, we treat is as not found, we will not retry and add the name into noexitnamelist.
				exceptionHandler.handleException(e, type.getResourceTableName(), Query, resourceNameList.get(i),errorList);
				failCount++; 
				notExistResourceNameList.add(resourceNameList.get(i));
			}
		}

		//print out result for each retry.
		if(failCount != 0){
			printStepLog(MessageFormat.format(CICSUCDMessage.CheckStatusFailedResult, failCount,successCount));
		}
		else{
			printStepLog(MessageFormat.format(CICSUCDMessage.CheckStatusSuccessResult, successCount,expectedStatus));
		}
		
	}

	//query status of a resource in target scope
	ResourceStatus queryAction(ICPSM cpsm, ICICSType type, IContext context,String resourceName,String expectedStatus,String checkedField){
		ResourceStatus returnedResult = ResourceStatus.OK;
		SMConnectionResponse resp = cpsm.get(type.getResourceTableName(),context); // get is used to generate a cache structure
		int recordcount = resp.getRecordTotal(); // get record number in this structure
		if(recordcount == 0){ //if we can not find the resource in target scope, we print a error message then exit.
			printStepLog(MessageFormat.format(CICSUCDMessage.ResourceNotExist,resourceName ));
			returnedResult = ResourceStatus.NOT_FOUND;
		}
		for(int i=1;i<=recordcount;i++){  //get the record from the structure one by one
			SMConnectionResponse resp2 = cpsm.fetch(resp, i, 1); // get one base on position
			SMConnectionRecord record = resp2.getRecord(0);
			if(record.get(checkedField).toUpperCase()!=expectedStatus.toUpperCase()){
				returnedResult = ResourceStatus.NOT_IN_EXPECTED_STATUS;
				printStepLog(MessageFormat.format(CICSUCDMessage.ResultofCheckIndividualResourceisFail,record.get(type.getNameAttribute()),record.get("EYU_CICSNAME"),record.get(checkedField),expectedStatus.toUpperCase()));
			}
			else{
				printStepLog(MessageFormat.format(CICSUCDMessage.CheckStatusSuccess,record.get(type.getNameAttribute()),record.get("EYU_CICSNAME"),expectedStatus.toUpperCase()));
			}

		}
		return returnedResult;
	}
	
	
}