import java.io.File;

import com.urbancode.air.AirPluginTool

final def apTool = new AirPluginTool(args[0], args[1])
def nodeName = apTool.getStepProperties()['nodeName']
def chefInstallDir = apTool.getStepProperties()['chefInstallDir']

try {
    
    //
    // validation
    //
    if( !nodeName ) {
        throw new RuntimeException("A Node Name to install is required and was undefined");
    }
    
    // remove '_node.json' if it was included by accident in the nodeName
    nodeName = nodeName.replaceAll("_node.json", "");
    
    if( !chefInstallDir ) {
        // if no path was provided assume chef-solo is on the System path
        chefInstallDir = "chef-solo" + (windows() ? ".bat" : "")
    } else {
        // remove trailing file seperator if found
        if( chefInstallDir.trim().endsWith(File.separator) ) {
            chefInstallDir += chefInstallDir.trim().substring(0, chefInstallDir.lastIndexOf(File.separator))
        }
        // construct a path to '/bin/chef-solo.(bat|sh)
        chefInstallDir += File.separator + "bin" + File.separator + "chef-solo" + (windows() ? ".bat" : "")
    }
    
    //
    // chef configuration file that set's up paths in a chef-solo runtime environment
    //
    File solo = new File(workDir(), "solo.rb")
    
    //
    // Generate solo.rb if not found in workDir
    //
    if( !solo.exists() ) {
        println("Creating chef configuration file");
        solo.withWriter { out ->
            out.writeLine("# Generated from uDeploy Chef plugin");
            out.writeLine("CHEF_DIR = File.expand_path(File.dirname(__FILE__))");
            out.writeLine("file_cache_path \"/tmp/chef-solo\"");
            out.writeLine("role_path CHEF_DIR + \"/roles\"");
            out.writeLine("environment_path CHEF_DIR + \"/environments\"");
            out.writeLine("data_bag_path CHEF_DIR + \"/data_bags\"");
            out.writeLine("cookbook_path CHEF_DIR + \"/cookbooks\"");
        }
    }
    
    //
    // Build Command Line
    //
    def commandLine = []
    commandLine.add(chefInstallDir)
    commandLine.add("-c")
    commandLine.add("solo.rb")
    commandLine.add("-j")
    commandLine.add("nodes/" + nodeName + "_node.json")
    commandLine.add("-N")    
    commandLine.add(nodeName)
     
    // print out command info
    println("")
    println("command line: ${commandLine.join(' ')}")
    println("working directory: ${workDir().path}")
    println('===============================')
    println("command output: ")

    //
    // Launch Process
    //
    final def processBuilder = new ProcessBuilder(commandLine as String[]).directory(workDir())
    final def process = processBuilder.start()
    process.out.close() // close stdin
    process.waitForProcessOutput(System.out, System.err) // forward stdout and stderr
    process.waitFor()
    
    // print results
    println('===============================')
    println("command exit code: ${process.exitValue()}")
    println("")
    
    exitCode = process.exitValue();

    System.exit(exitCode);

}

catch (Exception e) {
    System.out.println("Error installing chef node!");
    e.printStackTrace();
    System.exit(1);
}

def File workDir() {
    def work = new File('.').canonicalFile
    if (!work.exists()) {
        throw new RuntimeException("Cannot get working directory!");
    }
    
    return work
}

def String osName() {
    return System.getProperty('os.name').toLowerCase(Locale.US)
}

def boolean windows() {
    return (osName() =~ /windows/)
}
