#!/usr/bin/env groovy
/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

import com.urbancode.air.AirPluginTool
import com.urbancode.air.XTrustProvider

final def apTool = new AirPluginTool(this.args[0], this.args[1])

final def props = apTool.getStepProperties()

final def serverUrlArray = props['serverURL']?.split('://') as String[]
final def username = props['username']
final def password = props['password']
final def timeout = Integer.parseInt(props['timeout'])

final def poolNameList = props['poolName']
def poolNames = (poolNameList.tokenize('\n,').findAll {it && it.trim().length() > 0}).collect {it.trim()}
final def isPoolGlobal = props['isPoolGlobal']?.toBoolean()
final def nodeAddress = props['nodeAddress'] as String
final def nodePort = Long.parseLong(props['nodePort']) as long

try {
    if (serverUrlArray.length != 2)
        println("Malformed server URL: ${props['serverURL']} missing <protocol>://")
    if (serverUrlArray[1].endsWith('/'))
        serverUrlArray[1] = serverUrlArray[1].substring(0, serverUrlArray[1].length() - 1)

    final def connectionString = serverUrlArray[0] + "://" + username + ":" + password + "@" +
            serverUrlArray[1] + "/iControl/iControlPortal.cgi"

    XTrustProvider.install()
    def poolMemberAddress

    def getState = { poolName ->
        if (isPoolGlobal) {
            for (member in poolMemberAddress.get_enabled_state(
                    [poolName] as String[],
                    [new iControl.CommonIPPortDefinition(nodeAddress, nodePort)] as iControl.CommonIPPortDefinition[][])[0]) {
                if (member.member.address.equals(nodeAddress) && member.member.port == nodePort) return member.state
            }
        }
        else {
            for (member in poolMemberAddress.get_session_enabled_state(poolName)[0]) {
                if (member.member.address.equals(nodeAddress) && member.member.port == nodePort) return member.session_state
            }
        }
    }

    def addNode = {
        //------------------------------------------------------------------------------
        // Add node to pools
        //------------------------------------------------------------------------------
        println("Connecting to server ${props['serverURL']} using timeout of $timeout second(s)")

        if (isPoolGlobal) {
            poolMemberAddress =
                new iControl.GlobalLBPoolMemberLocator().getGlobalLBPoolMemberPort(new java.net.URL(connectionString)) as
                    iControl.GlobalLBPoolMemberBindingStub
        }
        else {
            poolMemberAddress =
                new iControl.LocalLBPoolMemberLocator().getLocalLBPoolMemberPort(new java.net.URL(connectionString)) as
                    iControl.LocalLBPoolMemberBindingStub
        }
        poolMemberAddress.setTimeout(timeout * 1000);
        poolNames.each { poolName ->
            def currentState = getState(poolName)
            println("Current node state in the pool $poolName: $currentState")
            if (currentState == iControl.CommonEnabledState.STATE_DISABLED) {
                println("Adding node $nodeAddress:$nodePort to pool $poolName")
                        if (isPoolGlobal) {
                            poolMemberAddress.set_enabled_state(
                                    [poolName] as String[],
                                    [new iControl.GlobalLBPoolMemberMemberEnabledState(
                                            new iControl.CommonIPPortDefinition(nodeAddress, nodePort),
                                            iControl.CommonEnabledState.STATE_ENABLED)
                                    ] as iControl.GlobalLBPoolMemberMemberEnabledState[][])
                        }
                        else {
                            poolMemberAddress.set_session_enabled_state(
                                    [poolName] as String[],
                                    [new iControl.LocalLBPoolMemberMemberSessionState(
                                            new iControl.CommonIPPortDefinition(nodeAddress, nodePort),
                                            iControl.CommonEnabledState.STATE_ENABLED)
                                    ] as iControl.LocalLBPoolMemberMemberSessionState[][])
                            poolMemberAddress.set_monitor_state(
                                [poolName] as String[],
                                [new iControl.LocalLBPoolMemberMemberMonitorState(
                                        new iControl.CommonIPPortDefinition(nodeAddress, nodePort),
                                        iControl.CommonEnabledState.STATE_ENABLED)
                                ] as iControl.LocalLBPoolMemberMemberMonitorState[][])
                        }
                println("New node state in pool $poolName: ${getState(poolName)}")
            }
            else if (currentState == null) {
                println("Node $nodeAddress:$nodePort was not found in pool $poolName")
            }
            else {
                println("Node $nodeAddress:$nodePort is enabled for pool $poolName so nothing to do")
            }
        }
    }

    addNode()
}
catch (Exception e) {
    e.printStackTrace()
    System.exit(1)
}
