/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Deploy
 * (c) Copyright IBM Corporation 2015. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
package com.urbancode.air.plugin.ssis

import com.urbancode.air.FileSet
import com.urbancode.air.CommandHelper

class SSISHelper {
    def workDir = new File(".").canonicalFile
    def commandPath
    def props
    def username
    def password
    def server
    def onFail

    public SSISHelper(def props) {
        this.props = props

        //optional properties
        def dirOffset = props['dirOffset']

        commandPath = props['commandPath'].trim()

        if (commandPath) {
            File dtutilPath = new File(commandPath)

            if (!dtutilPath.exists() && !dtutilPath.isFile()) {
                println("Specified DTUTIL path: ${commandPath} does not exist.")
                System.exit(1)
            }
        }
        else {
            commandPath = "dtutil"
        }

        username = props['username']
        password = props['password']
        server = props['server']
        onFail = props['onFail']

        if (dirOffset) {
            workDir = new File(workDir, dirOffset)
        }
    }

    int deployPackage() {
        CommandHelper helper = new CommandHelper(workDir)
        int exitVal
        def includes = props['includes']
        def mappedFiles = props['mappedFiles']
        def excludes = props['excludes'].split("\n")
        def absolutePaths = []
        def packages = []
        def failures = []
        def nameMappings = [:]

        if (!includes && !mappedFiles) {
            println("No SSIS packages were specified in the 'Includes' field or the 'File to Deployment Name Mappings' field.")
            System.exit(1)
        }

        //create set of included files, filtering out excludes
        FileSet fs = new FileSet(workDir)

        if (includes) {
            includes = includes.split("\n")
            for (include in includes) {
                if (include.contains('*') || include.contains('?')) {
                    fs.include(include.trim())
                }
                else {
                    File file = new File(include.trim())

                    //if user provided an absolute path, use that instead
                    if (file.exists() && file.isFile()) {
                        absolutePaths << file
                    }
                    else {
                        file = new File(workDir, include.trim())

                        if (file.exists() && file.isFile()) {
                            fs.include(include.trim())
                        }
                        else {
                            println("${include.trim()} does not exist in working directory: ${workDir.absolutePath}, nor is it a valid absolute path.")
                            System.exit(1)
                        }
                    }
                }
            }
        }

        if (mappedFiles) {
            mappedFiles = mappedFiles.split("\n")
            for (include in mappedFiles) {
                //split file name and deployment name by = character
                def splitFile = include.split('=')
                def fileName = splitFile[0].trim()
                def name = splitFile[1].trim()

                //check if user has already provided absolute file path
                def file = new File(fileName)

                if (file.exists() && file.isFile()) {
                    absolutePaths << file
                }
                //acquire absolute path from working directory
                else {
                    file = new File(workDir, fileName)
                    //absolute path provided is also invalid
                    if (!file.exists() || !file.isFile()) {
                        println("File does not exist: ${file.absolutePath}")
                        System.exit(1)
                    }
                }

                //map name and include file in fileset
                nameMappings << [(file.absolutePath) : name]
                fs.include(fileName)
            }
        }

        for (exclude in excludes) {
            fs.exclude(exclude.trim())
        }

        packages = fs.files()
        packages.addAll(absolutePaths)

        //if no matches occur for includes in the base directory
        if (!packages) {
            println("No specified files exist in the directory: ${workDir.absolutePath}")
            System.exit(1)
        }

        println("Attempting to deploy SSIS packages:")

        for (file in packages) {
            println(file.absolutePath)
        }

        for (file in packages) {
            def cmdArgs = [commandPath]
            def deployName

            //use name mapping for deployment
            if (nameMappings[file.absolutePath]) {
                deployName = nameMappings[file.absolutePath]
            }
            //create deployment name from package name
            else {
                def fileName = file.getName()
                deployName = fileName.substring(0, fileName.lastIndexOf('.'))
            }

            cmdArgs << "/FILE"
            cmdArgs << file.absolutePath
            cmdArgs << "/COPY"
            cmdArgs << "SQL;${deployName}"
            cmdArgs << "/QUIET"

            //optional arguments
            if (server) {
                cmdArgs << "/SERVER"
                cmdArgs << server
            }
            if (username) {
                cmdArgs << "/DESTUSER"
                cmdArgs << username
            }
            if (password) {
                cmdArgs << "/DESTPASSWORD"
                cmdArgs << password
            }

            exitVal = helper.runCommand("Deploying package: ${file.absolutePath}", cmdArgs)

            if (exitVal != 0) {
                if (onFail == "fastFail") {
                    println("Deployment failed on file: ${file.absolutePath} process will now exit.")
                    return exitVal
                }
                else if (onFail == "bestEffort") {
                    failures << file.absolutePath
                    continue
                }
                else {
                    println("Deployment failed on file: ${file.absolutePath} process will continue with remaining deployments.")
                    continue
                }
            }
            else {
                println("Successful deployment of file: ${file.absolutePath}")
            }
        }

        //best effort deployment reporting failures after attempting all files
        if (failures) {
            for (fail in failures) {
                println("Deployment failed on file: ${fail}")
            }

            println("Process will now exit due to errors.")
            System.exit(1)
        }
        else {
            return exitVal
        }
    }
}
