<?php
/****************************************************************
* IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2015
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
* **************************************************************/

/**
 * API for component requests via REST
 *
 * @author Felice Geracitano.......feliecege@ie.ibm.com
 */
class ComponentRestApi extends AbstractApi {
    
    /**
     *
     * Add files to a component version
     *
     * @param string $component
     * @param string $version
     * @param string $base
     * @param array  $include optional
     * @param array  $exclude optional
     */
    public function addVersionFiles($component = null, $version = null, $base = null, $include = null, $exclude = null) {
        $includeFiles = "";
        if (!empty($include)){
            if (is_string($include)) {
                $includeFiles = " -include {$include}";
            }
            else if (is_array($include)) {
                foreach ($include as $file) {
                    $includeFiles .= " -include {$file}";
                }
            }
        }
        $excludeFiles = "";
        if (!empty($exclude)){
            if (is_string($exclude)) {
                $excludeFiles = " -exclude {$exclude}";
            }
            else if (is_array($exclude)) {
                foreach ($exclude as $file) {
                    $excludeFiles .= " -exclude {$file}";
                }
            }
        }
        $action = "addVersionFiles -component '{$component}' -version {$version} -base {$base}{$includeFiles}{$excludeFiles}";
        return $this->exec($action);
    }
    
    
    /**
     * Set up a uDeploy API Request for Creation of a componentTemplate,
     * based on a JSON formatted file.
     *
     * @param string $componentTemplate Path to the JSON File, to be used for ComponentTemplate creation
     *
     * @return boolean|string String, holding the uDeploy API action to be executed OR
     *                        Boolean FALSE, when $componentTemplate File is not provided
     */
    public function createComponentTemplate($componentTemplate = null) {
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/deploy/componentTemplate -d @'{$componentTemplate}'";
    
        return $this->execRest($action);
    }
    
    /**
     *
     * Deletes a component using name or ID
     *
     * @param unknown $component
     */
    public function deleteComponent($component = null){
        $command = "-X DELETE ";
        $command .= "'{$this->weburl}/rest/deploy/component/{$component}'";
        return $this->execRest($command);
    }
    
    /**
     *
     * Deletes a componentProcess using ID
     *
     * @param unknown $component ID
     */
    public function deleteComponentProcess($componentProcessID = null){
        $command = "-X DELETE ";
        $command .= "'{$this->weburl}/rest/deploy/componentProcess/{$componentProcessID}'";
        return $this->execRest($command);
    }
    
    /**
     * Set a uDeploy DELETE action to be executed for
     * Deletion of a Component Template using its ID
     *
     * @param integer $componentTemplateID
     *
     * @return string String holding the uDeploy action for the Deletion of a Component Template
     */
    public function deleteComponentTemplate($componentTemplateID){
        $command = "-X DELETE ";
        $command .= "{$this->weburl}/rest/deploy/componentTemplate/{$componentTemplateID} ";
    
        return $this->execRest($command);
    }
    
    /**
     *
     * @param string $environment
     * @return string
     *
     * Get component environment properties for an environment
     */
    public function getComponentEnvironment($environmentId = null) {
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/{$environmentId}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/environment/{$environmentId}/componentProperties {$this->destination}";
        return $this->execRest( $action );
    }
    
    /**
     *
     * Export a component process to a file or returns json
     *
     * @param string $componentID
     * @param string $componentName
     * @return boolean|string
     */
    public function getComponentProcessRest($componentProcessID = null, $componentProcessName = null){
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/{$componentProcessName}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/componentProcess/{$componentProcessID}/-1 {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Export a component to a file or returns json
     *
     * @param string $componentID
     * @param string $componentName
     * @return boolean|string
     */
    public function getComponentRest($componentID = null, $componentName = null){
        if (empty($componentName)) {
            $componentName = $componentID;
        }
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/{$componentName}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/component/{$componentID}/export {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Export all components to a file or returns a json
     *
     * @return boolean|string
     */
    public function getComponentsRest(){
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/components.json'";
        }
        $action = "{$this->weburl}/rest/deploy/component?active=true {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * Set a uDeploy API request to obtain FULL data for a specified ComponentTemplate.
     * Note: ComponentTemplate - Details can be obtaine only by using ComponentTemplate ID.
     *
     * @param integer $template_id      Integer, holding the ID of the ComponentTemplate to be retrieved.
     *                                  This is required to build the API request.
     * @param string $template_name     String, holding the name of the ComponentTemplate to obtain.
     *                                  This is required for when we are saving the Template into a file.
     *
     * @return boolean|string           The uDeploy API request to be executed or
     *                                  FALSE in case of a problem with passed arguments.
     */
    public function getComponentTemplateDetails($template_id=null, $template_name=null) {
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$template_name}.json'";
        }
        $action = $this->weburl . '/rest/deploy/componentTemplate/' . $template_id . $this->destination;
        return $this->execRest($action);
    }
    
    /**
     * Set a uDeploy API request-command to obtain, ALL available Component Templates.
     * NOTE: The returned list with templates will ONLY hold a partial (general) Template data,
     *       for each ComponentTemplate.
     *
     * @return string - The uDeploy API request to be executed
     */
    public function getComponentTemplates() {
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/componentTemplates.json'";
        }
        $action = $this->weburl . '/rest/deploy/componentTemplate' . $this->destination;
        return $this->execRest($action);
    }
    
    /**
     * @param string    $environment
     * @return string
     *
     * Will return false if argument is incorrect
     */
    public function getComponentVersionPropDefs($component = null){
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.propDefs.json'";
        }
        $action = "getComponentVersionPropDefs -component '{$component}'{$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Returns a paginated list of component versions
     *
     * @param string $component
     * @param number $rowsPerPage
     * @param number $pageNumber
     * @param string $orderField
     * @param string $sortType
     * @return boolean|string
     */
    public function getComponentVersions($component = null, $rowsPerPage = 10, $pageNumber = 1, $orderField = 'dateCreated', $sortType = 'desc'){
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.versions.{$pageNumber}.json'";
        }
    
        /* DEPRECATED ***
         $action = "'{$this->weburl}/rest/deploy/component/{$component}/versions/paginated/false?"
        . "rowsPerPage={$rowsPerPage}"
        . "&pageNumber={$pageNumber}"
        . "&orderField={$orderField}"
        . "&sortType={$sortType}' "
        . "{$this->destination}";
        ***/
    
        $action = "'{$this->weburl}/rest/deploy/component/{$component}/versions/paginated' {$this->destination}";
    
        return $this->execRest($action);
    }
    
    /**
     *
     * Return componentVersions in a snapshot
     *
     * @param string $snapshotId
     * @return boolean|string
     */
    public function getComponentVersionsInSnapshot($snapshotId = null) {
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/{$snapshotId}.componentVersions.json'";
        }
        $action = $this->weburl . "/rest/deploy/snapshot/{$snapshotId}/versions {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Get all tags in a component
     *
     * @param string $component
     * @param string $componentName
     * @return boolean
     */
    public function getTagsOnComponent($component = null, $componentName = null) {
        if (empty($componentName)) {
            $componentName = $component;
        }
    
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$componentName}.tags.json'";
        }
        $action = "getTagsOnComponent -component '{$component}'{$this->destination}";
        return $this->exec($action);
    }
    
    /* Create a STRING hoolding a POST curl request to be used for
    * IMPORTING or UPGRADE of a Component.
    *
    * This function will CHECK if the component to be Imported,
    * is based on a Component Template and
    * WILL ENSURE the corresponding Template is PRESENT on the Server,
    * before to prepare the IMPORT Component request string.
    *
    * @param string $componentFile [REQUIRED] Path to a Component JSON file
    *                              If not provided => $this->args[0] will be used.
    *                              If BOTH are not set => function will return FALSE
    * @param boolean $upgrade      [REQUIRED] Boolean TRUE/FALSE, used to set whether
    *                              to set an UPGRADE or IMPORT POST request correspondingly.
    *                              If not provided => $this->args[1] will be used.
    *                              If BOTH are not set => function will return false
    *
    * @return boolean|string   String holding a POST request for Component Import/Upgrade to be executed
    *                          or Boolean FALSE in case of either of the required parameters is NOT provided.
    */
    public function importComponent ($componentFile = null, $upgrade = null) {
        /*
         * Check if Component is based on a Template and
        * make sure the Template is present on server before to import the component
        */
        // STEP 1: Get contents of the Component to be imported and use it as Object
        $component = file_get_contents($componentFile);
        if ( $component === false ) {
            $this->log->error(__CLASS__ . '::' . __FUNCTION__
                    . ": There is a problem with reading Component backup file: \n" . $componentFile
                    . " Component IMPORT CANNOT be COMPLETED");
            return false;
        }
        // Convert $component to an Object
        $component = json_decode($component);
         
        // STEP 2: Check if the Component is based on a Template
        if ( isset($component->componentTemplate) ) {
            Utils::CLIinfo('Component: "' . $component->name . '" is based on a Template: "'
                    . $component->componentTemplate->name . '"');
                    Utils::CLIinfo('Check if ComponentTemplate exists on Server, before to continue with Component IMPORT', false);
                    $templateFound = false;
                    self::setReturn('json');
                    /*
                     * Check if ComponentTemplate is present on This Server
                    *  1. Get All ComponentTemplates on the Server
                    *  2. Check if the Template is present
                    * NOTE: we cannot use: getComponentTemplateDetails() because it requires:
                    *       Template Name and ID, the Component does not hold its Template Id,
                    *       and we cannot rely on the ID, as it definitely will differ on different Servers.
                    *       Or between the IDs in the backups files and Server IDs.
                    */
                    $serverCompTemplates = self::getComponentTemplates();
                    $serverCompTemplates = Utils::outputToArray($serverCompTemplates);
                    // 2.1 Try to find the ComponentTemplate
                    if (count($serverCompTemplates) > 0) {
                        foreach ($serverCompTemplates as $compTemplate) {
                            if ($compTemplate['name'] == $component->componentTemplate->name) {
                                $templateFound = true;
                                // ComponentTemplate found -=> exit for loop and continue with import
                                break;
                            }
                        }
                    }
    
                    // 2.2 ComponentTemplate was NOT found on this Server
                    if ( !$templateFound ) {
                        $templateImported = false;
                        // Prompt message and try to import it to this Server
                        Utils::CLIinfo('Component Template was NOT found on this Server.');
                         
                        // Extract the PATH to backups root directory from the $componentFile
                        $componentFilePathParts = explode('/', $componentFile);
                        // Remove Component file and Components Sub-Directory  (last 2 elements in the ComponentFile path)
                        $backupRootPathParts = array_splice($componentFilePathParts, 0, (count($componentFilePathParts) -2));
                        // Set backups path
                        $backupsRootDir = implode('/', $backupRootPathParts);
                        // Set Path for the ComponentTemplate backup file
                        $componentTemplateBackup = $backupsRootDir . '/componentTemplates/'
                                . $component->componentTemplate->name . '.json';
                         
                        if ( is_file($componentTemplateBackup) ) {
                            Utils::CLIinfo('Import Component Template: "' . $component->componentTemplate->name . '"'
                                    .' from its backup File, before Component Import.');
                                    /*
                                     * Use: RestoreUCD->importAComponentTemplate()
                                    * NOTE: PHP alows us to use a Class's children functions as its own function.
                                    *       i.e. RestoreUCD->importAComponentTemplate() can be called as:
                                    *            $this->importAComponentTemplate()
                                    */
                                    $templateImported = $this->importAComponentTemplate($componentTemplateBackup);
                        } else {
                            Utils::CLIinfo('Import Component Template: "' . $component->componentTemplate->name . '"'
                                    .' from its copy in the Component: "' . $component->name . '" backup File.');
                             
                            // Set a temporary Component Template from the component backup data and import it
                            $templateTmpFile = $this->output . '/componentTemplate'
                                    . $component->componentTemplate->name . 'tmpfile.json';
                            if ( Utils::writeToFile($templateTmpFile, $component->componentTemplate, false)) {
                                // Import Component Template from temporary file
                                $templateImported = $this->importAComponentTemplate($templateTmpFile);
                                // remove temporary file
                                unlink($templateTmpFile);
                            } else {
                                $this->log->error(__CLASS__ . '::' . __FUNCTION__ . ': problem with writing to file: '
                                        ."\n" . $templateTmpFile);
                            }
                        }
                         
                        // Check if Template was Imported and continue or return false
                        if ( $templateImported === false ) {
                            Utils::CLIerror('There was a problem with Importing of Component Template: "'
                                    . $component->componentTemplate->name . '"');
                                    Utils::CLIinfo('For more details. please check: "' . $this->log->getLogFile() . '"', false, 1);
                                    // return
                                    return false;
                        } else {
                            Utils::CLIinfo('Component Template: "'
                                    . $component->componentTemplate->name . '" - IMPORTED to Server.');
                                    Utils::CLIinfo('OK to continue with Component Import.', false);
                        }
                    } else {
                        Utils::CLIinfo('Component Template EXISTS on this Server.');
                        Utils::CLIinfo('Continue with Component Import.', false);
                    }
        } // else: Component is Generic, i.e. NOT BASED on a Template -=> OK to Import it
         
        // Prepare POST Request-action to be executed
        $action = "-X POST ";
        // NOTE the whole URL part MUST be in quotes. Otherwise it will brake on the & sign
        $action .= "'" . $this->weburl . '/rest/deploy/component/';
        // Import or Upgrade ComponentTemplate
        $action .= $upgrade ? 'upgrade?' : 'import?';
        // Use existent ComponentTemplate or Create it if it is Missing
        $action .= 'upgradeType=USE_EXISTING_IF_EXISTS';
        // Use existent Process or Create it if it is missing
        $action .= '&processUpgradeType=USE_EXISTING_IF_EXISTS\'';
        // Set parameters required for the POSt request
        $action .= ' -H "Content-Type: multipart/form-data"';
        $action .= " -F file=@'" . $componentFile . "'";
         
        // Return
        return $this->execRest($action);
    }
    
    /**
     * Create a STRING hoolding a POST curl request to be used for
     * IMPORTING or UPGRADE of a ComponentTemplate.
     *
     * @param string $componentTemplate [REQUIRED] Path to a ComponentTemplate JSON file
     *                                  If not provided => $this->args[0] will be used.
     *                                  If BOTH are not set => function will return FALSE
     * @param boolean $upgrade          [REQUIRED] Boolean TRUE/FALSE, used to set whether
     *                                  to set an UPGRADE or IMPORT POST request correspondingly.
     *                                  If not provided => $this->args[1] will be used.
     *                                  If BOTH are not set => function will return false
     *
     * @return boolean|string   String holding a POST request for ComponentTemplate Import/Upgrade to be executed
     *                          or Boolean FALSE in case of either of the required parameters is NOT provided.
     */
    public function importComponentTemplate ($componentTemplate = null, $upgrade = null) {
        // Prepare POST Request-action to be executed
        $action = "-X POST ";
        $action .= $this->weburl . '/rest/deploy/componentTemplate/';
        // Import or Upgrade ComponentTemplate
        $action .= $upgrade ? 'upgrade?' : 'import?';
        // Use existent Process or Create it if it is missing
        $action .= 'processUpgradeType=USE_EXISTING_IF_EXISTS';
        // Set parameters required for the POSt request
        $action .= ' -H "Content-Type: multipart/form-data"';
        $action .= " -F file=@'" . $componentTemplate . "'";
         
        // Return
        return $this->execRest($action);
    }
    
    /**
     *
     * Get Version by id
     *
     * @return boolean|string
     */
    public function getVersionFromId($versionId) {
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/version.{$versionId}.json'";
        }
        $action = $this->weburl . "/rest/deploy/version/{$versionId} {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Upgrade a component using a json file.
     * First argument is the path of the component file. uDeploy will then upgrade the component with the same name
     * If upgradeTemplate is set (by default, USE_EXISTING_IF_EXISTS) also component template are upgraded
     * If upgradeProcess is set (by default, USE_EXISTING_IF_EXISTS) also component processes are upgraded
     *
     * allowed value for upgradeTemplate and upgradeProcess
     *  USE_EXISTING_IF_EXISTS
     *  CREATE_NEW_IF_EXISTS
     *  FAIL_IF_EXISTS
     *  FAIL_IF_DOESNT_EXIST
     *  UPGRADE_IF_EXISTS
     *
     * @param string $component
     * @param string $upgradeProcess
     * @param string $upgradeProcess
     */
    public function upgradeComponent($component = null, $upgradeTemplate = 'USE_EXISTING_IF_EXISTS', $upgradeProcess = 'USE_EXISTING_IF_EXISTS') {
        $command = "-F file=@'{$component}' ";
        $command .= "'{$this->weburl}/rest/deploy/component/upgrade?";
        if (in_array( $upgradeTemplate, $this->upgrade_values )) {
            $command .= "upgradeType=$upgradeTemplate";
        } else {
            $command .= "upgradeType=USE_EXISTING_IF_EXISTS";
        }
    
        if (in_array( $upgradeProcess, $this->upgrade_values )) {
            $command .= "&processUpgradeType=$upgradeProcess'";
        } else {
            $command .= "&processUpgradeType=USE_EXISTING_IF_EXISTS'";
        }
        return $this->execRest($command);
    }
    
    /**
     * Set up a uDeploy API Request for UPDATING of a specified ComponentTemplate
     * from a JSON formatted file.
     *
     * @param string $componentTemplate Path to the JSON File, to be used for ComponentTemplate update
     * @param string $upgradeProcess    Boolean FLAG to set wheather to update ComponentTemplate Process(es) or Not
     *                                  DEFAULT value: TRUE
     *
     * @return boolean|string   String, holding the uDeploy API UPDATE action to be executed OR
     *                          Boolean FALSE, when $componentTemplate File is not provided
     */
    public function upgradeComponentTemplate($componentTemplate = null, $upgradeProcess = true) {
        $command = "-F file=@'{$componentTemplate}' ";
        $command .= "{$this->weburl}/rest/deploy/componentTemplate/upgrade?";
        if ($upgradeProcess) {
            $command .= "&processUpgradeType=UPGRADE_IF_EXISTS ";
        } else {
            $command .= "&processUpgradeType=USE_EXISTING_IF_EXISTS";
        }
    
        return $this->execRest($command);
    }

}
