<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2014
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ***************************************************************/

require_once ('include/EntryPoint.php');

/**
 * Backup uDeploy generating webUI-like export files (json), using udclient's APIs
 *
 * @class BackupUCD
 * @author Marco Bassi marcobas@ie.ibm.com
 * @author Kostadin Ivanov kostadii@ie.ibm.com
 */
class BackupUCD extends EntryPoint {
    public function __construct() {
        parent::__construct();
        $this->setReturn( 'json' );
    }
    
    /**
     * Run a complete backup of uDeploy
     */
    public function backup() {
        $backUp_watch = time();
        StopWatch::getInstance()->start( $backUp_watch );
        
        $return = array ();
        // Prompt message for Backup Process
        Utils::CLIinfo( '### START of UrbanCode BACKUP process ###' );
        Utils::CLIstart( 'BACKUP process: ' . date( "d-m-Y h:i:s" ) );
        Utils::CLIinfo( 'Server to backup:', false, 4 );
        // create a api to save weburl and output based on config file
        Utils::CLIdest( $this->weburl, false, 6 );
        Utils::CLIinfo( 'Backup folder:', false, 4 );
        Utils::CLIdest( $this->output, false, 6 );
        
        
        /*
         * Backup the backups - root directory. 
         * This is getting cha  nged during each of below steps
         */
        $backupsRootDir = $this->output;
        
        /* ### Step 1: Backup Resources ### */
        $backupResources_watch = time();
        StopWatch::getInstance()->start( $backupResources_watch );
        // Set Resources backup sub-folder
        $this->setOutput( $this->output . "/resources" );
        $return ['resources'] = $this->exportResources();
        StopWatch::getInstance()->stop( $backupResources_watch );
        Utils::CLIcompleted( 'Backup Resources: '. StopWatch::getInstance()->get($backupResources_watch));
        // Re-Set output to the Backup root directory for the next step
        $this->setOutput( $backupsRootDir );
        
        /* ### Step 2: Backup Resource Templates ### */
        // Set Resource Templates backup sub-folder
        $this->setOutput( $this->output . "/resourceTemplates" );
        $backupResourceTemplates_watch = time();
        StopWatch::getInstance()->start( $backupResourceTemplates_watch );
        $return ['resourceTemplates'] = $this->exportResourceTemplates();
        StopWatch::getInstance()->stop( $backupResourceTemplates_watch );
        Utils::CLIcompleted( 'Backup Resource Templates: '. StopWatch::getInstance()->get($backupResourceTemplates_watch));
        // Re-Set output to the Backup root directory for the next step
        $this->setOutput( $backupsRootDir );
        
        /* ### Step 3: Backup ComponentTemplates ### */
        $this->setOutput( $this->output . "/componentTemplates" );
        $backupComponentTemplates_watch = time();
        StopWatch::getInstance()->start( $backupComponentTemplates_watch );
        $return ['componentTemplates'] = $this->exportComponentTemplates();
        StopWatch::getInstance()->stop( $backupComponentTemplates_watch );
        Utils::CLIcompleted( 'Backup Component Templates: '. StopWatch::getInstance()->get($backupComponentTemplates_watch));
        $this->setOutput( $backupsRootDir );
        
        /* ### Step 4: Backup ALL Components ### */
        $this->setOutput( $this->output . "/components" );
        $backupComponents_watch = time();
        StopWatch::getInstance()->start( $backupComponents_watch );
        $return ['components'] = $this->exportComponents( null, true );
        StopWatch::getInstance()->stop( $backupComponents_watch );
        Utils::CLIcompleted( 'Backup Components: '. StopWatch::getInstance()->get($backupComponents_watch));
        $this->setOutput( $backupsRootDir );
        
        /* ### Step 5: Backup ALL Applications ### */
        $this->setOutput( $this->output . "/applications" );
        $backupApplications_watch = time();
        StopWatch::getInstance()->start( $backupApplications_watch );
        $return ['applications'] = $this->exportApplications( true, false );
        StopWatch::getInstance()->stop( $backupApplications_watch );
        Utils::CLIcompleted( 'Backup Applications: '. StopWatch::getInstance()->get($backupApplications_watch));
        $this->setOutput( $backupsRootDir );
        
        /* ### Step 6: Backup ALL Post Processing Scripts ### */
        $this->setOutput( $this->output . "/settings" );
        $this->setReturn( 'file' );
        $backupPostProcessingScripts_watch = time();
        StopWatch::getInstance()->start( $backupPostProcessingScripts_watch );
        $return ['postProcessingScripts'] = $this->rest()->process()->getPostProcessingScripts();
        StopWatch::getInstance()->stop( $backupPostProcessingScripts_watch );
        Utils::CLIcompleted( 'Backup Post Processing Scripts: '. StopWatch::getInstance()->get($backupPostProcessingScripts_watch));
        $this->setReturn( 'json' );
        $this->setOutput( $backupsRootDir );
        
        /* ### Step 7: Backup ALL Tags ### */
        $this->setOutput( $this->output . "/tags" );
        $this->setReturn( 'file' );
        $backupTags_watch = time();
        StopWatch::getInstance()->start( $backupTags_watch );
        $return ['tags'] ['component'] = $this->rest()->system()->getTag( 'component' );
        $return ['tags'] ['resource'] = $this->rest()->system()->getTag( 'resource' );
        StopWatch::getInstance()->stop( $backupTags_watch );
        Utils::CLIcompleted( 'Backup Tags: '. StopWatch::getInstance()->get($backupTags_watch));
        $this->setReturn( 'json' );
        $this->setOutput( $backupsRootDir );
        
        /* ### Step 8: Backup ALL Blueprints ### */
        $this->setOutput( $this->output . "/blueprints" );
        $backupBlueprints_watch = time();
        StopWatch::getInstance()->start( $backupBlueprints_watch );
        $return ['blueprints'] = $this->exportAllBlueprints();
        StopWatch::getInstance()->stop( $backupBlueprints_watch );
        Utils::CLIcompleted( 'Backup Blueprints: '. StopWatch::getInstance()->get($backupBlueprints_watch));
        $this->setReturn( 'json' );
        $this->setOutput( $backupsRootDir );
        
        /* ### Step 9: Backup ALL Generic Processes ### */
        $this->setOutput( $this->output . "/processes" );
        $backupGenericProcesses_watch = time();
        StopWatch::getInstance()->start( $backupGenericProcesses_watch );
        $return ['processes'] = $this->exportAllProcesses();
        StopWatch::getInstance()->stop( $backupGenericProcesses_watch );
        Utils::CLIcompleted( 'Backup Generic Processes: '. StopWatch::getInstance()->get($backupGenericProcesses_watch));
        $this->setReturn( 'json' );
        $this->setOutput( $backupsRootDir );
        StopWatch::getInstance()->stop( $backUp_watch );
        Utils::CLIcompleted( 'BACKUP process: '. StopWatch::getInstance()->get($backUp_watch));
        Utils::CLIinfo( '### END of UrbanCode BACKUP process ###' );
        
        return $return;
    }
    
    /**
     * Run a complete backup of a particular Application.
     * This includes:
     * - Export of the specified Application to a JSON file
     * - Export of Application BluePrints into dedicated 'blueprints' sub-folder
     * - Export of Application Components into dedicated 'components' sub-folder
     *
     * @param string    $application Application Name or ID of the Application to be exported
     *            
     * @return boolean|array  Boolean FALSE in case of a problem with Application export
     *         Array in the form:
     *         $result[application] = result from exportApplication step
     *         $result[blueprints] = result from exportBlueprints step
     *         $result[components] = result from exportComponents step
     */
    public function exportFullApplication($application) {
        // Validate if application Exists in UCD before to attempt to back it up
        $this->setReturn( 'json' );
        $exists = $this->udclient()->application()->getApplication( $application );
        if ($exists === false) {
            return false;
        }
        
        $return = array ();
        
        // Backup current output directory
        $backup_output = $this->output;
        
        // Backup Application
        $return ['application'] = $this->exportApplication( $application );
        
        // There was problem in application backup - error messages are handled under exportApplication()
        if ($return ['application'] === false) {
            Utils::CLIwarning('There was a problem with Application backup process.');
            Utils::CLIinfo('Skip Application "' . $application . '" components and blueprints backup.',
                    false, 3);
            $return ['blueprints'] = false;
            $return ['components'] = false;
            
        // Application was backed up -=> backup its blueprints and components
        } else {
            // Get the output to current application backup directory
            $appDir = $this->output;
            
            // Backup application blueprints
            $this->setOutput( "{$this->output}/blueprints" );
            $return ['blueprints'] = $this->exportBlueprints( $application );
            // Revert output to current application backup directory
            $this->setOutput( $appDir );
            
            // Backup application Components
            $this->setOutput( $this->output . "/components" );
            $return['components'] = $this->exportComponents($application,true);
            $this->setOutput( $appDir );
        }
        
        // Restore output directory
        $this->setOutput( $backup_output );
        
        return $return;
    }
    
    /**
     * @param string $componentID            
     * @param string    $componentName  optional
     *            Export a component
     */
    public function exportComponent($componentID, $componentName = null) {
        $this->setReturn( 'file' );
        $output = $this->rest()->component()->getComponentRest( $componentID, 
                $componentName );
        $this->setReturn( 'json' );
            Utils::CLIdest($this->output . '/' . $componentName . '.json', false, 6);
        return $output;
        
        if ($output === false) {
                Utils::CLIwarning('There was a problem with obtaining a Component to backup.');
            return false;
        }
    }
    
    /**
     * Export all components or components in a specified Application.
     *
     * If $application is NOT provided, ALL components will be backed up.
     *
     * If $application is SPECIFIED, ONLY components which belong to this Application, will be backed up.
     *
     * @param string    $application    [OPTIONAL] String holding the Application Name or ID,
     *            for the Application which components are to be backed up.
     *            DEFAULT value: NULL
     * @param bool      $useName        [OPTIONAL] If is set to true,
     *            will use component name instead of ID to export Component.
     *            DEFAULT value: FALSE
     *            
     */
    public function exportComponents($application = null, $useName = false) {
        $this->setReturn( 'json' );
        if (empty( $application )) {
            $components = $this->udclient()->component()->getComponents();
        } else {
            $components = $this->udclient()->component()->getComponentsInApplication( $application );
        }
        
        if ($components === false) {
            Utils::CLIwarning('There was a problem with obtaining of Components to be backed up.');
            return false;
        }
        
        $components = Utils::outputToArray( $components );
        
        if (count( $components ) == 0) {
            Utils::CLIwarning( 'There is NO Components to be backed up.' );
            return null;
        }
        
        $return = array ();
        $componentOutput = $this->output;
        $tagOutput = $this->output . "/tags";
        foreach ( $components as $comp ) {
            Utils::CLIinfo( "Exporting Component '{$comp['name']}':", true, 3 );
            if ($useName) {
                $compKey = $comp ['name'];
            } else {
                $compKey = $comp ['id'];
            }
            $return [$comp ['id']] ['name'] = $comp ['name'];
            $return[$comp['id']]['return'] = $this->exportComponent($comp['id'], $compKey);
            
            // Export tags
            $this->setOutput( $tagOutput );
            $return[$comp['id']]['tags'] = $this->exportComponentTags($comp['id'], $compKey);
            $this->setOutput( $componentOutput );
        }
        
        return $return;
    }
    public function exportComponentTags($component, $componentName) {
        $this->setReturn( 'file' );
        if (empty( $component )) {
            return false;
        }
        $result = $this->rest()->component()->getTagsOnComponent( $component, $componentName );
        $this->setReturn( 'json' );
        return $result;
    }
    
    /**
     * Export a UCD Application into a JSON formatted file.
     *
     * @param string    $application The Application Name or ID
     *            of the Application to be backed up.
     *            
     * @return boolean|string   Boolean FALSE, in case of an ERROR during backup process.
     *         String holding the path to the Application backup file,
     *         on SUCCESS.
     */
    public function exportApplication($application) {
        // Export Application into a file - return will hold the destination of the Application backup file
        $this->setReturn( 'file' );
        $return ['backup_file'] = $this->rest()->application()->getApplicationRest( $application );
        
        // Validate result
        if ($return ['backup_file'] !== false) {
            Utils::CLIinfo('Application "' . $application . '" saved into:', false, 3);
            Utils::CLIdest( $return ['backup_file'], false, 3 );
        } else {
            Utils::CLIwarning('Application "' . $application . '" backup - FAILED');
            Utils::CLIinfo('For more details, please check: "' . $this->log->getLogFile() .'"', false, 3);
            return false;
        }
        
        $appDir = $this->output;
        
        // Export List of Application Components into a file
        $this->setOutput( $this->output . "/components" );
        $this->setReturn( 'file' );
        $return ['components_list'] = $this->udclient()->component()->getComponentsInApplication( 
                $application );
        $this->setOutput( $appDir );
        
        // Validate result
        if ($return ['components_list'] !== false) {
            Utils::CLIinfo('Application components list for "' . $application . '" saved into:', false, 3);
            Utils::CLIdest( $return ['components_list'], false, 3 );
        } else {
            Utils::CLIwarning('Application components list for "' . $application . '" backup - FAILED');
        }
        
        // Backup application blueprints
        $this->setOutput( "{$this->output}/blueprints" );
        $return ['blueprints'] = $this->exportBlueprints( $application );
        // Revert output to current application backup directory
        $this->setOutput( $appDir );
        
        return $return;
    }
    
    /**
     * Export all Applications.
     * Will get a complete list of available Application on specified in $configs UCD Server.
     * Will export each UCD Application to its dedicated backup folder, by calling helper functions:
     * - exportApplication() - Application backup
     *     - exportFullApplication()    - Application backup + underlying: blueprints, components etc.
     * depending on whether $fullExport is set to FALSE or TRUE, respectively.
     *
     * @param boolean  $useName    [OPTIONAL] If set to TRUE - Application Name will be used
     *            for each Application export.
     *            If set to FALSE - Application ID will be used.
     *            DEFAULT value: FALSE
     *            
     * @param boolean  $fullExport [OPTIONAL] If set to TRUE - each Application export
     *            will be handled by exportFullApplication().
     *            Otherwise - each Application export
     *            will be handled by the exportApplication() function.
     *            DEFAULT value: FALSE
     *            
     * @return boolean|NULL|array  Boolean FALSE in case of an ERROR 
     *         in retrieving Applications to backup from Server.
     *         NULL if there was NOT Available Applications to backup.
     *         Array in the form:
     *         array[AplicationID][name] = ApplicationName
     *         array[ApplicationID[backup] = result from App export step
     *        
     */
    public function exportApplications($useName = false, $fullExport = false) {
        $this->setReturn( 'json' );
        $apps = $this->udclient()->application()->getApplications();
        
        if ($apps === false) {
            Utils::CLIwarning('There was a problem with obtaining of Applications to be backed up.');
            return false;
        }
        
        $apps = Utils::outputToArray( $apps );
        
        if (count( $apps ) == 0) {
            Utils::CLIwarning( 'There is NO Applications to be backed up.' );
            return null;
        }
        
        $return = array ();
        foreach ( $apps as $app ) {
            Utils::CLIinfo( 'Exporting Application "' . $app ['name'] . '":' );
            
            if ($useName) {
                $appKey = $app ['name'];
            } else {
                $appKey = $app ['id'];
            }
            
            // Backup current output location
            $outputBackup = $this->output;
            
            // Set dedicated Application backup sub-directory
            $this->setOutput( $this->output . '/' . $app ['name'] );
            
            $return [$app ['id']] ['name'] = $app ['name'];
            if ($fullExport) {
                $return[$app['id']]['backup'] = $this->exportFullApplication($appKey);
            } else {
                $return[$app['id']]['backup'] = $this->exportApplication($appKey);
            }
            
            // Check backup result and prompt on console
            if ($return [$app ['id']] ['backup'] !== false) {
                Utils::CLIinfo('Application: "' . $app['name'] . '" backup completed - SUCCESS');
            } else {
                Utils::CLIwarning('Application: "' . $app['name'] . '" backup FAILED to complete');
            }
            
            // Revert to previous / backed output
            $this->setOutput( $outputBackup );
        }
        
        return $return;
    }
    
    /**
     * Exports ALL available ComponentTemplates in UrbanCode as JSON formatted sctrings into a File.
     * In case if there is NO ComponentTemplates to be exported, this function will return an empty Array.
     *
     * @return array Array in the form:
     *         array(
     *         'componentTemplateID' => array(
     *         'name' => 'ComponentTemplate Name',
     *         'return' => 'result from function exportComponentTemplate()'
     *         )
     *         )
     */
    public function exportComponentTemplates() {
        $this->setReturn( 'json' );
        $templates = $this->rest()->component()->getComponentTemplates();
        
        if ($templates === false) {
            Utils::CLIwarning('There was a problem with obtaining of ComponentTemplates to be backed up.');
            return false;
        }
        
        $templates = Utils::outputToArray( $templates );
        
        if (count( $templates ) == 0) {
            Utils::CLIwarning('There is NO ComponentTemplates to be backed up.');
            return null;
        }
        
        $return = array ();
        foreach ( $templates as $template ) {
            // Get Details for this $template and export it
            $template_data = $this->rest()->component()->getComponentTemplateDetails( 
                    $template ['id'], $template ['name'] );
            if ($template_data !== false) {
                $template_data = Utils::outputToArray( $template_data );
                // Set return and Export ComponentTemplate
                $return [$template ['id']] ['name'] = $template ['name'];
                $return[$template['id']]['return'] = $this->exportComponentTemplate($template_data);
            } else {
                $return [$template ['id']] ['name'] = $template ['name'];
                $return [$template ['id']] ['return'] = false;
            }
        }
        
        // Return
        return $return;
    }
    
    /**
     * Store a ComponentTemplate as a JSON formatted string into a File.
     *
     * @param String|Array|Object $template_data JSON formatted String OR Array OR Object, 
     *            which holds the DATA of a ComponentTemplate to be exported.
     * @return boolean Boolean TRUE/FALSE, based on whether the provided ComponentTemplate,
     *         has been saved to file or not.
     */
    public function exportComponentTemplate($template_data) {
        // Use $template_data as Object
        $template_obj = Utils::getAsObject( $template_data );
        
        Utils::CLIinfo( 'Exporting Component-Template "' . $template_obj->name . '": ' );
        
        Utils::CLIinfo('Save Component-Template "' . $template_obj->name . '" into:', false, 3);
        $templateFileName = str_replace(array(' ', '/'), array('_', '-'), $template_obj->name);
        Utils::CLIdest($this->output . '/' . $templateFileName . '.json', false, 3);
        
        // Set $template_obj to a JSON formatted string, before to save it to file
        $template_json = json_encode( $template_obj );
        
        $return = Utils::writeToFile($this->output . '/' . $templateFileName . '.json', $template_json, false);
        
        return $return;
    }
    
    /**
     *
     * Export a resource and saves it into json file
     *
     * @param string $resource            
     * @return array|boolean
     */
    public function exportResource($resource) {
        $this->setReturn( 'file' );
        
        // Export
        $result = $this->udclient()->resource()->getResource( $resource );
        
        if ($result === false) {
            Utils::CLIwarning( 'Resource "' . $resource . '" backup - FAILED.' );
        } else {
            Utils::CLIinfo('Resource "' . $resource . '" saved into:', true, 3);
            Utils::CLIdest($this->output . '/' . $resource . '.json', false, 3);
        }
        
        $this->setReturn( 'json' );
        
        return $result;
    }
    
    /**
     *
     * Export all resources available in uDeploy
     *
     * @return array|boolean
     */
    public function exportResources() {
        $result = array ();
        $this->setReturn( 'json' );
        $resources = $this->udclient()->resource()->getResources();
        if ($resources === false) {
            Utils::CLIwarning('There was a problem with obtaining of Resources to be backed up.');
            return false;
        }
        
        $resources = Utils::outputToArray( $resources );
        
        if (count( $resources ) == 0) {
            Utils::CLIwarning( 'There is NO Resources to be backed up.' );
            return null;
        }
        
        foreach ( $resources as $resource ) {
            $id = $resource ['id'];
            $result [$id] = $this->exportResource( $id );
        }
        return $result;
    }
    
    /**
     *
     * Export all blueprints for all the applications
     *
     * @return boolean|array
     */
    public function exportAllBlueprints() {
        Utils::CLIinfo( "Exporting blueprints" );
        
        $this->setReturn( 'json' );
        $apps = $this->udclient()->application()->getApplications();
        
        if ($apps === false) {
            Utils::CLIwarning('There was a problem with obtaining list of Applications.');
            return false;
        }
        
        $apps = Utils::outputToArray( $apps );
        
        if (count( $apps ) == 0) {
            Utils::CLIwarning( 'No application available.' );
            return null;
        }
        
        $return = array ();
        $backupOutput = $this->output;
        foreach ( $apps as $app ) {
            Utils::CLIout( "  {$app['name']}" );
            
            $this->setOutput( "{$backupOutput}/{$app['name']}" );
            $result [$app ['name']] = $this->exportBlueprints( $app ['name'] );
        }
        
        $this->setOutput( $backupOutput );
        
        return $result;
    }
    
    /**
     *
     * Export all blueprints for an application
     *
     * @param string $application            
     */
    public function exportBlueprints($application) {
        // Get blueprint list
        $this->setReturn( 'json' );
        $blueprints = $this->rest()->resource()->getBlueprintsInApplication( 
                $application );
        if ($blueprints === false) {
            Utils::CLIwarning('There was a problem with obtaining of BluePrints for Application: "'
                    . $application . '" to be backed up.');
            return false;
        }
        
        $blueprints = Utils::outputToArray( $blueprints );
        
        if (count( $blueprints ) == 0) {
            Utils::CLIwarning('There is NO BluePrints in Application: "'. $application .'" to be backed up.');
            return null;
        }
        
        // Set Export to json files
        $this->setReturn( 'file' );
        foreach ( $blueprints as $blueprint ) {
            $blueprintId = $blueprint ['id'];
            $blueprintName = $blueprint ['name'];
            $result [$blueprintName] = $this->rest()->resource()->getBlueprint( $blueprintId, 
                    $blueprintName );
            
            if ($result [$blueprintName] === false) {
                Utils::CLIwarning('BluePrint "' . $blueprintName . '" backup - FAILED.');
            } else {
                Utils::CLIinfo('BluePrint "' . $blueprintName . '" saved into:', true, 3);
                Utils::CLIdest( $result [$blueprintName], false, 3 );
                
                $outputBackup = $this->output;
                $this->setOutput( "{$this->output}/resources" );
                $result[$blueprintName]['resources'] = self::exportBlueprintResources($blueprintId,$blueprintName);
                $this->setOutput( $outputBackup );
            }
        }
        // Restore default return to json
        $this->setReturn( 'json' );
        
        return $result;
    }
    public function exportBlueprintResources($blueprintId, $blueprintName = null) {
        $this->setReturn( 'file' );
        
        $result = $this->rest()->resource()->getBlueprintResources( $blueprintId, $blueprintName );
        
        if (empty( $blueprintName )) {
            $blueprintName = $blueprintId;
        }
        if ($result === false) {
            Utils::CLIwarning('BluePrint resources for "' . $blueprintName . '" backup - FAILED.');
        } else {
                Utils::CLIinfo('BluePrint resources for "' . $blueprintName . '" saved into:', true, 3);
            Utils::CLIdest( $result, false, 3 );
        }
        
        $this->setReturn( 'json' );
        
        return $result;
    }
    public function exportAllApplicationProcesses($application) {
        // Get a List of Application processes
        $this->setReturn( 'json' );
        $appProcesses = $this->rest()->application()->getApplicationProcesses( $application );
        if ($appProcesses === false) {
            Utils::CLIwarning('There was a problem with obtaining of Application: "'
                    . $application . '" Processes to be backed up.');
            return false;
        }
        
        $appProcesses = Utils::outputToArray( $appProcesses );
        
        if (count( $appProcesses ) == 0) {
            Utils::CLIwarning('There is NO Processes in Application: "'. $application .'" to be backed up.');
            return null;
        }
        
        $result = array ();
        foreach ( $appProcesses as $appProcess ) {
            $result[$appProcess['name']] = $this->exportApplicationProcess($application, $appProcess['name'], $appProcess['id']);
        }
        
        // Return
        return $result;
    }
    public function exportApplicationProcess($application, $processName, $processId) {
         Utils::CLIinfo('Exporting Application: "' . $application .'" '
                 . 'process: "' . $processName .'"', true, 3);
        
        // Export Application Process into a file
        $this->setReturn( 'file' );
        $result = $this->rest()->application()->getApplicationProcessRest( $processId, $processName );
        $this->setReturn( 'json' );
        
        if ($result === false) {
            Utils::CLIwarning('Application: "' . $application .'" '
                 . 'process: "' . $processName .'"" backup - FAILED.');
        } else {
            Utils::CLIinfo('Application: "' . $application .'" '
                 . 'process: "' . $processName .'"" saved into:', true, 3);
            Utils::CLIdest( $result, false, 3 );
        }
        
        return $result;
    }
    public function exportResourceTemplates() {
        $this->setReturn( 'json' );
        $templates = $this->rest()->resource()->getResourceTemplates();
        $templates = Utils::outputToArray( $templates, true );
        
        foreach ( $templates as $template ) {
            $this->setReturn( 'file' );
            $result[$template['id']][$template['name']] = $this->rest()->resource()->getResourceTemplate($template['id'],$template['name']);
            $file = file_get_contents("{$this->output}/{$template['name']}.json");
            $templateResources = json_decode( $file, true );
            
            $result[$template['id']]['children'] = self::exportResourceTemplateResources($templateResources);
            
        }
        $this->setReturn( 'json' );
        
        return $result;
    }
    public function exportResourceTemplateResources($resources) {
        foreach ( $resources as $resource ) {
            if (array_key_exists( 'role', $resource )) {
                if (array_key_exists( 'specialType', $resource ['role'] )) {
                    if ($resource ['role'] ['specialType'] == 'AGENT_PLACEHOLDER') {
                        Utils::CLIout("Exporting resources for resource template '{$resource['name']}'");
                        
                        $this->setReturn( 'file' );
                        $backupDir = $this->output;
                        $this->setOutput( $this->output . "/resources" );
                        
                        $result [$resource ['id']] = $this->rest()->resource()->getResourceTemplateResources( 
                                $resource ['id'] );
                        
                        $this->setOutput( $backupDir );
                        $this->setReturn( 'json' );
                    }
                }
            }
        }
        
        return $result;
    }
    public function exportAllProcesses() {
        Utils::CLIinfo( "Exporting Processes" );
        // Get all processes in the server
        $processes = Utils::outputToArray( $this->rest()->process()->getProcesses() );
        
        // Export processes
        $this->setReturn( 'file' );
        foreach ( $processes as $process ) {
            $result [$process ['id']] = $this->rest()->process()->exportProcess( $process ['id'] );
            
            if ($result [$process ['id']] !== false) {
                Utils::CLIout( "  {$process['name']}" );
                Utils::CLIout( "  {$result[$process['id']]}" );
            } else {
                Utils::CLIerror( "{$process['name']}" );
            }
        }
        
        $this->setReturn( 'json' );
        return $result;
    }
}